/* $Id: return.c,v 1.1 1996/04/11 03:30:00 bert Exp $
 *
 * calculate return probabilities for packets in a Nelson potential
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <time.h>
#include "complex.h"
#include "fft.h"
#include "oppack.h"
#include "propagate.h"
#include "expect.h"
#include "util.h"

/* directory for temporary files */
#define SCRATCH "/var/tmp/"

/* parameters that can be set by command-line options */
real delta,dt, eps, vee_coef,pack_coef,mu_coef;
unsigned long niter, riter, size;
unsigned char debug, nocalc;

int angles;
real energy, xstep,ystep, lowX,lowY, highX,highY;

/* the generating function for the potential*/
Vprop_gen_f generate_Vprop = generate_Vprop_nelson;

/* global variables for getopt() */
extern char *optarg;
extern int optind;

/***** code *****/

/*
 * describe program usage.
 */
void usage(char* name)
{
  fprintf(stderr, 
"usage: %s [arguments]\n\
options:\n\
   -i #iter          number of initial iterations\n\
   -# #iter          number of iterations with return probabilities\n\
   -n mesh_size      size of the discrete grid\n\
   -s delta_r        size of the discrete space step\n\
   -t delta_t        size of the discrete time slice\n\
   -e width          initial width of the packet\n\
   -v propV_coef     coefficient for the potential\n\
   -m mu             Nelson x^2/2 coefficient\n\
   -p packet_coef    coefficient for the packet\n\
   -h hbar           value of h-bar\n\
choice of parameters for return-probability calculations:
   -E energy         classical energy of the packet\n\
   -a #angles        number of choices for angle\n\
   -q xstep          step for packet X coordinate\n\
   -w ystep          step for packet Y coordinate\n\
   -x xmin -X xmax   limits for packet X coordinate\n\
   -y ymin -Y ymax   limits for packet Y coordinate\n\
choice of potential:\n\
  (default)          Nelson potential      ( V = mu x^2/2 + (y - x^2/2)^2 )\n\
   -o                harmonic oscillator potential        ( V = k/2 |r|^2 )\n\
   -c                constant potential                           ( V = k )\n\
debugging options:\n\
   -C                only count valid points (don't calculate)\n\
   -d                turn on debugging\n",
	  name);
  exit (2);
}

/*
 * read and parse the command-line arguments.
 */
void read_arguments(int argc, char **argv)
{
  int c;
  int arg_err=0;

  /* set parameters: first the defaults... */

  debug=0;
  nocalc=0;

  niter = 100;  riter = 101;
  size = 64;
  delta = .1;   dt = .01; 
  eps=2.;
  vee_coef=1.;  mu_coef=0.1;  pack_coef=1.;

  energy = .5;
  angles = 18;
  xstep = ystep = 0.;

  lowX = 0.0;   highX = 1.5;   lowY = -1.0;   highY = 1.0;

  /* ...then check arguments */

  while ((c = getopt(argc, argv,
		     "i:#:n:s:t:e:v:m:p:h:E:a:q:w:x:y:X:Y:ocCd?")) != EOF)
    switch (c) {
    case 'i':  niter = atol(optarg);          break;
    case '#':  riter = atol(optarg);          break;
    case 'n':  size = atol(optarg);           break;
    case 's':  delta = atof(optarg);          break;
    case 't':  dt = atof(optarg);             break;
    case 'e':  eps = atof(optarg);            break;
    case 'v':  vee_coef = atof(optarg);       break;
    case 'm':  mu_coef = atof(optarg);        break;
    case 'p':  pack_coef = atof(optarg);      break;
    case 'h':  hbar = atof(optarg);           break;
    case 'E':  energy = atof(optarg);         break;
    case 'a':  angles = atol(optarg);         break;
    case 'q':  xstep = atof(optarg);          break;
    case 'w':  ystep = atof(optarg);          break;
    case 'x':  lowX = atof(optarg);           break;
    case 'y':  lowY = atof(optarg);           break;
    case 'X':  highX = atof(optarg);          break;
    case 'Y':  highY = atof(optarg);          break;
    case 'o':  generate_Vprop = generate_Vprop_harmonic;  break;
    case 'c':  generate_Vprop = generate_Vprop_constant;  break;
    case 'C':  nocalc++;                      break;
    case 'd':  debug++;                       break;
    case '?':  arg_err++;                     break;
    }

  if ((optind<argc) || arg_err) {
    usage(progname);
  }
  /* status and assignments that depend on the ones above */

  if (xstep <= 0.)  xstep=delta;
  if (ystep <= 0.)  ystep=delta;

  if (debug) {
    printf("niter = %10ld\n", niter);
    printf("riter = %10ld\n", riter);
    printf("size  = %10ld\n", size);
    printf("delta = %10.8" LFMT "g\n", delta);
    printf("dt    = %10.8" LFMT "g\n", dt);
    printf("eps   = %10.8" LFMT "g\n", eps);
    printf("mu    = %10.8" LFMT "g\n", mu_coef);
    printf("Vcoef = %10.8" LFMT "g\n", vee_coef);
    printf("pcoef = %10.8" LFMT "g\n\n", pack_coef);
    printf("E     = %10.8" LFMT "g\n", energy);
    printf("angles= %10d\n", angles);
    printf("packX = %10.8" LFMT "g : %" LFMT "g : %" LFMT "g\n",
	   lowX, xstep, highX);
    printf("packY = %10.8" LFMT "g : %" LFMT "g : %" LFMT "g\n",
	   lowY, ystep, highY);
    printf("hbar  = %10.8" LFMT "g\n", hbar);
    printf("mass  = %10.8" LFMT "g\n\n", mass);
    printf("T     = %10.8" LFMT "g ... %" LFMT "g\n\n",
	   niter*dt, (niter+riter-1)*dt);
  }

  eps *= delta;

  if (debug) {
    printf("s_eps = %10.8" LFMT "g\n\n", eps);
  }
}

/***** main program *****/

int main(int argc, char **argv)
{
  long xmin,ymin, xmax,ymax, nxsteps,nysteps, xx,yy;
  long ith, iter, numpts, cntpts;

  /* "real" and "complex" types are defined in complex.h */
  complex *packet,*ipacket, *propV,*propV2, *propK;
  real *result, *lres;
  real rx, rx22, ry, kx, ky, V, p, theta;
  real norm1;

  /* output file for results */
  FILE *ff;
  char tmpname[128];

  /* this is for ETA calculations */
  time_t t0;
  unsigned long eta;

  progname = argv[0];

  /* read the command line arguments */

  read_arguments(argc, argv);

  /* allocate everything */

  packet = (complex*)safe_malloc(size*size*sizeof(complex),"packet");
  propK =  (complex*)safe_malloc(size*size*sizeof(complex),"propK");
  propV =  (complex*)safe_malloc(size*size*sizeof(complex),"propV");
  propV2 = (complex*)safe_malloc(size*size*sizeof(complex),"propV2");
  ipacket =(complex*)safe_malloc(size*size*sizeof(complex),"ipacket");

  /* now, allocate memory to store the results in */

  xmin = lowX / xstep;   xmax = highX / xstep;
  nxsteps = xmax - xmin + 1;
  ymin = lowY / ystep;   ymax = highY / ystep;
  nysteps = ymax - ymin + 1;
  result = (real*)safe_malloc(riter*angles*nxsteps*nysteps*sizeof(real),
			      "result");

  /* allocate and initialize data */

  generate_Kprop (propK, size, dt,delta);
  fprintf(stdout,"propK initialized\n");

  generate_Vprop (propV, size, dt,delta,vee_coef,mu_coef);
  fprintf(stdout,"propV initialized\n");

  generate_Vprop (propV2, size, dt/2,delta,vee_coef,mu_coef);
  fprintf(stdout,"propV2 initialized\n");

  /* loop over all coordinates, count valid */

  numpts=0;
  for (xx = xmin;  xx <= xmax;  xx++) {
    rx = xstep*xx;
    rx22 = (rx*rx)/2.;
    for (yy = ymin;  yy <= ymax;  yy++) {
      ry = ystep*yy;

      V = ry - rx22;
      V = mu_coef * rx22 + V*V;

      if (V <= energy)
	numpts++;
    }
  }
  printf("%ld valid points ", numpts);
  numpts *= angles;
  printf("(%ld valid point-angle pairs)\n", numpts);

  /* if we're only counting points, we're done */
  if (nocalc)  {
    printf("%ld total points (%ld total point-angle pairs)\n",
	   nxsteps*nysteps, nxsteps*nysteps*angles);
    exit(0);
  }

  /* loop over all coordinates, calculate results */

  t0 = time(NULL);

  lres = result;
  cntpts = 0;
  for (xx = xmin;  xx <= xmax;  xx++) {
    rx = xstep*xx;
    rx22 = (rx*rx)/2.;
    for (yy = ymin;  yy <= ymax;  yy++) {
      ry = ystep*yy;

      V = ry - rx22;
      V = mu_coef * rx22 + V*V;

      if (V > energy) {
	/** clasically forbidden region; return probabilities are 0 **/
	for (ith = 0;  ith < angles;  ith++) {
	  for (iter = 0;  iter < riter;  iter++) {
	    lres[iter] = 0.;
	  }
	  lres += riter;                 /* next "slice" of the result array */
	}

      } else {
	/** clasically allowed region **/
	p = sqrt( (energy - V) * 2 * mass );

	for (ith = 0;  ith < angles;  ith++) {
	  theta = M_PI * ith / angles;
	  /* to convert to 1/2-circle, remove the "2.*" */

	  kx = p/hbar * cos(theta);
	  ky = p/hbar * sin(theta);

	  generate_packet(packet, size, rx,ry,kx,ky,delta,eps,pack_coef);
	  copy_data (packet, ipacket, size);
	  fprintf(stdout,"r = (%" LFMT "g,%" LFMT "g);  p = (%" LFMT
		  "g;%" LFMT "g)\n", rx,ry, p,theta*180./M_PI);
	  norm1 = sum_sq(packet,size);

	  /* do the actual propagation */
	  propagate (packet, propV2, propV, propK, size, niter, 0);
	  /* calculate return probabilities */
	  lres[0] = cabs(overlap (ipacket, packet, size) / norm1);
	  for (iter = 1;  iter < riter;  iter++) {
	    propagate (packet, propV2, propV, propK, size, 1, iter+niter-1);
	    lres[iter] = cabs(overlap (ipacket, packet, size) / norm1);
	  }
#if defined(DO_TRACK) || defined(DO_TRACK_SYNC)
	  fprintf(stdout,".. done.\n");
#endif

	  lres += riter;                 /* next "slice" of the result array */

	  cntpts++;
	  eta = ((real)(time(NULL)-t0) * (numpts-cntpts) / cntpts);
	  printf("%" LFMT "f%% done [%ld of %ld], est %ld:%02ld:%02ld left\n",
		 100.*cntpts/numpts, cntpts, numpts,
		 eta/3600, (eta/60) % 60, eta % 60);
	}

      }
    }
  }

  /* dump the results */
  for (iter = 0;  iter < riter;  iter++) {
    sprintf(tmpname, SCRATCH "return.%" LFMT "g", (niter+iter)*dt);

    ff = fopen (tmpname, "w");
    if (ff==(FILE*)NULL) {
      fprintf(stderr,"%s: can't open %s for writing [%s]\n",
	      progname, tmpname, strerror(errno));
      exit(99);
    }

    for (xx = xmin;  xx <= xmax;  xx++) {
      for (yy = ymin;  yy <= ymax;  yy++) {
	int idx = ((xx-xmin)*nysteps+(yy-ymin)) * angles * riter + iter;

	fprintf(ff, "%20.12" LFMT "g %20.12" LFMT "g", xx*xstep, yy*ystep);
	for (ith = 0;  ith < angles;  ith++) {
	  fprintf(ff, " %20.12" LFMT "g", result[ idx + ith*riter ] );
	}
	fprintf(ff, "\n");

      }
    }

    fclose(ff);
  }

  free(result);
  free(packet);
  free(propK);
  free(propV);
  free(propV2);
  return 0;
}
