/* Driver for routine fourn */

#include <stdio.h>
#include <stdlib.h>
#include "complex.h"
#include <sys/times.h>
#include <unistd.h>

#define DEF_TESTS 10

void lap(clock_t start, struct tms *before,
	 clock_t stop,  struct tms *after)
{
#ifdef VERBOSITY_FROM_HELL
  printf("  Total:   %ld real, %ld user, %ld system\n",
	 (long) stop,
	 (long) after->tms_utime,  (long) after->tms_stime);
#endif
  printf("  Elapsed: %ld real, %ld user, %ld system\n",
	 (long) (stop-start),
	 (long) (after->tms_utime - before->tms_utime),
	 (long) (after->tms_stime - before->tms_stime));
}

int main(int argc, char** argv)
{
  complex *data;
  clock_t start, stop;
  struct tms before, after;
  int count, dummy, i, j;
  int ntests = DEF_TESTS;
  int nloops = 1;
  real limit0, limit1;

  if (argc > 1)  ntests = atoi(argv[1]);
  if (argc > 2)  nloops = atoi(argv[2]);

  printf("Generating data for %d x %d tests.\n", nloops, ntests);

  srand48(getpid() + times(&before));

  printf("Allocating %d bytes.\n", ntests * sizeof(complex));

  data = malloc( ntests * sizeof(complex) );
  if (!data) abort();

  for (j=0; j<nloops; j++) {
    for (i=0; i<ntests; i++) {
      data[i] = drand48() + drand48()*1i;
    }
  }

  printf("Running empty loop with false exit condition.\n");
  dummy = 1;   count = 0;
  start = times(&before);
  for (j=0; j<nloops; j++) {
    for (i=0; i<ntests; i++) {
      if (dummy==0)
	count++;
    }
  }
  stop = times(&after);
  lap(start, &before, stop, &after);
  printf("%d matches.\n", count);

  printf("Running empty loop with true exit condition.\n");
  dummy = 0;   count = 0;
  start = times(&before);
  for (j=0; j<nloops; j++) {
    for (i=0; i<ntests; i++) {
      if (dummy==0)
	count++;
    }
  }
  stop = times(&after);
  lap(start, &before, stop, &after);
  printf("%d matches.\n", count);

  printf("Running a == for complex numbers.\n");
  count = 0;
  start = times(&before);
  for (j=0; j<nloops; j++) {
    for (i=0; i<ntests; i++) {
      if (data[i] == 0)
	count++;
    }
  }
  stop = times(&after);
  lap(start, &before, stop, &after);
  printf("%d matches.\n", count);

  printf("Running an abs() comparison for complex numbers.\n");
  count = 0;   limit0 = 1.0;
  start = times(&before);
  for (j=0; j<nloops; j++) {
    for (i=0; i<ntests; i++) {
      if (cabs(data[i]) > limit0)
	count++;
    }
  }
  stop = times(&after);
  lap(start, &before, stop, &after);
  printf("%d matches.\n", count);

  printf("Running an abs()^2 comparison for complex numbers.\n");
  count = 0;
  start = times(&before);
  for (j=0; j<nloops; j++) {
    for (i=0; i<ntests; i++) {
      if (csqabs(data[i]) > limit0)
	count++;
    }
  }
  stop = times(&after);
  lap(start, &before, stop, &after);
  printf("%d matches.\n", count);

#define sq(x) ((x)*(x))

  printf("Running a re^2+im^2 comparison for complex numbers.\n");
  count = 0;
  start = times(&before);
  for (j=0; j<nloops; j++) {
    for (i=0; i<ntests; i++) {
      if ((sq(Re(data[i])) + sq(Im(data[i]))) > limit0)
	count++;
    }
  }
  stop = times(&after);
  lap(start, &before, stop, &after);
  printf("%d matches.\n", count);

  printf("Running a `box' comparison for complex numbers.\n");
  count = 0;  limit1 = limit0 / sqrt(2);
  start = times(&before);
  for (j=0; j<nloops; j++) {
    for (i=0; i<ntests; i++) {
      if ((Re(data[i]) < -limit1) || (Re(data[i]) > limit1) ||
	  (Im(data[i]) < -limit1) || (Im(data[i]) > limit1))
	count++;
    }
  }
  stop = times(&after);
  lap(start, &before, stop, &after);
  printf("%d matches.\n", count);

  printf("Running an `opti-box' comparison for complex numbers.\n");
  count = 0;  limit0 = -limit1;
  start = times(&before);
  for (j=0; j<nloops; j++) {
    for (i=0; i<ntests; i++) {
    if ((Re(data[i]) < limit0) || (Re(data[i]) > limit1) ||
	(Im(data[i]) < limit0) || (Im(data[i]) > limit1))
      count++;
    }
  }
  stop = times(&after);
  lap(start, &before, stop, &after);
  printf("%d matches.\n", count);

  printf("Running a `hardcoded box' comparison for complex numbers.\n");
  count = 0;
  start = times(&before);
  for (j=0; j<nloops; j++) {
    for (i=0; i<ntests; i++) {
      if ((Re(data[i]) < -0.707106781186548) ||
	  (Re(data[i]) > 0.707106781186548) ||
	  (Im(data[i]) < -0.707106781186548) ||
	  (Im(data[i]) > 0.707106781186548))
	count++;
    }
  }
  stop = times(&after);
  lap(start, &before, stop, &after);
  printf("%d matches.\n", count);

  printf("Running a `split box' comparison for complex numbers.\n");
  count = 0;
  start = times(&before);
  for (j=0; j<nloops; j++) {
    for (i=0; i<ntests; i++) {
      if (Re(data[i]) < -0.707106781186548)
	count++;
      if (Re(data[i]) > 0.707106781186548)
	count++;
      if (Im(data[i]) < -0.707106781186548)
	count++;
      if (Im(data[i]) > 0.707106781186548)
	count++;
    }
  }
  stop = times(&after);
  lap(start, &before, stop, &after);
  printf("%d matches.\n", count);

  exit(0);
}
