/* $Id: twodim.c,v 1.4 1996/04/11 03:33:03 bert Exp $
 *
 * 2-dimensional exact solution of Schr{\""o}dinger's eqn
 * using short-time propagators
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "complex.h"
#include "fft.h"
#include "oppack.h"
#include "propagate.h"
#include "expect.h"
#include "util.h"

/* directory for temporary files */
#define SCRATCH "/var/tmp/"

/* parameters that can be set by command-line options */
real delta,dt, rx,ry, kx,ky, eps, vee_coef,pack_coef,mu_coef;
unsigned long niter, size;
int debug, intermediate, expect;

/* the generating function for the potential*/
Vprop_gen_f generate_Vprop = generate_Vprop_nelson;

/* global variables for getopt() */
extern char *optarg;
extern int optind;

/***** code *****/

/*
 * describe program usage.
 */
void usage(char* name)
{
  fprintf(stderr, 
"usage: %s [arguments]\n\
options:\n\
   -i #iter          number of iterations\n\
   -@ #iter          save the packet each #iter steps\n\
   -n mesh_size      size of the discrete grid\n\
   -s delta_r        size of the discrete space step\n\
   -t delta_t        size of the discrete time slice\n\
   -x r0x -y r0y     initial packet position\n\
   -X k0x -Y k0y     initial packet momentum\n\
   -e width          initial width of the packet\n\
   -v propV_coef     coefficient for the potential\n\
   -m mu             Nelson x^2/2 coefficient\n\
   -p packet_coef    coefficient for the packet\n\
   -h hbar           value of h-bar\n\
choice of potential:\n\
  (default)          Nelson potential      ( V = mu x^2/2 + (y - x^2/2)^2 )\n\
   -o                harmonic oscillator potential        ( V = k/2 |r|^2 )\n\
   -c                constant potential                           ( V = k )\n\
debugging options:\n\
   -d                turn on debugging\n",
	  name);
  exit (2);
}

/*
 * read and parse the command-line arguments.
 */
void read_arguments(int argc, char **argv)
{
  int c;
  int arg_err=0;

  /* set parameters: first the defaults... */

  debug=0;  intermediate=0;  expect=0;

  niter = 100;
  size = 64;
  delta = .1;  dt = .01; 
  rx=0; ry=0; kx=0; ky=0;
  eps=2.;
  vee_coef=.05;  mu_coef=0.1;  pack_coef=1;

  /* ...then check arguments */

  while ((c = getopt(argc, argv, "i:@:n:s:t:x:y:X:Y:e:v:m:p:h:ocdE?")) != EOF)
    switch (c) {
    case 'i':  niter = atol(optarg);          break;
    case '@':  intermediate = atol(optarg);   break;
    case 'n':  size = atol(optarg);           break;
    case 's':  delta = atof(optarg);          break;
    case 't':  dt = atof(optarg);             break;
    case 'x':  rx = atof(optarg);             break;
    case 'y':  ry = atof(optarg);             break;
    case 'X':  kx = atof(optarg);             break;
    case 'Y':  ky = atof(optarg);             break;
    case 'e':  eps = atof(optarg);            break;
    case 'v':  vee_coef = atof(optarg);       break;
    case 'm':  mu_coef = atof(optarg);        break;
    case 'p':  pack_coef = atof(optarg);      break;
    case 'h':  hbar = atof(optarg);           break;
    case 'o':  generate_Vprop = generate_Vprop_harmonic;  break;
    case 'c':  generate_Vprop = generate_Vprop_constant;  break;
    case 'd':  debug++;                       break;
    case 'E':  expect++;                      break;
    case '?':  arg_err++;                     break;
    }

  if ((optind<argc) || arg_err) {
    usage(progname);
  }
  /* status and assignments that depend on the ones above */

  if (debug) {
    if (intermediate)
      printf("intermediate saves enabled (1 in %d)\n", intermediate);
    printf("niter = %3ld\n", niter);
    printf("size  = %3ld\n", size);
    printf("delta = %10.8" LFMT "g\n", delta);
    printf("dt    = %10.8" LFMT "g\n", dt);
    printf("r0    = (%10.8" LFMT "g,%10.8" LFMT "g)\n", rx, ry);
    printf("k0    = (%10.8" LFMT "g,%10.8" LFMT "g)\n", kx, ky);
    printf("eps   = %10.8" LFMT "g\n", eps);
    printf("mu    = %10.8" LFMT "g\n", mu_coef);
    printf("Vcoef = %10.8" LFMT "g\n", vee_coef);
    printf("pcoef = %10.8" LFMT "g\n\n", pack_coef);
    printf("hbar  = %10.8" LFMT "g\n", hbar);
    printf("mass  = %10.8" LFMT "g\n\n", mass);
  }

  eps *= delta;

  if (debug)
    printf("s_eps = %10.8" LFMT "g\n\n", eps);
}

/***** main program *****/

int main(int argc, char **argv)
{
  unsigned long off, itr;

  /* "real" and "complex" types are defined in complex.h */
  complex *packet,*ipacket, *propV,*propV2, *propK;
  real norm1, norm2;
  complex over;
  char tmpname[128];

  progname = argv[0];

  /* read the command line arguments */

  read_arguments(argc, argv);

  /* allocate everything */

  packet = (complex*)safe_malloc(size*size*sizeof(complex),"packet");
  propK =  (complex*)safe_malloc(size*size*sizeof(complex),"propK");
  propV =  (complex*)safe_malloc(size*size*sizeof(complex),"propV");
  propV2 = (complex*)safe_malloc(size*size*sizeof(complex),"propV2");
  ipacket =(complex*)safe_malloc(size*size*sizeof(complex),"ipacket");

  /* allocate and initialize data */

  generate_packet(packet, size, rx,ry,kx,ky,delta,eps,pack_coef);
  copy_data (packet, ipacket, size);
  fprintf(stdout,"packet initialized (%ldx%ld)\n", size,size);
  if (debug || intermediate) {
    dump_Re(packet, size, SCRATCH "packet0_r");
    dump_Im(packet, size, SCRATCH "packet0_i");
  }

  generate_Kprop (propK, size, dt,delta);
  fprintf(stdout,"propK initialized\n");
  if (debug>1) {
    dump_Re(propK, size, SCRATCH "propK_r");
    dump_Im(propK, size, SCRATCH "propK_i");
  }

  generate_Vprop (propV, size, dt,delta,vee_coef,mu_coef);
  fprintf(stdout,"propV initialized\n");
  if (debug>1) {
    dump_Re(propV, size, SCRATCH "propV_r");
    dump_Im(propV, size, SCRATCH "propV_i");
  }

  generate_Vprop (propV2, size, dt/2,delta,vee_coef,mu_coef);
  fprintf(stdout,"propV2 initialized\n");
  if (debug>1) {
    dump_Re(propV2, size, SCRATCH "propV2_r");
    dump_Im(propV2, size, SCRATCH "propV2_i");
  }

  norm1 = sum_sq(packet,size);
  printf("probability at t=0: %20.16" LFMT "e\n", norm1);

  /* do the actual propagation */
  off = 0;
  itr = niter;

  while (intermediate && (intermediate < itr)) {
    propagate (packet, propV2, propV, propK, size, intermediate, off);
    itr -= intermediate;
    off += intermediate;

    /* intermediate results */
    sprintf(tmpname, SCRATCH "packet%ld_r", off);
    dump_Re(packet, size, tmpname);
    sprintf(tmpname, SCRATCH "packet%ld_i", off);
    dump_Im(packet, size, tmpname);
  }

  propagate (packet, propV2, propV, propK, size, itr, off);
  fprintf(stdout,".. done.\n");

  /* results */
  sprintf(tmpname, SCRATCH "packet%ld_r", niter);
  dump_Re(packet, size, tmpname);
  sprintf(tmpname, SCRATCH "packet%ld_i", niter);
  dump_Im(packet, size, tmpname);

  norm2 = sum_sq(packet,size);
  printf("probability at t=T: %20.16" LFMT "e\n", norm2);
  printf("relative error:    %+21.16" LFMT "e\n", (norm2/norm1)-1);

  over = overlap (ipacket, packet, size);
  over /= norm1;
  printf("overlap at t=%" LFMT "f:   %14.10" LFMT "f (%10.6" LFMT "e + %10.6"
	 LFMT "e i)\n", dt*niter, csqabs(over), Re(over),Im(over));

  free(packet);
  free(propK);
  free(propV);
  free(propV2);
  return 0;
}
