/*****************************************************************************
 *
 * Microchip DeviceNet Stack 
 * (Initializers, Process Manager, and Network Access State Machine)
 *
 *****************************************************************************
 * FileName:        GoDNet.c
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.20.00 or higher
 * Linker:          MPLINK 03.40.00 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 * This file contains the unified control functions for the DeviceNet stack.
 *
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			06/06/03	...	
 * 
 *****************************************************************************/


#include	"dnet.def"				// Global definitions file
#include 	"typedefs.h"			// Global type definitions

#include	"services.h"			// Service codes
#include	"errors.h"				// Error codes
#include	"class.h"				// Class codes

#include	"CAN.H"					// CAN driver
#include	"NASM.H"				// Network Access State Machine managing functions
#include 	"conn.h"				// Connection object 
#include	"dnet.h"				// DeviceNet Object
#include	"ident.h"				// Identity Object
#include	"UEMM.h"				// Unconnected Explicit Messaging Manager
#include	"EMM.h"					// Explicit Messaging Manager

#include	"usrident.h"			// User level Identity object functions
#include	"usrdnet.h"				// User level DeviceNet object functions


#if USE_ACCESS == 1
#pragma	udata access	_A_GO_DNET_FUNCTIONS
#endif





/*********************************************************************
 * Function:        void GoDNetProcessAllMsgEvents(void)
 *
 * PreCondition:  	none  
 *
 * Input:       	none			
 *                  
 * Output:     		none
 * 	
 * Side Effects:   	none 
 *
 * Overview:        This function processes all messaging events in the 
 *					DeviceNet protocol stack.  
 *
 * Note:            This function should be called as often as 
 *					possible to prevent overflow.
 ********************************************************************/
void GoDNetProcessAllMsgEvents(void)
{
	// Capture bus-off errors if any
	_ConnErrorManager();
		
	switch (_aNASMStates)
	{
		// Perform automatic baud rate detection
//		case	_NASM_AUTOBAUD:
//			_ConnRxManager();
//			_NASMAutoBaudStateManager();
//			break;
			
		// Process only duplicate ID events	
		case	_NASM_SENDING_DID:
			_ConnRxManager();
			_NASMSendingDupIDStateManagder();
			break;
			
		// Process only duplicate ID events	
		case	_NASM_WAITING_DID:
			_ConnRxManager();
			_NASMWaitingForDupIDStateManager();
			break;
			
		// Process normal events	
		case	_NASM_ON_LINE:	
			// If any data was received by an existing connection
			if (_ConnRxManager())
			{
				// Process any on-line Duplicate ID requests
				_NASMOnLineStateManager();	
			
				// Process explicit messages
				ExplicitMsgManager();		
			
				// Process unconnected explicit messages		
				UnconnectedExplicitMsgManager();
			}		
			break;
			
		// Process only communication fault events	
		case 	_NASM_COMM_FAULT:
			// Process communication faults
			UsrIdentityCommunicationFaultEvent();
			break;
		
		// Unknown state, reset the states back to a known state
		default:
			UsrIdentityFaultEvent();
			break;
	}
	
	// Process any messages queued to send
	_ConnTxOpenManager();
	
	// Capture any flags indicating data placed on the bus
	_ConnTxManager();
}



/*********************************************************************
 * Function:        void GoDNetProcessAllTickEvents(void)
 *
 * PreCondition:  	none  
 *
 * Input:       	none			
 *                  
 * Output:     		none
 * 	
 * Side Effects:   	none 
 *
 * Overview:        This function processes all timer events in the 
 *					DeviceNet protocol stack.  
 *
 * Note:            This should be called once every TICK_RESOLUTION
 *					milliseconds.
 ********************************************************************/
void GoDNetProcessAllTickEvents(void)
{
	switch (_aNASMStates)
	{			
		// Process only duplicate ID events	
		case	_NASM_WAITING_DID:
			// Process 1 second duplicate ID timer
			_NASMWaitTimeManager();
			break;
			
		// Process normal events	
		case	_NASM_ON_LINE:			
			// Process connection timers and states
			_ConnTimeManager();
			_ConnStateManager();
			break;
	}
}




/*********************************************************************
 * Function:        void GoDNetInitializeAll(void)
 *
 * PreCondition:  	none  
 *
 * Input:       	none			
 *                  
 * Output:     		none
 * 	
 * Side Effects:   	none 
 *
 * Overview:        Initialize everything for the DeviceNet protocol 
 *					stack to function.  
 *
 * Note:            none
 ********************************************************************/
void GoDNetInitializeAll(void)
{
	// Initialize the DeviceNet object
	UsrDNetInitEvent();
	mDNetSetBusOffCount(0);				// Default is 0
	mDNetSetAllocChoice(0);				// Default is 0
	mDNetSetMasterMACID(255);			// Default is FF

	// Initialize the Network Access State Machine
	_NASMInit();
	
	// Initialize identity stuff
	UsrIdentityInitEvent();	

	// Initialize the CAN Driver
	CANInit();
	CANSetBitRate(mDNetGetBaudRate());
	CANOpen();
			
	// Initialize all connection stuff
	mConnInit();
}




