/*****************************************************************************
 *
 * Microchip DeviceNet Stack (Unconnected Explicit Messaging Manager Source)
 *
 *****************************************************************************
 * FileName:        uemm.c
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.20.00 or higher
 * Linker:          MPLINK 03.40.00 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 * This file is simply a managing routine that parses data from the 
 * Unconnected Explicit Messaging Connection for the router. Refer to
 * section 5-5 of the DeviceNet specification.
 * 
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			05/03/03	...	
 * 
 *****************************************************************************/




#include	"dnet.def"				// Global definitions file
#include 	"typedefs.h"

#include	"route.h"				// Router prototypes and macros

#include	"services.h"			// Service codes
#include	"errors.h"				// Error codes
#include	"class.h"				// Class codes

#include 	"conn.h"				// Connection object 

   

/*********************************************************************
 * Function:        unsigned char UnconnectedExplicitMsgManager(void)
 *
 * PreCondition:    
 *
 * Input:       	none		
 *                  
 * Output:      	status of the process	
 *
 * Side Effects:    none
 *
 * Overview:        This function manages any Unconnected Explicit 
 *					Message data and provides parsing for the Router.
 *
 * Note:           
 ********************************************************************/
unsigned char UnconnectedExplicitMsgManager(void)
{
	USINT retStatus;

	retStatus = 0;

	// Do only if the connection object is prepared to receive and transmit
	if (_ConnReadRdy(6) && _ConnWriteRdy(6))
	{
		// Insure that sufficient data has been received to process
		if (uConn6.rx.len >= 4)
		{	
			// Set the buffer pointers and other info based on the connection object settings
			route.pInBuf = uConn6RxBuffer;  
			route.pOutBuf = uConn6TxBuffer + 2;
			route.inBufLen = uConn6.rx.lenMax - 4;
			route.inBufDataLen = uConn6.rx.len - 4;
			route.outBufLen = uConn6.tx.lenMax - 2;
			route.outBufDataLen = 0;

			// Parse the buffer using pointers provided from the connection object
			route.header = *route.pInBuf; route.pInBuf++; 
			route.service = *route.pInBuf; route.pInBuf++;
			
			#if (CLASS_WIDTH_16BIT)
			route.classID.bytes.LSB = *(route.pInBuf); route.pInBuf++;
			route.classID.bytes.MSB = 0;
			#else
			route.classID = *(route.pInBuf); route.pInBuf++;
			#endif
			
			#if (INSTANCE_WIDTH_16BIT)
			route.instanceID.bytes.LSB = *(route.pInBuf); route.pInBuf++;
			route.instanceID.bytes.MSB = 0;
			#else
			route.instanceID = *(route.pInBuf); route.pInBuf++;
			#endif 
			route.attributeID = *route.pInBuf;

			// Route the data to the DeviceNet object, but only if the service
			// code is allocate or release
			#if (CLASS_WIDTH_16BIT)
			if ((route.classID.bytes.LSB == CLASS_DEVICENET) && 
				((route.service == SRVS_ALLOCATE_CONN) || (route.service == SRVS_RELEASE_CONN)))
			#else
			if ((route.classID == CLASS_DEVICENET) && 
				((route.service == SRVS_ALLOCATE_CONN) || (route.service == SRVS_RELEASE_CONN)))
			#endif
			{
				retStatus = RouteMessage();	
			}
			// Otherwise do nothing
			else
			{	
				retStatus = 0;	
			}			
				
			// Write the transmit buffer header and service info
			// all other data should be provided by the DeviceNet Object
			route.pOutBuf = uConn6TxBuffer;
			*route.pOutBuf = route.header; route.pOutBuf++;
			*route.pOutBuf = route.service | 0x80; 	// Set the response bit
			uConn6.tx.len = route.outBufDataLen + 2;
					
			_ConnRead(6);					// Unlock the read for explicit messaging
			
			if (retStatus) _ConnWrite(6);	// Enable write for explicit messaging
		}

		return (retStatus);
	}
	else 
		return (0);
}
