
/*****************************************************************************
 *
 * Microchip DeviceNet Stack (CAN Driver Header)
 *
 *****************************************************************************
 * FileName:        CAN.c
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.20.00 or higher
 * Linker:          MPLINK 03.40.00 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *	This version of the CAN driver for DeviceNet is designed to be very
 *	simple. Filtering options are static and limited. Receive buffering is
 *	is restricted to only the two hardware receive buffers. And transmit 
 *	buffering is restricted to only one of three transmit buffers. However,
 *	the architecture of this driver lends itself to be upgradable to achieve 
 *	better performance without changes to the firmware using this driver.
 * 
 *
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			04/28/03	...	
 * 
 *****************************************************************************/



/*********************************************************************
 * NEAR Access control
 ********************************************************************/	
#if USE_ACCESS == 1
#pragma	udata access	_A_CAN_FUNCTIONS
#define	NEAR near
#else
#define NEAR
#endif 

 

/*********************************************************************
 * Function:        NEAR unsigned char CANOpen(void)
 *
 * PreCondition:   	Bitrate and filters should be setup 
 *
 * Input:       	void
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	 
 *
 * Overview:        Open CAN communications 
 *
 * Note:            This function should be interprited as a request
 *					rather than a command. Use CANIsOpen() to 
 *					verify the bus has been opened.
 ********************************************************************/
NEAR unsigned char CANOpen(void);


/*********************************************************************
 * Function:        NEAR unsigned char CANClose(void)
 *
 * PreCondition:   	Bus communications should be open. 
 *
 * Input:       	void
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	 
 *
 * Overview:        Close CAN communications 
 *
 * Note:            This function should be interprited as a request
 *					rather than a command. Use CANIsOpen() to 
 *					verify the bus is still open.
 ********************************************************************/
NEAR unsigned char CANClose(void);


/*********************************************************************
 * Function:        NEAR unsigned char CANIsOpen(void)
 *
 * PreCondition:   	 
 *
 * Input:       	void
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	 
 *
 * Overview:        Return the status of communication. 
 *					TRUE = Open, FALSE = Closed 
 *
 * Note:            
 ********************************************************************/
NEAR unsigned char CANIsOpen(void);


/*********************************************************************
 * Function:        NEAR unsigned char CANSetFilter(NEAR unsigned int filterID)
 *
 * PreCondition:   	 
 *
 * Input:       	16-bit filterID
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	Communication may be temporarily interrupted
 *
 * Overview:        Adds a filter to the scan list.
 *
 * Note:            No scan list is actually created in this version;
 *					a hardware filter is to allow a CAN ID or group 
 *					of IDs to be received.
 ********************************************************************/
NEAR unsigned char CANSetFilter(NEAR unsigned int filterID);


/*********************************************************************
 * Function:        NEAR unsigned char CANClrFilter(NEAR unsigned int filterID)
 *
 * PreCondition:   	 
 *
 * Input:       	16-bit filterID
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	Communication may be temporarily interrupted
 *
 * Overview:        Removes a filter from the scan list. 
 *
 * Note:           	No scan list is actually created in this version;
 *					a hardware filter is used to allow a CAN ID or 
 *					group of IDs to	be received.
 ********************************************************************/
NEAR unsigned char CANClrFilter(NEAR unsigned int filterID);


/*********************************************************************
 * Function:        NEAR unsigned char CANSetBitRate(NEAR unsigned char bitrate)
 *
 * PreCondition:   	 
 *
 * Input:       	bitrate
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	Communication may be temporarily interrupted
 *
 * Overview:        Changes the bitrate of the node. 
 *
 * Note:           	
 ********************************************************************/
NEAR unsigned char CANSetBitRate(NEAR unsigned char bitrate);


/*********************************************************************
 * Function:        NEAR unsigned char CANIsBusError(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	
 *
 * Overview:        Returns an error code. 
 *
 * Note:           	
 ********************************************************************/
NEAR unsigned char CANIsBusOffError(void);


/*********************************************************************
 * Function:        NEAR unsigned char CANIsRxRdy(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	
 *
 * Overview:        Returns TRUE if there is data waiting. 
 *
 * Note:           	
 ********************************************************************/
NEAR unsigned char CANIsRxRdy(void);


/*********************************************************************
 * Function:        void CANRead(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	none
 *
 * Side Effects:   	
 *
 * Overview:        Indicates to the module that the data has been read.
 *
 * Note:           	This effectively frees the active hardware buffer 
 *					to continue receiving.
 ********************************************************************/
void CANRead(void);


/*********************************************************************
 * Function:        NEAR unsigned char CANIsTxRdy(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	
 *
 * Overview:        Returns TRUE if the transmit engine is ready to 
 *					accept new data to transmit.
 *
 * Note:           	This function is as simple as it gets, i.e. direct
 *					access to only one TX buffer. 
 ********************************************************************/
NEAR unsigned char CANIsTxRdy(void);


/*********************************************************************
 * Function:        NEAR unsigned char CANIsMsgSent(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	
 *
 * Overview:        Returns the tag of the data that was last transmitted. 
 *					Any additional calls without additional transmissions
 *					will return NULL.
 *
 * Note:           	This is useful to determine when and who's data was 
 *					placed on the bus. This may seem redundent since
 *					only one hardware buffer is used; however, this
 *					architecture allows for expansion to deep software/
 *					hardware buffer designs without major changes to 
 *					the upper level firmware.
 ********************************************************************/
NEAR unsigned char CANIsMsgSent(void);


/*********************************************************************
 * Function:        void CANSend(NEAR unsigned char txTag)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	none
 *
 * Side Effects:   	
 *
 * Overview:        Tell the transmit engine that the loaded data is
 *					is ready to send.
 *
 * Note:          	Data may not be sent immediately. The tag is used
 *					to trace the transmission since other data may
 *					also be queued to transmit. 	
 ********************************************************************/
void CANSend(NEAR unsigned char txTag);


/*********************************************************************
 * Function:        NEAR unsigned int CANGetRxCID(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned int connection ID (11-bit CAN ID)
 *
 * Side Effects:   	
 *
 * Overview:        Get the connection ID from the current receive
 *					message. 
 *
 * Note:          	The CAN ID is assumed to be left justified within
 *					the 16-bit word. 	
 ********************************************************************/
NEAR unsigned int CANGetRxCID(void);


/*********************************************************************
 * Function:        NEAR unsigned char CANGetRxCnt(void)
 *
 * PreCondition:   	Data should have been received prior to calling 
 *					this function; otherwise, it is likely the returned
 *					count is either not valid or associated to the 
 *					previous message. Use CANIsRxRdy() to determine if 
 *					data is available.
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char count of the data
 *
 * Side Effects:   	
 *
 * Overview:        This function returns the number of bytes waiting
 *					in the receive buffer.
 *
 * Note:          	
 ********************************************************************/
NEAR unsigned char CANGetRxCnt(void);


/*********************************************************************
 * Function:        unsigned char * NEAR CANGetRxDataPtr(void)
 *
 * PreCondition:   	Data should have been received prior to calling 
 *					this function; otherwise, it is likely the returned
 *					pointer is either not valid or associated to the 
 *					previous message. Use CANIsRxRdy() to determine if 
 *					data is available. 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:   	
 *
 * Overview:        This function returns the pointer to the RX buffer.
 *					
 * Note:          	Use the direct block copy methods for faster data
 *					movement. 	
 ********************************************************************/
unsigned char * NEAR CANGetRxDataPtr(void);


/*********************************************************************
 * Function:        void CANGetRxDataTyp0(unsigned char * NEAR usrBuf)
 *					void CANGetRxDataTyp1(unsigned char * NEAR usrBuf)
 *					void CANGetRxDataTyp2(unsigned char * NEAR usrBuf)
 *
 * PreCondition:   	Data should have been received prior to calling 
 *					this function; otherwise, it is likely the returned
 *					count is either not valid or associated to the 
 *					previous message. Use CANIsRxRdy() to determine if 
 *					data is available. 
 *
 * Input:       	pointer to user supplied buffer
 *                  
 * Output:      	void
 *
 * Side Effects:   	
 *
 * Overview:        Get block data from the buffer. 
 *					
 * Note:          	These functions copy blocks of data rather than 
 *					single bytes. This method is very fast; however,
 *					data limits are not checked. It is up to the 
 *					caller to verify the count to toss data that is  
 *					not valid.
 ********************************************************************/
void CANGetRxDataTyp0(unsigned char * NEAR usrBuf);
void CANGetRxDataTyp1(unsigned char * NEAR usrBuf);
void CANGetRxDataTyp2(unsigned char * NEAR usrBuf);


/*********************************************************************
 * Function:        void CANPutTxCID(NEAR unsigned int txCID)
 *
 * PreCondition:   	The function CANIsTxRdy() must be called prior
 *					to using this function; otherwise, the previous
 *					transmission request may be corrupted. 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:   	
 *
 * Overview:        Write the CID to the TX buffer.
 *					
 * Note:          	 	
 ********************************************************************/
void CANPutTxCID(NEAR unsigned int txCID);


/*********************************************************************
 * Function:        void CANPutTxCnt(NEAR unsigned char txCount)
 *
 * PreCondition:   	The function CANIsTxRdy() must be called prior
 *					to using this function; otherwise, the previous
 *					transmission request may be corrupted.  
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:   	
 *
 * Overview:        Write the count to the TX buffer.
 *					
 *
 * Note:          	 	
 ********************************************************************/
void CANPutTxCnt(NEAR unsigned char txCount);


/*********************************************************************
 * Function:        unsigned char * NEAR CANGetTxDataPtr(void)
 *
 * PreCondition:   	The function CANIsTxRdy() must be called prior
 *					to using this function; otherwise, the pointer
 *					may be invalid.
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:   	
 *
 * Overview:        Return a pointer to the buffer.
 *					
 *
 * Note:          	 	
 ********************************************************************/
unsigned char * NEAR CANGetTxDataPtr(void);


/*********************************************************************
 * Function:        void CANPutTxDataTyp0(unsigned char * NEAR usrBuf)
 *					void CANPutTxDataTyp1(unsigned char * NEAR usrBuf)
 *					void CANPutTxDataTyp2(unsigned char * NEAR usrBuf)
 *
 * PreCondition:   	The function CANIsTxRdy() must be called prior
 *					to using this function; otherwise, the previous
 *					transmission request may be corrupted.  
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:   	
 *
 * Overview:        Return a pointer to the buffer.
 *					
 *
 * Note:          	 	
 ********************************************************************/
void CANPutTxDataTyp0(unsigned char * NEAR usrBuf);
void CANPutTxDataTyp1(unsigned char * NEAR usrBuf);
void CANPutTxDataTyp2(unsigned char * NEAR usrBuf);


/*********************************************************************
 * Function:        void CANInit(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:   	
 *
 * Overview:        Initialize the module.
 *					
 *
 * Note:          	
 ********************************************************************/
void CANInit(void);




#pragma	udata













		
		 
