;*******************************************************************************;
;*		                                                                        ;
;*  This implements a generic library functionality to support SPI              ;
;*  for PIC16 family                                                            ;
;*  It adds additional functionality of Rx/Tx user defined Cicular buffer       ;
;*                                                                              ;
;*******************************************************************************;
;* FileName:            SPIMInt.asm                                     ;    
;* Dependencies:        P18xxx.inc                                      ;
;*                      SPIMInt.Def                                     ;
;* Processor:           PIC18xxxx                                       ;
;* Assembler:           MPASMWIN 02.70.02 or higher                     ;
;* Linker:              MPLINK 2.33.00 or higher                        ;
;* Company:             Microchip Technology, Inc.                      ;
;*                                                                      ;
;* Software License Agreement                                           ;
;*                                                                      ;
;* The software supplied herewith by Microchip Technology Incorporated  ;
;* (the "Company") for its PICmicro Microcontroller is intended and    ;
;* supplied to you, the Company's customer, for use solely and          ;
;* exclusively on Microchip PICmicro Microcontroller products. The      ;
;* software is owned by the Company and/or its supplier, and is         ;
;* protected under applicable copyright laws. All rights are reserved.  ;
;* Any use in violation of the foregoing restrictions may subject the   ;
;* user to criminal sanctions under applicable laws, as well as to      ;
;* civil liability for the breach of the terms and conditions of this   ;
;* license.                                                             ;
;*                                                                      ;
;* THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES,    ;
;* WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED    ;
;* TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A          ;
;* PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,    ;
;* IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR           ;
;* CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.                    ;
;*                                                                      ;
;*                                                                      ;
;* ANY SPECIAL DESCRIPTION THAT MIGHT BE USEFUL FOR THIS FILE.          ;
;*                                                                      ;
;* Author               Date            Comment                         ;
;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~;
;* Vidyadhar       Apr 14, 2003    Initial Release (V1.0)               ;
;*                                                                      ;
;***********************************************************************;


;***********************************************************************;
SPIMINTCODE   CODE                                                      ;
                                                                        ;
;***********************************************************************;
; Function: SPIMIntInit                                                 ;
;                                                                       ;
; PreCondition: None                                                    ;
;                                                                       ;
; Overview:                                                             ;
;       This routine is used for MSSP/SSP/BSSP Module Initialization    ;
;       It initializes Module according to compile time selection and   ;
;       flushes the Rx and Tx buffer. It clears all SPI errors          ;
;                                                                       ;
; Input: CLM options                                                    ;
;                                                                       ;
;                                                                       ;
; Output: None                                                          ;
;                                                                       ;
; Side Effects: Bank selection bits and 'W' register are changed        ;
;                                                                       ;
; Stack requirement: 1 level deep                                       ;
;                                                                       ;
;***********************************************;***********************;
                                                ;
SPIMIntInit:                                    ;
                                                ;
        GLOBAL  SPIMIntInit                     ;
                                                ;
        movlw   SPIM_SPEED                      ;
        movwf   SSPCON1                         ;
                                                ;
        bsf     INTCON,PEIE                     ;
        bsf     INTCON,GIE                      ;
                                                ;
        BANKSEL _vSPIMIntTxBufWrPtr             ;
        clrf    _vSPIMIntTxBufWrPtr             ;
        clrf    _vSPIMIntTxBufRdPtr             ;
        clrf    _vSPIMIntRxBufWrPtr             ;
        clrf    _vSPIMIntRxBufRdPtr             ;
        clrf    _vSPIMTxDataCount               ;
        clrf    _vSPIMRxDataCount               ;
                                                ;
        clrf    vSPIMIntStatus                  ;
        bsf    vSPIMIntStatus,SPIMTxBufEmpty    ;
        bsf    vSPIMIntStatus,SPIMRxBufEmpty    ;
                                                ;
        return                                  ;
                                                ;
;***********************************************;18



;***********************************************************************;
; Function: SPIMIntPut                                                  ;
;                                                                       ;
; PreCondition: SPIMIntInit should have called.                         ;
;                                                                       ;
; Overview:                                                             ;
;       This writes data into buffer if the old data from buffer is     ;
;    being transmitted else start transmiting this data.                ;
;                                                                       ;
; Input: 'W' Register                                                   ;
;                                                                       ;
; Output: Buffer or SPI Bus                                             ;
;                                                                       ;
; Side Effects: None                                                    ;
;                                                                       ;
; Stack requirement: 1 level deep                                       ;
;                                                                       ;
;***********************************************;***********************;
                                                ;
SPIMIntPut:                                     ;
                                                ;
        GLOBAL  SPIMIntPut                      ;
                                                ;
        btfss   vSPIMIntStatus,SPIMBusy         ;check is busy
        bra     SPIDirectTx                     ;
                                                ;
        #if SPIM_TX_BUF_LENGTH == 1             ;
                                                ;
        movwf   vSPIMIntTxBuffer                ;
        bsf     vSPIMIntStatus,SPIMTxBufFull    ;
        bcf     vSPIMIntStatus,SPIMTxBufEmpty   ;9
                                                ;
        #else                                   ;
                                                ;
        bcf     PIE1,SSPIE                      ;
                                                ;
        call    _SPIMIntWrTxBuf                 ;Save in buffer
                                                ;
        BANKSEL _vSPIMTxDataCount               ;
        incf    _vSPIMTxDataCount,f             ;Increment Data Count
                                                ;
        bsf     PIE1,SSPIE                      ;
                                                ;
        #endif                                  ;
                                                ;
        return                                  ;12
                                                ;
SPIDirectTx                                     ;
        bsf     vSPIMIntStatus,SPIMBusy         ;If not busy set busy
                                                ;
        movwf   SSPBUF                          ;write data into SSPBUf
                                                ;
        bcf     PIR1,SSPIF                      ;clear serial_sync intrupt flag
                                                ;
        bsf     PIE1,SSPIE                      ;enable SSP module interrupt
                                                ;
        return                                  ;
                                                ;
;***********************************************;46    



        #if SPIM_TX_BUF_LENGTH != 1             ;
                                                ;
;***********************************************************************;
; Function: _SPIMIntWrTxBuf                                             ;
;                                                                       ;
; PreCondition: None.                                                   ;
;                                                                       ;
; Overview:                                                             ;
;       This writes data into buffer.                                   ;
;                                                                       ;
; Input: 'W' Register                                                   ;
;                                                                       ;
; Output: None                                                          ;
;                                                                       ;
; Side Effects: None                                                    ;
;                                                                       ;
; Stack requirement: 1 level deep                                       ;
;                                                                       ;
;***********************************************;***********************;
                                                ;
_SPIMIntWrTxBuf:                                ;
        btfsc    vSPIMIntStatus,SPIMTxBufFull   ;
        return                                  ;
                                                ;
        movff   WREG,_vSPIMIntTempReg           ;save the wreg content (data) in temflg
                                                ;
        movff   FSR0H,_vSPIMIntDupFSRH          ;
        movff   FSR0L,_vSPIMIntDupFSR           ;Save FSR
                                                ;
        #ifdef  _DONT_USE_LFSR                  ;
                                                ;
        BANKSEL _vSPIMIntTxBufWrPtr             ;
        movlw   HIGH(vSPIMIntTxBuffer)          ;
        movwf   FSR0H                           ;
        movlw   LOW(vSPIMIntTxBuffer)           ;load wreg with write pointer address
        addwf   _vSPIMIntTxBufWrPtr,w           ;
        movwf   FSR0L                           ;
                                                ;
        #else                                   ;
                                                ;
        BANKSEL _vSPIMIntTxBufWrPtr             ;
        lfsr    FSR0,vSPIMIntTxBuffer           ;load fsr with read pointer address
        movf    _vSPIMIntTxBufWrPtr,w           ;
        addwf   FSR0L                           ;
                                                ;
        #endif                                  ;
                                                ;
        btfsc   STATUS,C                        ;
        incf    FSR0H,f                         ;
                                                ;
        movff    _vSPIMIntTempReg,INDF0         ;move tempreg content to write pointer pointing location
                                                ;
        movff   _vSPIMIntDupFSR,FSR0L           ;
        movff   _vSPIMIntDupFSRH,FSR0H          ;Retrive FSR
                                                ;
        incf    _vSPIMIntTxBufWrPtr,f           ;increment write pointer
        movlw   SPIM_TX_BUF_LENGTH              ;
        xorwf   _vSPIMIntTxBufWrPtr,w           ;
        btfsc   STATUS,Z                        ;
        clrf    _vSPIMIntTxBufWrPtr             ;
                                                ;
        movf    _vSPIMIntTxBufWrPtr,w           ;
        xorwf   _vSPIMIntTxBufRdPtr,w           ;Check is buffer full
        btfsc   STATUS,Z                        ;
        bsf     vSPIMIntStatus,SPIMTxBufFull    ;
                                                ;
        bcf     vSPIMIntStatus,SPIMTxBufEmpty   ;
                                                ;
        return                                  ;
                                                ;
;***********************************************;34

        #endif                                  ;
                                                ;


;***********************************************************************;
; Function: SPIMIntGet                                                  ;
;                                                                       ;
; PreCondition: SPIMIntRecv should have called.                         ;
;                                                                       ;
; Overview:                                                             ;
;       This reads data from buffer.                                    ;
;                                                                       ;
; Input: Buffer                                                         ;
;                                                                       ;
; Output: 'W' Register                                                  ;
;                                                                       ;
; Side Effects: None                                                    ;
;                                                                       ;
; Stack requirement: 1 level deep                                       ;
;                                                                       ;
;***********************************************;***********************;
                                                ;
SPIMIntGet:                                     ;
                                                ;
        GLOBAL  SPIMIntGet                      ;
                                                ;
        #if SPIM_RX_BUF_LENGTH == 1             ;
                                                ;
        movf    vSPIMIntRxBuffer,W              ;
        bcf     vSPIMIntStatus,SPIMRxBufFull    ;
        bsf     vSPIMIntStatus,SPIMRxBufEmpty   ;7
                                                ;
        #else                                   ;
                                                ;
        bcf     PIE1,SSPIE                      ;
                                                ;
        call    _SPIMIntRdRxBuf                 ;Read the byte from buffer
                                                ;
        bsf     PIE1,SSPIE                      ;
                                                ;
        #endif                                  ;
                                                ;
        return                                  ;8
                                                ;
;***********************************************;41    



        #if SPIM_RX_BUF_LENGTH != 1             ;
                                                ;
;***********************************************************************;
; Function: _SPIMIntRdRxBuf                                             ;
;                                                                       ;
; PreCondition: None.                                                   ;
;                                                                       ;
; Overview:                                                             ;
;       This reads data from buffer.                                    ;
;                                                                       ;
; Input: None                                                           ;
;                                                                       ;
; Output: 'W' Register                                                  ;
;                                                                       ;
; Side Effects: None                                                    ;
;                                                                       ;
; Stack requirement: 1 level deep                                       ;
;                                                                       ;
;***********************************************;***********************;
                                                ;
_SPIMIntRdRxBuf:                                ;
                                                ;
        btfsc    vSPIMIntStatus,SPIMRxBufEmpty  ;
        return                                  ;
                                                ;
        movff   FSR0H,_vSPIMIntDupFSRH          ;
        movff   FSR0L,_vSPIMIntDupFSR           ;Save FSR
                                                ;
        #ifdef  _DONT_USE_LFSR                  ;
                                                ;
        BANKSEL _vSPIMIntRxBufRdPtr             ;
        movlw   HIGH(vSPIMIntRxBuffer)          ;load wreg with read pointer address
        movwf   FSR0H                           ;
        movlw   LOW(vSPIMIntRxBuffer)           ;
        addwf   _vSPIMIntRxBufRdPtr,w           ;load fsr with read pointer address
        movwf   FSR0L                           ;
                                                ;
        #else                                   ;
                                                ;
        BANKSEL _vSPIMIntRxBufRdPtr             ;
        lfsr    FSR0,vSPIMIntRxBuffer           ;load fsr with read pointer address
        movf    _vSPIMIntRxBufRdPtr,w           ;
        addwf   FSR0L                           ;
                                                ;
        #endif                                  ;
                                                ;
        btfsc   STATUS,C                        ;
        incf    FSR0H,f                         ;
                                                ;
        incf    _vSPIMIntRxBufRdPtr,f           ;increment read pointer
        movlw   SPIM_RX_BUF_LENGTH              ;
        xorwf   _vSPIMIntRxBufRdPtr,w           ;
        btfsc   STATUS,Z                        ;
        clrf    _vSPIMIntRxBufRdPtr             ;
                                                ;
        movf    _vSPIMIntRxBufRdPtr,w           ;
        xorwf   _vSPIMIntRxBufWrPtr,w           ;Check is buffer empty
        btfsc   STATUS,Z                        ;
        bsf     vSPIMIntStatus,SPIMRxBufEmpty   ;
                                                ;
        movf    INDF0,w                         ;move the content of read pointer address to W
                                                ;
        movff   _vSPIMIntDupFSR,FSR0L           ;
        movff   _vSPIMIntDupFSRH,FSR0H          ;Retrive FSR
                                                ;
        bcf     vSPIMIntStatus,SPIMRxBufFull    ;
        bcf     vSPIMIntStatus,SPIMRxBufOverFlow;
                                                ;
        return                                  ;
                                                ;
;***********************************************;33

        #endif                                  ;
                                                ;


;***********************************************************************;
; Function: SPIMIntSetGetCount                                          ;
;                                                                       ;
; PreCondition: SPIMIntInit should have called.                         ;
;                                                                       ;
; Overview:                                                             ;
;       This receives data from Slave.                                  ;
;                                                                       ;
; Input: 'W' Register                                                   ;
;                                                                       ;
; Output: None                                                          ;
;                                                                       ;
; Side Effects: None                                                    ;
;                                                                       ;
; Stack requirement: 1 level deep                                       ;
;                                                                       ;
;***********************************************;***********************;
                                                ;
SPIMIntSetGetCount                              ;
                                                ;
        GLOBAL  SPIMIntSetGetCount              ;
                                                ;
        BANKSEL _vSPIMRxDataCount               ;
        addwf   _vSPIMRxDataCount,f             ;Append Data Count
                                                ;
        btfss   vSPIMIntStatus,SPIMBusy         ;check is busy
        bra     SPINoBusy                       ;
                                                ;
        #if SPIM_TX_BUF_LENGTH == 1             ;
                                                ;
        clrf    _vSPIMStartRxCount              ;
        bsf     _vSPIMStartRxCount,0            ;
                                                ;
        #else                                   ;
                                                ;
        movf    _vSPIMTxDataCount,w             ;If busy
;        addlw    001h                          ;
        movwf   _vSPIMStartRxCount              ;Record the count to start reception
                                                ;
        #endif                                  ;
                                                ;
        return                                  ;
                                                ;
SPINoBusy                                       ;
        bsf     vSPIMIntStatus,SPIMBusy         ;If not busy set busy
                                                ;
        bsf     vSPIMIntStatus,SPIMSave         ;If not busy set busy
                                                ;
        movlw   000h                            ;
        movwf   SSPBUF                          ;
                                                ;
        bsf     PIE1,SSPIE                      ;enable SSP module interrupt
                                                ;
        return                                  ;
                                                ;
;***********************************************;13    




;***********************************************************************;
; Function: SPIMIntISR                                                  ;
;                                                                       ;
; PreCondition: Generation of Interrupt vector                          ;
;                                                                       ;
; Overview:                                                             ;
;       This is a Interrupt service routine for Serial Interrupt.       ;
;       It handles Reception and Transmission of data on interrupt.     ;
;       Call it from Interrupt service routine.                         ;
;                                                                       ;
; Input:                                                                ;
;    Data in Buffer while transmission and SSPBUF while reception       ;
;                                                                       ;
; Output:                                                               ;
;    If data is received it puts it in vSPIMIntBuffer and accordingly   ;
;    adjusts the vSPIMIntBufWrPtr and clears SPIMIntBufEmpty flag.      ;
;    If Buffer becomes full then it will set SPIMIntBufFull flag.       ;
;    If data is received when buffer was full it will set SPIMBufOF     ;
;    flag to indicate that transmitted data has been missed because     ;
;    of full Buffer. It will set ErrDataNotRcvd flag.                   ;
;                                                                       ;
;       If last data is transmitted then it will transmit next pending  ;
;       data if any. It will accordingly adjust the _vSPIIntBufRdPtr    ;
;    and clears the SPIIntBufFull flag. Which indicates that space      ;
;    is available for data in vSPIMIntBuffer.                           ;
;                                                                       ;
; Side Effects: None                                                    ;
;                                                                       ;
; Stack requirement: 2 level deep                                       ;
;                                                                       ;
;***********************************************;***********************;
                                                ;
SPIMIntISR:                                     ;
        GLOBAL  SPIMIntISR                      ;
                                                ;
        btfss   PIR1,SSPIF                      ;check which interrupt has occurred sspif or bclif
        return                                  ;
                                                ;
        movf    SSPCON1,w                       ;
        andlw   00ch                            ;
        btfss   STATUS,Z                        ;
        return                                  ;check is Master Mode
                                                ;
;-----------------------------------------------;
                                                ;
        movf    SSPBUF,w                        ;
                                                ;
        btfss   vSPIMIntStatus,SPIMSave         ;check is data to be stored
        bra     SPIMNoSave                      ;
                                                ;If so
        btfsc   vSPIMIntStatus,SPIMRxBufFull    ;check is buffer empty
        bra     SPIMRxBufFullErr                ;
                                                ;
        #if SPIM_RX_BUF_LENGTH == 1             ;
                                                ;
        movwf   vSPIMIntRxBuffer                ;
        bsf     vSPIMIntStatus,SPIMRxBufFull    ;
        bcf     vSPIMIntStatus,SPIMRxBufEmpty   ;+1
                                                ;
        #else                                   ;
                                                ;
        call    _SPIMIntWrRxBuf                 ;store in buffer
                                                ;
        #endif                                  ;
                                                ;
        BANKSEL _vSPIMRxDataCount               ;
        decf    _vSPIMRxDataCount,f             ;decrement RxData Count
                                                ;
SPIMNoSave                                      ;
        btfsc   vSPIMIntStatus,SPIMTxBufEmpty   ;
        bra     SPIChkRx                        ;
                                                ;
        #if SPIM_TX_BUF_LENGTH == 1             ;
                                                ;
        movf    vSPIMIntTxBuffer,W              ;
        bcf     vSPIMIntStatus,SPIMTxBufFull    ;
        bsf     vSPIMIntStatus,SPIMTxBufEmpty   ;-2
                                                ;
        #else                                   ;
                                                ;
        call    _SPIMIntRdTxBuf                 ;
                                                ;
        BANKSEL _vSPIMTxDataCount               ;
        decf    _vSPIMTxDataCount,f             ;
                                                ;
        #endif                                  ;
                                                ;
        movf    _vSPIMRxDataCount,f             ;decrement RxData Count
        btfss   STATUS,Z                        ;check is '0'
        bra     SPITxRx                         ;
                                                ;
        dcfsnz  _vSPIMStartRxCount,f            ;decrement RxData Count
        bsf     vSPIMIntStatus,SPIMSave         ;
                                                ;
SPITxRx                                         ;
        movwf   SSPBUF                          ;
                                                ;
        bcf     PIR1,SSPIF                      ;
        return                                  ;32
                                                ;
SPIChkRx                                        ;
        bsf     vSPIMIntStatus,SPIMSave         ;
        BANKSEL _vSPIMRxDataCount               ;
        movlw   000h                            ;
        movf    _vSPIMRxDataCount,f             ;decrement RxData Count
        btfss   STATUS,Z                        ;check is '0'
        bra     SPITxRx                         ;
        bcf     vSPIMIntStatus,SPIMSave         ;
                                                ;
;-----------------------------------------------;
SPIMEnd                                         ;
        bcf     vSPIMIntStatus,SPIMBusy         ;
        bcf     PIR1,SSPIF                      ;
        bcf     PIE1,SSPIE                      ;
        return                                  ;
                                                ;
SPIMRxBufFullErr                                ;
        bsf     vSPIMIntStatus,SPIMRxBufOverFlow;
        return                                  ;
                                                ;
;***********************************************;32 + 33 + 34



        #if SPIM_TX_BUF_LENGTH != 1             ;
                                                ;
;***********************************************************************;
; Function: _SPIMIntRdTxBuf                                             ;
;                                                                       ;
; PreCondition: None.                                                   ;
;                                                                       ;
; Overview:                                                             ;
;       This reads data from buffer.                                    ;
;                                                                       ;
; Input: None                                                           ;
;                                                                       ;
; Output: 'W' Register                                                  ;
;                                                                       ;
; Side Effects: None                                                    ;
;                                                                       ;
; Stack requirement: 1 level deep                                       ;
;                                                                       ;
;***********************************************;***********************;
                                                ;
_SPIMIntRdTxBuf:                                ;
                                                ;
        btfsc   vSPIMIntStatus,SPIMTxBufEmpty   ;
        return                                  ;
                                                ;
        movff   FSR0H,_vSPIMIntDupFSRH          ;
        movff   FSR0L,_vSPIMIntDupFSR           ;Save FSR
                                                ;
        #ifdef  _DONT_USE_LFSR                  ;
                                                ;
        BANKSEL _vSPIMIntTxBufRdPtr             ;
        movlw   HIGH(vSPIMIntTxBuffer)          ;load wreg with read pointer address
        movwf   FSR0H                           ;
        movlw   LOW(vSPIMIntTxBuffer)           ;
        addwf   _vSPIMIntTxBufRdPtr,w           ;load fsr with read pointer address
        movwf   FSR0L                           ;
                                                ;
        #else                                   ;
                                                ;
        BANKSEL _vSPIMIntTxBufRdPtr             ;
        lfsr    FSR0,vSPIMIntTxBuffer           ;load fsr with read pointer address
        movf    _vSPIMIntTxBufRdPtr,w           ;
        addwf   FSR0L                           ;
                                                ;
        #endif                                  ;
                                                ;
        btfsc   STATUS,C                        ;
        incf    FSR0H,f                         ;
                                                ;
        incf    _vSPIMIntTxBufRdPtr,f           ;increment read pointer
        movlw   SPIM_TX_BUF_LENGTH              ;
        xorwf   _vSPIMIntTxBufRdPtr,w           ;
        btfsc   STATUS,Z                        ;
        clrf    _vSPIMIntTxBufRdPtr             ;
                                                ;
        movf    _vSPIMIntTxBufRdPtr,w           ;
        xorwf   _vSPIMIntTxBufWrPtr,w           ;Check is buffer empty
        btfsc   STATUS,Z                        ;
        bsf     vSPIMIntStatus,SPIMTxBufEmpty   ;
                                                ;
        movf    INDF0,W                         ;read the content of read pointer address
                                                ;
        movff   _vSPIMIntDupFSR,FSR0L           ;
        movff   _vSPIMIntDupFSRH,FSR0H          ;Retrive FSR
                                                ;
        bcf     vSPIMIntStatus,SPIMTxBufFull    ;
                                                ;
        return                                  ;
                                                ;
;***********************************************;

        #endif                                  ;
                                                ;


        #if SPIM_RX_BUF_LENGTH != 1             ;
                                                ;
;***********************************************************************;
; Function: _SPIMIntWrRxBuf                                             ;
;                                                                       ;
; PreCondition: None.                                                   ;
;                                                                       ;
; Overview:                                                             ;
;       This writes data into buffer.                                   ;
;                                                                       ;
; Input: 'W' Register                                                   ;
;                                                                       ;
; Output: None                                                          ;
;                                                                       ;
; Side Effects: None                                                    ;
;                                                                       ;
; Stack requirement: 1 level deep                                       ;
;                                                                       ;
;***********************************************;***********************;
                                                ;
_SPIMIntWrRxBuf:                                ;
        btfsc    vSPIMIntStatus,SPIMRxBufFull   ;
        return                                  ;
                                                ;
        movff   WREG,_vSPIMIntTempReg           ;save the wreg content (data) in temflg
                                                ;
        movff   FSR0H,_vSPIMIntDupFSRH          ;
        movff   FSR0L,_vSPIMIntDupFSR           ;Save FSR
                                                ;
        movff   FSR0H,_vSPIMIntDupFSRH          ;Save FSR
        movff   FSR0L,_vSPIMIntDupFSR           ;
                                                ;
        #ifdef  _DONT_USE_LFSR                  ;
                                                ;
        BANKSEL _vSPIMIntRxBufWrPtr             ;
        movlw   HIGH(vSPIMIntRxBuffer)          ;
        movwf   FSR0H                           ;
        movlw   LOW(vSPIMIntRxBuffer)           ;load wreg with write pointer address
        addwf   _vSPIMIntRxBufWrPtr,w           ;
        movwf   FSR0L                           ;
                                                ;
        #else                                   ;
                                                ;
        BANKSEL _vSPIMIntRxBufWrPtr             ;
        lfsr    FSR0,vSPIMIntRxBuffer           ;load fsr with read pointer address
        movf    _vSPIMIntRxBufWrPtr,w           ;
        addwf   FSR0L                           ;
                                                ;
        #endif                                  ;
                                                ;
        btfsc   STATUS,C                        ;
        incf    FSR0H,f                         ;
                                                ;
        movff    _vSPIMIntTempReg,WREG          ;
        movwf   INDF0                           ;move wreg content to write pointer pointing location
                                                ;
        movff   _vSPIMIntDupFSR,FSR0L           ;
        movff   _vSPIMIntDupFSRH,FSR0H          ;Retrive FSR
                                                ;
        incf    _vSPIMIntRxBufWrPtr,f           ;increment write pointer
        movlw   SPIM_RX_BUF_LENGTH              ;
        xorwf   _vSPIMIntRxBufWrPtr,w           ;
        btfsc   STATUS,Z                        ;
        clrf    _vSPIMIntRxBufWrPtr             ;
                                                ;
        movf    _vSPIMIntRxBufWrPtr,w           ;
        xorwf   _vSPIMIntRxBufRdPtr,w           ;Check is buffer full
        btfsc   STATUS,Z                        ;
        bsf     vSPIMIntStatus,SPIMRxBufFull    ;
                                                ;
        bcf     vSPIMIntStatus,SPIMRxBufEmpty   ;
                                                ;
        return                                  ;
                                                ;
;***********************************************;

        #endif                                  ;
                                                ;



;***********************************************************************;
; Function: SPIMIntDiscardRxBuf                                         ;
;                                                                       ;
; PreCondition: None.                                                   ;
;                                                                       ;
; Overview:                                                             ;
;       This flushes the buffer.                                        ;
;                                                                       ;
; Input: 'W' Register                                                   ;
;                                                                       ;
; Output: Buffer                                                        ;
;                                                                       ;
; Side Effects: None                                                    ;
;                                                                       ;
; Stack requirement: 1 level deep                                       ;
;                                                                       ;
;***********************************************;***********************;
                                                ;
SPIMIntDiscardRxBuf:                            ;
        GLOBAL  SPIMIntDiscardRxBuf             ;
                                                ;
        BANKSEL _vSPIMIntRxBufRdPtr             ;
        clrf    _vSPIMIntRxBufRdPtr             ;
        clrf    _vSPIMIntRxBufWrPtr             ;
        bsf     vSPIMIntStatus,SPIMRxBufEmpty   ;
        bcf     vSPIMIntStatus,SPIMRxBufFull    ;
                                                ;
        return                                  ;
                                                ;
;***********************************************;9




;***********************************************;
;       END                                     ;
;***********************************************;

