#ifndef lint
static char *RCSid = "$Header: /m/webster/src/c/client/RCS/webster.c,v 1.3 86/12/26 17:45:20 davy Exp $";
#endif

/*
 * webster - look words up in the dictionary
 *
 * This program connects to the Webster server to get definitions of words.
 * Words may be given on the command line, or, if no arguments are given,
 * the program runs interactively.  If -s or -d is given, it puts the program
 * in "spell" or "define" mode, respectively.  "define" is the default.
 *
 * In either mode, a word may include the wildcard characters '%' and '*'.
 * The '%' character matches exactly one character, while the '*' matches
 * zero or more characters.  If wildcards are used, the program will
 * return either "No match" or a list of matching words.  "!word" looks the
 * word up in the alternate mode.
 *
 * In interactive mode only, Tenex-style command completion may also be
 * used.  Typing a '?' following part of a word will cause the program
 * to print all words which begin with the partial word, or the program
 * will beep if nothing matches.  Typing an ESCape character causes the
 * program to attempt to complete the word.  If the word can be completed,
 * the new word is printed; otherwise, the program beeps.  Wildcards
 * may be used to specify the partial words.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * April, 1986
 *
 * $Log:	webster.c,v $
 * Revision 1.3  86/12/26  17:45:20  davy
 * Updated header comments.
 * 
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <signal.h>
#include <sgtty.h>
#include <netdb.h>
#include <ctype.h>
#include <stdio.h>

#include "../h/webster.h"

#define COMPLETE	033		/* word completion character ^[ */
#define ENDINGS		'?'		/* print matches character      */
#define ALTMODE		'!'		/* use other lookup mode	*/
#define DEFINE		0		/* DEFINE mode			*/
#define SPELL		1		/* SPELL mode			*/

struct sgttyb sgttyb;			/* tty modes when interactive	*/
struct sgttyb rsgttyb;			/* original tty modes		*/

struct tchars tchars;			/* for getting eof char		*/
struct ltchars ltchars;			/* for getting word/line erase	*/

char endoffile;
char backspace;
char worderase;
char lineerase;
char linertype;

int mode = DEFINE;
FILE *WebsterSock;			/* for reading from the server	*/
int interactive = 0;			/* 1 when running interactive	*/

main(argc, argv)
int argc;
char **argv;
{
	int didone = 0;

	/*
	 * Connect to the server.
	 */
	connectup();

	/*
	 * If we were given command line arguments, just
	 * try to define each word.
	 */
	if (argc > 1) {
		while (--argc) {
			if (**++argv == '-') {
				switch (*++*argv) {
				case 'd':
					mode = DEFINE;
					continue;
				case 's':
					mode = SPELL;
					continue;
				default:
					fprintf(stderr, "Usage: webster [-d] [-s] [words...]\n");
					exit(1);
				}
			}

			/*
			 * Look up the word.
			 */
			if (mode == DEFINE)
				define(*argv);
			else
				spell(*argv);

			didone++;
		}

		if (didone)
			exit(0);
	}

	/*
	 * If no arguments were given, set up the
	 * terminal modes and run interactively.
	 */
	setup();
	interact();
}

/*
 * connectup - connects to the Webster server.
 */
connectup()
{
	register int s;
	struct sockaddr_in sin;
	register struct hostent *hp;
	register struct servent *sp;
	struct hostent *gethostbyname();
	struct servent *getservbyname();

	/*
	 * Look up the host in the host file.
	 */
	if ((hp = gethostbyname(WEBSTERHOST)) == NULL) {
		fprintf(stderr, "webster: %s: unknown host.\n", WEBSTERHOST);
		exit(1);
	}

	bzero(&sin, sizeof(struct sockaddr_in));

	/*
	 * Build the server's address.
	 */
	sin.sin_family = AF_INET;
	bcopy(hp->h_addr, &sin.sin_addr, hp->h_length);

	if ((sp = getservbyname(WEBSTERNAME, "tcp")) == NULL)
		sin.sin_port = htons(WEBSTERPORT);
	else
		sin.sin_port = sp->s_port;

	/*
	 * Get a TCP socket.
	 */
	if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("webster: socket");
		exit(1);
	}

	/*
	 * Try to connect.
	 */
	if (connect(s, &sin, sizeof(struct sockaddr_in)) < 0) {
		perror("webster: connect");
		exit(1);
	}

	/*
	 * Open the socket for stdio.
	 */
	WebsterSock = fdopen(s, "r");
}

/*
 * setup - turns on CBREAK, turns off ECHO.  This is where we find out the
 *	   special terminal characters (erase, etc.).  Also trap signals.
 */
setup()
{
	extern int byebye();
	extern int suspend();

	interactive = 1;
	ioctl(0, TIOCGETP, &sgttyb);
	ioctl(0, TIOCGETC, &tchars);
	ioctl(0, TIOCGLTC, &ltchars);

	rsgttyb = sgttyb;
	endoffile = tchars.t_eofc;
	lineerase = sgttyb.sg_kill;
	backspace = sgttyb.sg_erase;
	worderase = ltchars.t_werasc;
	linertype = ltchars.t_rprntc;

	signal(SIGINT, byebye);
	signal(SIGQUIT, byebye);
	signal(SIGTSTP, suspend);

	sgttyb.sg_flags |= CBREAK;
	sgttyb.sg_flags &= ~ECHO;
	ioctl(0, TIOCSETP, &sgttyb);
}

/*
 * interact - interact with the user.
 */
interact()
{
	int c;
	char buf[1024];
	register char *s, *t;

	/*
	 * Forever...
	 */
	for (;;) {
		/*
		 * Prompt for a word.
		 */
		s = buf;
		write(1, "Word: ", 6);

		/*
		 * Forever... read characters.  We
		 * break out of this from inside.
		 */
		for (;;) {
			if (read(0, &c, 1) <= 0)
				byebye();

			if (c == endoffile) {
				/*
				 * If at the start of a line,
				 * treat it as end of file.
				 * Otherwise, ignore it.
				 */
				if (s == buf)
					byebye();
				continue;
			}
			else if (c == backspace) {
				/*
				 * If not at the beginning of a line,
				 * back up one character.
				 */
				if (s > buf) {
					write(1, "\b \b", 3);
					s--;
				}
				continue;
			}
			else if (c == worderase) {
				/*
				 * Until we hit beginning of line
				 * or beginning of word, back up.
				 */
				while ((s > buf) && (*s != ' ')) {
					write(1, "\b \b", 3);
					s--;
				}
				continue;
			}
			else if (c == lineerase) {
				/*
				 * Until we hit beginning of line,
				 * back up.
				 */
				while (s > buf) {
					write(1, "\b \b", 3);
					s--;
				}
				continue;
			}
			else if (c == linertype) {
				/*
				 * Retype the line.
				 */
				write(1, "\r\nWord: ", 8);

				for (t=buf; t < s; t++)
					write(1, t, 1);
				continue;
			}
			else if (c == COMPLETE) {
				/*
				 * Try to complete what they typed
				 * so far.  Put the pointer at the
				 * end of the new word.
				 */
				*s = NULL;
				complete(buf);
				for (s=buf; *s; s++)
					;
				continue;
			}
			else if (c == ENDINGS) {
				/*
				 * If it's the first character,
				 * then print some help.  Otherwise,
				 * try to find endings for the word.
				 * endings() returns 1 if no endings
				 * were found, 0 if some were found.
				 * This tells us whether to reprint
				 * the current word or not.
				 */
				if (s == buf) {
					help();
				}
				else {
					*s = NULL;
					if (endings(buf) == 0) {
						write(1, "Word: ", 6);

						for (s=buf; *s; s++)
							write(1, s, 1);
					}

					continue;
				}
			}
			else if (c == '\n') {
				/*
				 * If at the start of a word,
				 * newline is exit.
				 */
				if (s == buf)
					byebye();

				/*
				 * Otherwise, try to define
				 * the word.
				 */
				*s = NULL;
				write(1, "\n", 1);

				if (*buf == ALTMODE) {
					if (strlen(buf) == 1)
						break;
					if (mode == DEFINE)
						spell(buf+1);
					else
						define(buf+1);
				}
				else {
					if (mode == DEFINE)
						define(buf);
					else
						spell(buf);
				}
			}
			else {
				/*
				 * Echo the character and copy it.
				 */
				write(1, &c, 1);
				*s++ = c;
				continue;
			}

			break;
		}
	}
}

/*
 * define - try to define a word and print its definition.
 */
define(word)
char *word;
{
	int c, refs;
	char buf[1024];
	register char *s;

	/*
	 * Command is "DEFINE<space>word<nl>".
	 */
	sprintf(buf, "DEFINE %s\r\n", word);

	/*
	 * Send the command.
	 */
	if (send(fileno(WebsterSock), buf, strlen(buf), 0) < 0) {
		perror("webster: send");
		byebye();
	}

	/*
	 * Read the first line back from the server.  This
	 * line tells us what the result of our DEFINE
	 * request was.
	 */
	getline(buf);

	/*
	 * "WILD<space>0<nl>" means they used wild cards and no
	 * matches were found.
	 */
	if (!strncmp(buf, "WILD 0", 6)) {
		printf("No match.\n");
		return;
	}

	/*
	 * "WILD<nl>" means that the wildcard matched, so we
	 * print a list of possible matches.
	 */
	if (!strncmp(buf, "WILD", 4)) {
		printf("Possible matches are:\n");

		/*
		 * List lines.
		 */
		listlines(0, 1);
		putchar('\n');
		return;
	}

	/*
	 * "SPELLING<space>0<nl>" means the word is not defined,
	 * and there are no alternate spellings.
	 */
	if (!strncmp(buf, "SPELLING 0", 10)) {
		printf("No definition for '%s'.\n", word);
		return;
	}

	/*
	 * "SPELLING<nl>" means the word is not defined, but
	 * some alternate spellings were found.  Print
	 * them out.
	 */
	if (!strncmp(buf, "SPELLING", 8)) {
		printf("No definition for '%s'.  Maybe you mean:\n", word);

		/*
		 * List lines.
		 */
		listlines(0, 1);
		putchar('\n');
		return;
	}

	/*
	 * "DEFINITION<space>n<nl>" means the word is defined,
	 * and there are n cross-references.
	 */
	if (!strncmp(buf, "DEFINITION", 10)) {
		sscanf(buf+11, "%d", &refs);

		/*
		 * Print any cross references.
		 */
		if (refs > 0) {
			printf("Cross references:\n");

			/*
			 * List lines.
			 */
			listlines(refs, 1);
			putchar('\n');
		}

		/*
		 * Print the definition.
		 */
		while ((c = getc(WebsterSock)) != EOF) {
			if (c == EOFCH)
				break;

			c &= 0177;
			putchar(c);
		}

		putchar('\n');
		return;
	}

	/*
	 * An error message.
	 */
	if (!strncmp(buf, "ERROR ", 6)) {
		if (!strncmp(buf+6, "FATAL", 5)) {
			printf("%s\n", buf+11);
			byebye();
		}
		else {
			printf("%s\n", buf+17);
			return;
		}
	}

	/*
	 * Should never get here.
	 */
	while (((c = getc(WebsterSock)) != EOF) && (c != EOFCH))
		;
}

/*
 * spell - look up a word and see if it's spelled correctly.
 */
spell(word)
char *word;
{
	int c, refs;
	char buf[1024];
	register char *s;

	/*
	 * Command is "SPELL<space>word<nl>".
	 */
	sprintf(buf, "SPELL %s\r\n", word);

	/*
	 * Send the command.
	 */
	if (send(fileno(WebsterSock), buf, strlen(buf), 0) < 0) {
		perror("webster: send");
		byebye();
	}

	/*
	 * Read the first line back from the server.  This
	 * line tells us what the result of our SPELL
	 * request was.
	 */
	getline(buf);

	/*
	 * "SPELLING<space>0<nl>" means the word is not spelled correctly,
	 * and there are no alternate spellings.
	 */
	if (!strncmp(buf, "SPELLING 0", 10)) {
		printf("'%s' is not a correct spelling.\n", word);
		return;
	}

	/*
	 * "SPELLING<space>1<nl>" means the word is spelled correctly.
	 */
	if (!strncmp(buf, "SPELLING 1", 10)) {
		printf("'%s' is spelled correctly.\n", word);
		return;
	}

	/*
	 * "SPELLING<nl>" means the word is not spelled correctly, but
	 * some alternate spellings were found.  Print them out.
	 */
	if (!strncmp(buf, "SPELLING", 8)) {
		printf("No spelling for '%s'.  Maybe you mean:\n", word);

		/*
		 * List lines.
		 */
		listlines(0, 1);
		putchar('\n');
		return;
	}

	/*
	 * An error message.
	 */
	if (!strncmp(buf, "ERROR ", 6)) {
		if (!strncmp(buf+6, "FATAL", 5)) {
			printf("%s\n", buf+11);
			byebye();
		}
		else {
			printf("%s\n", buf+17);
			return;
		}
	}

	/*
	 * Should never get here.
	 */
	while (((c = getc(WebsterSock)) != EOF) && (c != EOFCH))
		;
}

/*
 * complete - try to complete the word.
 */
complete(word)
char *word;
{
	int c;
	char buf[1024];
	register char *s;

	/*
	 * Command is "COMPLETE<space>word<nl>".
	 */
	sprintf(buf, "COMPLETE %s\r\n", word);

	/*
	 * Send the command.
	 */
	if (send(fileno(WebsterSock), buf, strlen(buf), 0) < 0) {
		perror("webster: send");
		byebye();
	}

	/*
	 * Get the first line from the server, which tells
	 * us the reult of our request.
	 */
	getline(buf);

	/*
	 * "AMBIGUOUS<space>n<nl>" means the word is ambiguous,
	 * with n possible matches.  We ignore the n, and just
	 * beep.
	 */
	if (!strncmp(buf, "AMBIGUOUS", 9)) {
		write(1, "\007", 1);
		return;
	}

	/*
	 * "COMPLETION<space>full-word<nl>" means the
	 * word was completed.  Erase what they typed
	 * and print the new word over it.  This takes
	 * care of things if they used wildcards.
	 */
	if (!strncmp(buf, "COMPLETION", 10)) {
		for (s=word; *s; s++)
			write(1, "\b", 1);

		s = buf+11;
		while (((*s & 0177) != '\r') && ((*s & 0177) != '\n') &&
		       ((*s & 0177) != NULL)) {
			write(1, s, 1);
			s++;
		}

		/*
		 * Put the new word back into word.  This
		 * gets rid of the wildcards here.
		 */
		*s = NULL;
		strcpy(word, buf+11);

		return;
	}

	/*
	 * An error message.
	 */
	if (!strncmp(buf, "ERROR ", 6)) {
		if (!strncmp(buf+6, "FATAL", 5)) {
			printf("%s\n", buf+11);
			byebye();
		}
		else {
			printf("%s\n", buf+17);
			return;
		}
	}

	/*
	 * Should never get here.
	 */
	while (((c = getc(WebsterSock)) != EOF) && (c != EOFCH))
		;
}

/*
 * endings - find possible endings for a word.
 */
endings(word)
char *word;
{
	int c;
	char buf[1024];
	register char *s;

	/*
	 * Command is "ENDINGS<space>word<nl>".
	 */
	sprintf(buf, "ENDINGS %s\r\n", word);

	/*
	 * Send the command.
	 */
	if (send(fileno(WebsterSock), buf, strlen(buf), 0) < 0) {
		perror("webster: send");
		byebye();
	}

	/*
	 * Get the first line from the server, which tells
	 * us the result of the search.
	 */
	getline(buf);

	/*
	 * "MATCHS<space>0<nl>" means nothing matched,
	 * so we beep at them.
	 */
	if (!strncmp(buf, "MATCHS 0", 8)) {
		write(1, "\007", 1);
		return(1);
	}

	/*
	 * "MATCHS<nl>" means there were matches, so
	 * print them out.
	 */
	if (!strncmp(buf, "MATCHS", 6)) {
		printf("\nMaybe you mean:\n");

		/*
		 * List lines.
		 */
		listlines(0, 0);
		putchar('\n');
		return(0);
	}

	/*
	 * An error message.
	 */
	if (!strncmp(buf, "ERROR ", 6)) {
		if (!strncmp(buf+6, "FATAL", 5)) {
			printf("%s\n", buf+11);
			byebye();
		}
		else {
			printf("%s\n", buf+17);
			return;
		}
	}

	/*
	 * Should never get here.
	 */
	while (((c = getc(WebsterSock)) != EOF) && (c != EOFCH))
		;

	return(0);
}

/*
 * getline - read one line from the server and put it in s.
 */
getline(s)
register char *s;
{
	register int c;

	/*
	 * Read in chars.  If we hit EOFCH, return
	 * 0.
	 */
	while ((c = getc(WebsterSock)) != EOF) {
		if (c == EOFCH)
			return(0);

		c &= 0177;

		if (c == '\r')
			continue;

		if (c == '\n')
			break;

		*s++ = c;
	}

	*s = NULL;
	return(1);
}

/*
 * listlines - list WILD-style lines on the screen.
 */
listlines(n, num)
register int n;
int num;
{
	char buf[1024];
	register int col;

	printf(" ");

	/*
	 * If n is non-zero, we only want to list n lines.
	 * Otherwise, we go till we hit EOFCH.  Lines are
	 * printed in four columns.
	 */
	if (n) {
		col = 0;
		while (n-- > 0) {
			getline(buf);
			putline(buf, num);

			if (++col == 3) {
				printf("\n ");
				col = 0;
			}
		}
	}
	else {
		col = 0;
		while (getline(buf) > 0) {
			putline(buf, num);

			if (++col == 3) {
				printf("\n ");
				col = 0;
			}
		}
	}

	if (col)
		putchar('\n');
}

/*
 * putline - put out a line, if num is 0, skip the line number.
 */
putline(buf, num)
char *buf;
int num;
{
	int lnum;
	char line[1024];

	sscanf(buf, "%d %[^\n]", &lnum, line);

	if (num)
		printf("%2d. %-22s", lnum, line);
	else
		printf("%-26s", line);
}

/*
 * help - print a help message.
 */
help()
{
	printf("\n   Type in the word you want defined, or a blank line to exit. Additionally,\n");
	printf("Webster can match words using wildcards.  The character '%%' in a word means\n");
	printf("match exactly one character; while the character '*' means match zero or more\n");
	printf("characters.  Typing \"!word\" will check the spelling of the word instead\n");
	printf("of checking its definition.\n");
	printf("   Typing a partial word followed by '?' will print all the words in the\n");
	printf("dictionary which match your partial word. Typing a partial word followed by an\n");
	printf("ESCape character will try to complete the word for you.  If the partial word\n");
	printf("is ambiguous, Webster will beep at you.  Note that you can use the wildcards\n");
	printf("along with ESC and ?.  For example (the underlined parts are typed by the\n");
	printf("user, the rest by Webster),\n");
	printf("\n");
	printf("Word: balla?   Maybe you mean:\n");
	printf("      ------\n");
	printf("  1. ballad           2. ballade          3. baladry         4. ballast\n");
	printf("Word: pluria<ESC>xial\n");
	printf("      -----------\n");
	printf("Word: plu*x<ESC>\n");
	printf("      ----------\n");
	printf("Word: pluriaxial\n");
	printf("\n---- End of Help File ----\n\n");
}

/*
 * byebye - called on exit.
 */
byebye()
{
	/*
	 * If interactive, reset the tty modes.
	 */
	if (interactive)
		ioctl(0, TIOCSETP, &rsgttyb);

	/*
	 * Close the socket and exit.
	 */
	fclose(WebsterSock);
	write(1, "\n", 1);
	exit(0);
}

/*
 * suspend - reset tty modes and supend ourselves.
 */
suspend()
{
	long blocked;
	extern int suspend();

	/*
	 * Reset tty modes and suspend.
	 */
	ioctl(0, TIOCSETP, &rsgttyb);
	signal(SIGTSTP, SIG_DFL);
	blocked = sigsetmask(0);
	kill(0, SIGTSTP);

	/*
	 * We come here on SIGCONT.  Reset
	 * the signal mask and tty modes.
	 */
	sigsetmask(blocked);
	signal(SIGTSTP, suspend);
	ioctl(0, TIOCSETP, &sgttyb);	
}
