/* ==== sleep.c ============================================================
 * Copyright (c) 1993, 1994 by Chris Provenzano, proven@mit.edu
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *  This product includes software developed by Chris Provenzano.
 * 4. The name of Chris Provenzano may not be used to endorse or promote 
 *	  products derived from this software without specific prior written
 *	  permission.
 *
 * THIS SOFTWARE IS PROVIDED BY CHRIS PROVENZANO ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL CHRIS PROVENZANO BE LIABLE FOR ANY 
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * Description : All the appropriate sleep routines.
 *
 *  1.00 93/12/28 proven
 *      -Started coding this file.
 *
 *	1.36 94/06/04 proven
 *		-Use new timer structure pthread_timer, that uses seconds
 *		-nano seconds. Rewrite all routines completely.
 *
 *	1.38 94/06/13 proven
 *		-switch pthread_timer to timespec
 */

#ifndef lint
static const char rcsid[] = "$Id: sleep.c,v 1.50.2.5 94/10/25 05:21:26 proven Exp $";
#endif

#include <sys/time.h>
#include <pthread.h>
#include <signal.h>
#include <unistd.h>

semaphore sleep_semaphore = SEMAPHORE_CLEAR;
struct pthread * pthread_sleep = NULL;

/* ==========================================================================
 * sleep_compare_time()
 */
/* static inline int sleep_compare_time(struct timespec * time1, 
  struct timespec * time2) */
static int sleep_compare_time(struct timespec * time1, 
  struct timespec * time2)
{
	if ((time1->ts_sec < time2->ts_sec) || 
	  ((time1->ts_sec == time2->ts_sec) && (time1->ts_nsec < time2->ts_nsec))) {
		return(-1);
	}
	if ((time1->ts_sec == time2->ts_sec) && (time1->ts_nsec == time2->ts_nsec)) {
		return(0);
	}
	return(1);
}

/* ==========================================================================
 * machdep_stop_timer()
 *
 * Returns the time left on the timer.
 */
static struct itimerval timestop = { { 0, 0 }, { 0, 0 } };

void machdep_stop_timer(struct timespec *current)
{
	struct itimerval timenow;

	setitimer(ITIMER_REAL, & timestop, & timenow);
	if (current) {
		current->ts_nsec = timenow.it_value.tv_usec * 1000;
		current->ts_sec = timenow.it_value.tv_sec;
	}
}

/* ==========================================================================
 * machdep_start_timer()
 */
int machdep_start_timer(struct timespec *current, struct timespec *wakeup)
{
	struct itimerval timeout;

	timeout.it_value.tv_usec = (wakeup->ts_nsec - current->ts_nsec) / 1000;
	timeout.it_value.tv_sec = wakeup->ts_sec - current->ts_sec;
	timeout.it_interval.tv_usec = 0;
	timeout.it_interval.tv_sec = 0;
	if (timeout.it_value.tv_usec < 0) {
		timeout.it_value.tv_usec += 1000000;
		timeout.it_value.tv_sec--;
	}

	if ((!(timeout.it_value.tv_sec < 0)) &&
	  ((timeout.it_value.tv_usec) || (timeout.it_value.tv_sec))) {
		setitimer(ITIMER_REAL, & timeout, NULL);
	} else {
		/*
		 * There is no time on the timer.
		 * This shouldn't happen,
		 * but isn't fatal.
		 */
		sig_handler_fake(SIGALRM);
	}
	return(OK);
}

/* ==========================================================================
 * sleep_schedule()
 *
 * Assumes that the current thread is the thread to be scheduled
 * and that the thread is already locked and that the sleep_lock is
 * not locked.
 */
void sleep_schedule(struct timespec *current_time, struct timespec *new_time)
{
	struct pthread * pthread_sleep_current, * pthread_sleep_prev;
	semaphore * lock;

	/* Lock sleep queue */
	lock = &(sleep_semaphore);
	while (SEMAPHORE_TEST_AND_SET(lock)) {
		pthread_yield();
	}

	/* any threads? */
	if (pthread_sleep_current = pthread_sleep) {
		if (sleep_compare_time(&(pthread_sleep_current->wakeup_time),
		  new_time) <= 0) {
			/* Don't need to restart timer */
			while (pthread_sleep_current->sll) {

				pthread_sleep_prev =  pthread_sleep_current;
				pthread_sleep_current = pthread_sleep_current->sll;
				
				if (sleep_compare_time(&(pthread_sleep_current->wakeup_time),
				  new_time) > 0) {
					pthread_run->sll = pthread_sleep_current;
					pthread_sleep_prev->sll = pthread_run;

					/* Unlock sleep mutex */
					SEMAPHORE_RESET(lock);
					return;
				}
			} 

			/* No more threads in queue, attach pthread_run to end of list */
			pthread_sleep_current->sll = pthread_run;
			pthread_run->sll = NULL;

		} else {
			/* Start timer and enqueue thread */
			machdep_start_timer(current_time, new_time);
			pthread_run->sll = pthread_sleep_current;
			pthread_sleep = pthread_run;
		}
	} else {
		/* Start timer and enqueue thread */
		machdep_start_timer(current_time, new_time);
		pthread_sleep = pthread_run;
		pthread_run->sll = NULL;
	}

	/* Unlock sleep mutex */
	SEMAPHORE_RESET(lock);
	return;
}

/* ==========================================================================
 * sleep_basic_wakeup()
 *
 * The real work of sleep_wakeup is done here.
 * Assumes pthread_sleep in locked
 */
/* static inline int sleep_basic_wakeup() */
static int sleep_basic_wakeup()
{
	struct pthread *pthread_sleep_next;
	struct timespec current_time;
	semaphore *plock, *qlock;

	plock = &(pthread_sleep->lock);
	if (SEMAPHORE_TEST_AND_SET(plock)) {
		return(NOTOK);
	}
	current_time.ts_sec = pthread_sleep->wakeup_time.ts_sec;
	current_time.ts_nsec = pthread_sleep->wakeup_time.ts_nsec;

	/* 
 	 * This loop does extra work to ensure that if the next thread
	 * on the queue is busy that the previous thread isn't removed.
	 * This way the routine just returns an error indicating that
	 * the timeout hasn't been handled and the scheduler will try
	 * again soon.
	 */
	do {

		if (pthread_sleep->sll) {
			/* Lock the next thread */
			qlock = &(pthread_sleep->sll->lock);
			if (SEMAPHORE_TEST_AND_SET(qlock)) {
				return(NOTOK);
			}

			/* Clean up removed thread and start it running again. */
			pthread_sleep_next = pthread_sleep->sll;
			pthread_sleep->state = PS_RUNNING;
			pthread_sleep->sll = NULL;

			pthread_sleep = pthread_sleep_next;
			SEMAPHORE_RESET(plock);
			plock = qlock;

		} else {

			/* No more threads on sleep queue */
			pthread_sleep->state = PS_RUNNING;
			SEMAPHORE_RESET(plock);
			pthread_sleep = NULL;
			return(OK);
		}

	} while (sleep_compare_time(&(pthread_sleep->wakeup_time), &(current_time)) <= 0);
		
	/* Start timer for next time interval */
	machdep_start_timer(&current_time, &(pthread_sleep->wakeup_time));
	SEMAPHORE_RESET(plock);
	return(OK);
}

/* ==========================================================================
 * sleep_wakeup()
 *
 * This routine is called by the interrupt handler. It cannot call
 * pthread_yield() therefore it returns NOTOK to inform the handler
 * that it will have to be called at a later time.
 */
int sleep_wakeup()
{
	semaphore *lock, *plock;
	int ret;

	/* Lock sleep queue */
	lock = &(sleep_semaphore);
	if (SEMAPHORE_TEST_AND_SET(lock)) {
		return(NOTOK);
	}

	if (pthread_sleep) {
		ret = sleep_basic_wakeup();
	} else {
		ret = 1;
	}

	SEMAPHORE_RESET(lock);
	return(ret);
}

/* ==========================================================================
 * __sleep()
 */
void __sleep(struct timespec * time_to_sleep)
{
	struct pthread *pthread_sleep_current, *pthread_sleep_prev;
	struct timespec current_time, new_time;
	semaphore *lock;

	/* Lock current thread */
	lock = &(pthread_run->lock);
	while (SEMAPHORE_TEST_AND_SET(lock)) {
		pthread_yield();
	}

	/* Get real time */
	machdep_gettimeofday(&current_time);
	pthread_run->wakeup_time.ts_sec = current_time.ts_sec + time_to_sleep->ts_sec;
	pthread_run->wakeup_time.ts_nsec = current_time.ts_nsec + time_to_sleep->ts_nsec;

	sleep_schedule(&current_time, &(pthread_run->wakeup_time));

	SEMAPHORE_RESET(lock);

	/* Reschedule thread */
	reschedule(PS_SLEEP_WAIT);

	/* Return actual time slept */
	time_to_sleep->ts_sec = pthread_run->wakeup_time.ts_sec;
	time_to_sleep->ts_nsec = pthread_run->wakeup_time.ts_nsec;
}

/* ==========================================================================
 * nanosleep()
 */
unsigned int nanosleep(unsigned int nseconds)
{
	struct timespec time_to_sleep;

	if (nseconds) {
		time_to_sleep.ts_nsec = nseconds;
		time_to_sleep.ts_sec = 0;
		__sleep(&time_to_sleep);
		nseconds = time_to_sleep.ts_nsec;
	}
	return(nseconds);
}

/* ==========================================================================
 * usleep()
 */
void usleep(unsigned int useconds)
{
	struct timespec time_to_sleep;

	if (useconds) {
		time_to_sleep.ts_nsec = (useconds % 1000000) * 1000;
		time_to_sleep.ts_sec = useconds / 1000000;
		__sleep(&time_to_sleep);
		useconds = time_to_sleep.ts_sec * 1000000 + time_to_sleep.ts_nsec * 1000;
	}
	/* return(useconds); */
}

/* ==========================================================================
 * sleep()
 */
unsigned int sleep(unsigned int seconds)
{
	struct timespec time_to_sleep;

	if (seconds) {
		time_to_sleep.ts_sec = seconds;
		time_to_sleep.ts_nsec = 0;
		__sleep(&time_to_sleep);
		seconds = time_to_sleep.ts_sec;
	}
	return(seconds);
}

/* ==========================================================================
 * sleep_cancel()
 *
 * Cannot be called while kernel is locked.
 */
int sleep_cancel(struct pthread * pthread)
{
	struct timespec current_time, delta_time;
	struct pthread * pthread_last;
	semaphore *lock, *plock, *qlock;
	int rval = NOTOK;

	/* Lock sleep queue */
	lock = &(sleep_semaphore);
	while (SEMAPHORE_TEST_AND_SET(lock)) {
		pthread_yield();
	}
	/* Lock thread to be removed */
	plock = &(pthread->lock);
	while (SEMAPHORE_TEST_AND_SET(plock)) {
		pthread_yield();
	}

	if (pthread_sleep) {
		if (pthread == pthread_sleep) {
			rval = OK;
			machdep_stop_timer(&delta_time);
			if (pthread_sleep = pthread_sleep->sll) {
				current_time.ts_sec 	= delta_time.ts_sec;
				current_time.ts_nsec 	= delta_time.ts_nsec;
				current_time.ts_sec 	+= pthread_sleep->wakeup_time.ts_sec;
				current_time.ts_nsec 	+= pthread_sleep->wakeup_time.ts_nsec;
				if (current_time.ts_nsec > 1000000000) {
					current_time.ts_nsec -= 1000000000;
					current_time.ts_sec++;
				}
				machdep_start_timer(&(current_time), 
				  &(pthread_sleep->wakeup_time));
			}
		} else {
			/*
			 * Locking is necessary to ensure that no other thread is modifing
			 * the threads internal structures. It is unnecessary to maintain
			 * a lock while we get another because the queue cannot be modified
			 * without the lock on sleep_semaphore.
			 */
			pthread_last = pthread_sleep;
			qlock = &(pthread_last->lock);
			while (SEMAPHORE_TEST_AND_SET(qlock)) {
				pthread_yield();
			}
			while (pthread_last = pthread_last->sll) {
				SEMAPHORE_RESET(qlock);
				qlock = &(pthread_last->lock);
				while (SEMAPHORE_TEST_AND_SET(qlock)) {
					pthread_yield();
				}
				if (pthread_last->sll == pthread) {
					pthread_last->sll = pthread->sll;
					rval = OK;
					break;
				}
			}
			SEMAPHORE_RESET(qlock);
		}
	}
	pthread->state = PS_RUNNING;
	pthread->sll = NULL;

	SEMAPHORE_RESET(plock);
	SEMAPHORE_RESET(lock);
	return(rval);
}
