/*
 * Hardware definitions for HP iPAQ Handheld Computers
 *
 * Copyright 2000-2003 Hewlett-Packard Company.
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * History:
 * 2004-??-??  Shawn Anderson    Derived the from aximx3.c aximx5.c e7xx.c 
 *                               h1900.c h2200.c h3900.c h5400.c and friends.
 * 2004-04-01  Eddi De Pieri     Move lcd stuff so it can be used as a module.
 * 2004-04-19  Eddi De Pieri     Moving to new 2.6 standard 
 *                               (platform_device / device_driver structure)
 * 2007-03-24  Paul Sokolovsky   Major cleanup.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/gpiodev.h>
#include <linux/dpm.h>
#include <asm/irq.h>
#include <asm/mach-types.h>
#include <asm/hardware.h>
#include <asm/setup.h>
#include <asm/arch/bitfield.h>
#include <asm/arch/pxa-regs.h>
#include <asm/mach/irq.h>
#include <asm/mach/arch.h>
#include "../generic.h"
// Specific devices
#include <linux/ads7846.h>
#include <linux/touchscreen-adc.h>
#include <linux/adc_battery.h>
#include <linux/pda_power.h>
#include <linux/rs232_serial.h>
#include <asm/arch/pxa2xx_udc_gpio.h>
#include <asm/arch/pxafb.h>
#include <linux/soc/asic3_base.h>
#include <asm/arch/h4000-gpio.h>
#include <asm/arch/h4000-init.h>
#include <asm/arch/h4000-asic.h>


void h4000_ll_pm_init(void);

/* Misc devices */
extern struct platform_device h4000_bl;
static struct platform_device h4000_lcd       = { .name = "h4000-lcd", };
static struct platform_device h4300_kbd       = { .name = "h4300-kbd", };
static struct platform_device h4000_buttons   = { .name = "h4000-buttons", };
static struct platform_device h4000_pcmcia    = { .name = "h4000-pcmcia", };
static struct platform_device h4000_batt      = { .name = "h4000-batt", };
static struct platform_device h4000_bt        = { .name = "h4000-bt", };
static struct platform_device h4000_irda      = { .name = "h4000-irda", };

/* Power */
static int h4000_is_ac_online(void)
{
	return !GET_H4000_GPIO(AC_IN_N);
}

static struct pda_power_pdata h4000_power_pdata = {
	.is_ac_online = h4000_is_ac_online,
};

static struct resource h4000_power_resourses[] = {
	[0] = {
		.name = "ac",
		.flags = IORESOURCE_IRQ | IORESOURCE_IRQ_HIGHEDGE |
		         IORESOURCE_IRQ_LOWEDGE,
		.start = H4000_IRQ(AC_IN_N),
		.end = H4000_IRQ(AC_IN_N),
	},
};

static struct platform_device h4000_power = {
	.name = "pda-power",
	.id = -1,
	.resource = h4000_power_resourses,
	.num_resources = ARRAY_SIZE(h4000_power_resourses),
	.dev = {
		.platform_data = &h4000_power_pdata,
	},
};

/* Batteries */
static int h4000_is_charging(void)
{
	return GET_H4000_GPIO(CHARGING);
}

static struct battery_adc_platform_data h4000_main_batt_params = {
	.battery_info = {
		.name = "main-battery",
		.max_voltage = 4200,
		.min_voltage = 0,
		.max_current = 500,
		.min_current = 0,
		.max_capacity = 1000,
		.main_battery = 1,
	},
	.voltage_pin = "ads7846-ssp:c_vbat",
	.current_pin = "ads7846-ssp:c_ibat",
	.temperature_pin = "ads7846-ssp:c_tbat",
	/* Coefficient is 1.01471 */
	.voltage_mult = 1015,

	.is_charging = h4000_is_charging,
};

static struct platform_device h4000_main_batt = { 
	.name = "adc-battery",
	.id = 1,
	.dev = {
		.platform_data = &h4000_main_batt_params,
	}
};

static struct battery_adc_platform_data h4000_backup_batt_params = {
	.battery_info = {
		.name = "backup-battery",
		.max_voltage = 4200,
		.min_voltage = 0,
		.max_current = 500,
		.min_current = 0,
		.max_capacity = 1000,
	},
	.voltage_pin = "ads7846-ssp:c_vbck",
};

static struct platform_device h4000_backup_batt = { 
	.name = "adc-battery",
	.id = 2,
	.dev = {
		.platform_data = &h4000_backup_batt_params,
	}
};

/* UDC device */
struct pxa2xx_udc_gpio_info h4000_udc_info = {
	.detect_gpio = {&pxagpio_device.dev, GPIO_NR_H4000_USB_DETECT_N},
	.detect_gpio_negative = 1,
	.power_ctrl = {
		.power_gpio = {&h4000_asic3.dev, ASIC3_GPIOD_IRQ_BASE + GPIOD_USB_PULLUP},
	},

};

static struct platform_device h4000_udc = { 
	.name = "pxa2xx-udc-gpio",
	.dev = {
		.platform_data = &h4000_udc_info
	}
};

/* ADC device */
static int h4000_ads7846_mux(int custom_pin)
{
        int mux_b = 0, mux_d = 0;

        custom_pin -= AD7846_PIN_CUSTOM_START;
        if (custom_pin & 1)  mux_d |= GPIOD_MUX_SEL0;
        if (custom_pin & 2)  mux_d |= GPIOD_MUX_SEL1;
        if (custom_pin & 4)  mux_b |= GPIOB_MUX_SEL2;
        /* Select sampling source */
	asic3_set_gpio_out_b(&h4000_asic3.dev, GPIOB_MUX_SEL2, mux_b);
	asic3_set_gpio_out_d(&h4000_asic3.dev, GPIOD_MUX_SEL1|GPIOD_MUX_SEL0, mux_d);
	/* Likely not yet fully settled, by seems stable */
	udelay(50);
	return AD7846_PIN_VAUX;
}

/* FIXME: taken from h4000_battery.c, TODO: create dedicated header */
#define AD7846_PIN_CUSTOM_VBAT AD7846_PIN_CUSTOM_START
#define AD7846_PIN_CUSTOM_IBAT AD7846_PIN_CUSTOM_START + 1
#define AD7846_PIN_CUSTOM_VBACKUP AD7846_PIN_CUSTOM_START + 2
#define AD7846_PIN_CUSTOM_TBAT AD7846_PIN_CUSTOM_START + 3
#define AD7846_PIN_CUSTOM_UNK AD7846_PIN_CUSTOM_START + 7

static struct adc_classdev h4000_custom_pins[] = {
	{
		.name = "c_vbck",
		.pin_id = AD7846_PIN_CUSTOM_VBACKUP,
	},
	{
		.name = "c_vbat",
		.pin_id = AD7846_PIN_CUSTOM_VBAT,
	},
	{
		.name = "c_ibat",
		.pin_id = AD7846_PIN_CUSTOM_IBAT,
	},
	{
		.name = "c_tbat",
		.pin_id = AD7846_PIN_CUSTOM_TBAT,
	},
};
static struct ads7846_ssp_platform_data h4000_ts_ssp_params = {
	.port = 1,
	.pd_bits = 1,
	.freq = 100000,
	.mux_custom = h4000_ads7846_mux,
	.custom_adcs = h4000_custom_pins,
	.num_custom_adcs = ARRAY_SIZE(h4000_custom_pins),
};
static struct platform_device ads7846_ssp     = { 
	.name = "ads7846-ssp", 
	.id = -1,
	.dev = {
		.platform_data = &h4000_ts_ssp_params,
	}
};

/* TS device (virtual on top of ADC) */
static struct tsadc_platform_data h4000_ts_params = {
	.pen_irq  = IRQ_GPIO(GPIO_NR_H4000_PEN_IRQ_N),
	.pen_gpio = GPIO_NR_H4000_PEN_IRQ_N,
	.x_pin = "ads7846-ssp:x",
	.y_pin = "ads7846-ssp:y",
	.z1_pin = "ads7846-ssp:z1",
	.z2_pin = "ads7846-ssp:z2",
	.pressure_factor = 100000,
	.min_pressure = 2,
	.max_jitter = 8,
};
static struct platform_device h4000_ts        = { 
	.name = "ts-adc-debounce",
	.id = -1,
	.dev = {
		.platform_data = &h4000_ts_params,
	}
};

/* RS232 device */
static struct rs232_serial_pdata h4000_rs232_data = {
	.detect_gpio = {&pxagpio_device.dev, GPIO_NR_H4000_SERIAL_DETECT},
	.power_ctrl = {	
		.power_gpio = {&h4000_asic3.dev, ASIC3_GPIOA_IRQ_BASE + GPIOA_RS232_ON},
	},
};
static struct platform_device h4000_serial = { 
	.name	= "rs232-serial",
	.dev	= {
		.platform_data = &h4000_rs232_data
	},
};

/* Framebuffer device */
/*
 * LCCR0: 0x003008f9 -- ENB=0x1,  CMS=0x0, SDS=0x0, LDM=0x1,
 *                      SFM=0x1,  IUM=0x1, EFM=0x1, PAS=0x1,
 *                      res=0x0,  DPD=0x0, DIS=0x0, QDM=0x1,
 *                      PDD=0x0,  BM=0x1,  OUM=0x1, res=0x0
 * LCCR1: 0x13070cef -- BLW=0x13, ELW=0x7, HSW=0x3, PPL=0xef
 * LCCR2: 0x0708013f -- BFW=0x7,  EFW=0x8, VSW=0x0, LPP=0x13f
 * LCCR3: 0x04700008 -- res=0x0,  DPC=0x0, BPP=0x4, OEP=0x0, PCP=0x1
 *                      HSP=0x1,  VSP=0x1, API=0x0, ACD=0x0, PCD=0x8
 */

static struct pxafb_mode_info h4000_lcd_modes[] = {
{
        .pixclock     = 171521,     // (160756 > 180849)
        .xres         = 240,        // PPL + 1
        .yres         = 320,        // LPP + 1
        .bpp          = 16,         // BPP (0x4 == 16bits)
        .hsync_len    = 4,          // HSW + 1
        .vsync_len    = 1,          // VSW + 1
        .left_margin  = 20,         // BLW + 1
        .upper_margin = 8,          // BFW + 1
        .right_margin = 8,          // ELW + 1
        .lower_margin = 8,          // EFW + 1
        .sync         = 0,
},
};

static struct pxafb_mach_info sony_acx502bmu = {
        .modes		= h4000_lcd_modes,
        .num_modes	= ARRAY_SIZE(h4000_lcd_modes), 
        
        .lccr0        = LCCR0_Act | LCCR0_Sngl | LCCR0_Color,
        .lccr3        = LCCR3_OutEnH | LCCR3_PixFlEdg | LCCR3_Acb(0),

        //.pxafb_backlight_power = ,
        //.pxafb_lcd_power =       ,
};

/* SoC device */
static struct platform_device *h4000_asic3_devices[] __initdata = {
	&h4000_serial,
	&h4000_lcd,
#ifdef CONFIG_IPAQ_H4000_BACKLIGHT
	&h4000_bl,
#endif
	&h4300_kbd,
	&h4000_buttons,
	&h4000_ts,
	&h4000_udc,
	&h4000_pcmcia,
	&h4000_power,
	&h4000_main_batt,
	&h4000_backup_batt,
	&h4000_batt,
	&h4000_bt,
	&h4000_irda,
};

static struct asic3_platform_data h4000_asic3_platform_data = {
	.gpio_a = {
		.dir            = 0xfc7f,
		/* Turn (keep) serial on, in case we have initial console there.
		   h4000_serial.c will verify if anything is actually connected there,
		   and turn it off otherwise. */
		.init           = 1<<GPIOA_RS232_ON,
		.sleep_out      = 0x0000,
		.batt_fault_out = 0x0000,
		.alt_function   = 0x0000,
		.sleep_conf     = 0x000c,
	},
	.gpio_b = {
		.dir            = 0xddbf, /* 0xdfbd for h4300 kbd irq */
		/* 0x1c00, 0x1c05 for h4300 kbd wakeup/power */
		/* LCD and backlight active */
		.init           = GPIOB_LCD_ON | GPIOB_LCD_PCI | GPIOB_BACKLIGHT_POWER_ON,
		.sleep_out      = 0x0000,
		.batt_fault_out = 0x0000,
		.alt_function   = 0x0000,
		.sleep_conf     = 0x000c,
	},
	.gpio_c = {
		.dir            = 0xffff, /* 0xfff7 for h4300 key rxd spi */
		/* 0x4700 */
		/* Flash and LCD active */
		.init           = GPIOC_FLASH_RESET_N | GPIOC_LCD_3V3_ON | GPIOC_LCD_5V_EN | GPIOC_LCD_N3V_EN,
		/* Keep flash active even during sleep/battery fault */
		.sleep_out      = GPIOC_FLASH_RESET_N,
		.batt_fault_out = GPIOC_FLASH_RESET_N,
		.alt_function   = 0x0003, /* 0x003b for h4300 kbd spi */ 
		.sleep_conf     = 0x000c,
	},
	.gpio_d = {
		.dir            = 0xef03, /* 0xef7b for h4300, no buttons here*/
		/* 0x0f02 */
		/* LCD active, IR inactive, MUX_SEL likely random */
		.init           = GPIOD_PCI | GPIOD_MUX_SEL0 | GPIOD_MUX_SEL1 | GPIOD_IR_ON_N 
				  | GPIOD_LCD_RESET_N 
				  /* Be ready to boot via NFS */
				  | GPIOD_USB_ON | 1<<GPIOD_USB_PULLUP, 
		.sleep_out      = GPIOD_IR_ON_N,
		.batt_fault_out = GPIOD_IR_ON_N,
		.alt_function   = 0x0000,
		.sleep_conf     = 0x000c,
	},
	.child_platform_devs     = h4000_asic3_devices,
	.num_child_platform_devs = ARRAY_SIZE(h4000_asic3_devices),
};

static struct resource h4000_asic3_resources[] = {
        /* GPIO part */
	[0] = {
		.start  = H4000_ASIC3_PHYS,
		.end    = H4000_ASIC3_PHYS + IPAQ_ASIC3_MAP_SIZE - 1,
		.flags  = IORESOURCE_MEM,
	},
	[1] = {
		.start  = IRQ_GPIO(GPIO_NR_H4000_ASIC3_IRQ),
		.flags  = IORESOURCE_IRQ,
	},
        /* SD part */
	[2] = {
		.start  = H4000_ASIC3_SD_PHYS,
		.end    = H4000_ASIC3_SD_PHYS + IPAQ_ASIC3_MAP_SIZE - 1,
		.flags  = IORESOURCE_MEM,
	},
	[3] = {
		.start  = IRQ_GPIO(GPIO_NR_H4000_SD_IRQ_N),
		.flags  = IORESOURCE_IRQ,
	},
};

struct platform_device h4000_asic3 = {
	.name           = "asic3",
	.id             = 0,
	.num_resources  = ARRAY_SIZE(h4000_asic3_resources),
	.resource       = h4000_asic3_resources,
	.dev = { .platform_data = &h4000_asic3_platform_data, },
};
EXPORT_SYMBOL(h4000_asic3);



#if 0
// This function is useful for low-level debugging only
void h4000_set_led(int color, int duty_time, int cycle_time)
{
        if (color == H4000_RED_LED) {
                asic3_set_led(asic3, 0, duty_time, cycle_time);
                asic3_set_led(asic3, 1, 0, cycle_time);
        }

        if (color == H4000_GREEN_LED) {
                asic3_set_led(asic3, 1, duty_time, cycle_time);
                asic3_set_led(asic3, 0, 0, cycle_time);
        }

        if (color == H4000_YELLOW_LED) {
                asic3_set_led(asic3, 1, duty_time, cycle_time);
                asic3_set_led(asic3, 0, duty_time, cycle_time);
        }
}
EXPORT_SYMBOL(h4000_set_led);
#endif

static void __init h4000_map_io(void)
{
	pxa_map_io();

	/* Wake up enable. */
	PWER = PWER_GPIO0 | PWER_RTC;
	/* Wake up on falling edge. */
	PFER = PWER_GPIO0 | PWER_RTC;
	/* Wake up on rising edge. */
	PRER = 0;
	/* 3.6864 MHz oscillator power-down enable */
	PCFR = PCFR_OPDE;
}

static void __init h4000_init(void)
{
#if 0
        // Reset ASIC3
	// So far, this is for debugging only
	// 100ms is *too* much for real use
	// (but visible by naked eye).
	SET_H4000_GPIO(ASIC3_RESET_N, 0);
	mdelay(100);
	SET_H4000_GPIO(ASIC3_RESET_N, 1);
	mdelay(10);
#endif

	/* Configure FFUART, for RS232 and serial console */
	pxa_gpio_mode(GPIO34_FFRXD_MD);
	pxa_gpio_mode(GPIO35_FFCTS_MD);
	pxa_gpio_mode(GPIO36_FFDCD_MD);
	pxa_gpio_mode(GPIO37_FFDSR_MD);
	pxa_gpio_mode(GPIO38_FFRI_MD);
	pxa_gpio_mode(GPIO39_FFTXD_MD);
	pxa_gpio_mode(GPIO40_FFDTR_MD);
	pxa_gpio_mode(GPIO41_FFRTS_MD);

#ifdef CONFIG_PM
	h4000_ll_pm_init();
#endif
	set_pxa_fb_info(&sony_acx502bmu);
	platform_device_register(&ads7846_ssp);
	platform_device_register(&h4000_asic3);
	printk("machine_arch_type=%d\n", machine_arch_type);
}

MACHINE_START(H4000, "HP iPAQ H4000")
	/* Maintainer h4000 port team h4100-port@handhelds.org */
	.phys_io	= 0x40000000,
	.io_pg_offst	= (io_p2v(0x40000000) >> 18) & 0xfffc,
	.boot_params	= 0xa0000100,
	.map_io		= h4000_map_io,
	.init_irq	= pxa_init_irq,
	.init_machine	= h4000_init,
	.timer		= &pxa_timer,
MACHINE_END

MACHINE_START(H4300, "HP iPAQ H4000")
	/* Maintainer h4000 port team h4100-port@handhelds.org */
	.phys_io	= 0x40000000,
	.io_pg_offst	= (io_p2v(0x40000000) >> 18) & 0xfffc,
	.boot_params	= 0xa0000100,
	.map_io		= h4000_map_io,
	.init_irq	= pxa_init_irq,
	.init_machine	= h4000_init,
	.timer		= &pxa_timer,
MACHINE_END
