/*
 * Copyright (c) 2007 Eugeny Boger
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 */

#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/pm.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/battery.h>

#include <asm/irq.h>
#include <asm/apm.h>
#include <linux/list.h>

#define BATTERY_PROPERTY(property) (main_battery->get_##property ? \
			main_battery->get_##property(main_battery) : 0)

static struct battery *main_battery;

static void (*old_apm_get_power_status)(struct apm_power_info*);

static void apm_battery_find_main_battery(void)
{
	struct device *dev;
	struct battery *bat, *batm;
	int max_capacity = 0;

	main_battery = NULL;
	batm = NULL;
	list_for_each_entry(dev, &battery_class->devices, node) {
		bat = dev_get_drvdata(dev);
		/* If none of battery devices cantains 'main_battery' flag,
		   choice one with max capacity */
		if (bat->get_max_capacity)
			if (bat->get_max_capacity(bat) > max_capacity) {
				batm = bat;
				max_capacity = bat->get_max_capacity(bat);
			}

		if (bat->main_battery)
			main_battery = bat;
	}
	if (!main_battery)
		main_battery = batm;
}

static void apm_battery_apm_get_power_status(struct apm_power_info *info)
{
	int bat_current;

	down(&battery_class->sem);
	apm_battery_find_main_battery();
	if (!main_battery) {
		up(&battery_class->sem);
		return;
	}

	if (BATTERY_PROPERTY(status) == BATTERY_STATUS_FULL)
		info->battery_life = 100;
	else
		if (BATTERY_PROPERTY(max_capacity) -
		                                BATTERY_PROPERTY(min_capacity))
			info->battery_life = ((BATTERY_PROPERTY(capacity) -
			             BATTERY_PROPERTY(min_capacity)) * 100) /
			             (BATTERY_PROPERTY(max_capacity) -
			              BATTERY_PROPERTY(min_capacity));
		else
			info->battery_life = -1;
	if ((BATTERY_PROPERTY(status) == BATTERY_STATUS_CHARGING)
	    || (BATTERY_PROPERTY(status) == BATTERY_STATUS_NOT_CHARGING)
	    || (BATTERY_PROPERTY(status) == BATTERY_STATUS_FULL))
		info->ac_line_status = APM_AC_ONLINE;
	else
		info->ac_line_status = APM_AC_OFFLINE;

	if (BATTERY_PROPERTY(status) == BATTERY_STATUS_CHARGING)
		info->battery_status = APM_BATTERY_STATUS_CHARGING;
	else {
		if (info->battery_life > 50)
			info->battery_status = APM_BATTERY_STATUS_HIGH;
		else if (info->battery_life > 5)
			info->battery_status = APM_BATTERY_STATUS_LOW;
		else
			info->battery_status = APM_BATTERY_STATUS_CRITICAL;
	}
	info->battery_flag = info->battery_status;

	bat_current = BATTERY_PROPERTY(current);
	if (bat_current)
		info->time = ((BATTERY_PROPERTY(capacity) - 
		              BATTERY_PROPERTY(min_capacity)) * 60) /
		             bat_current;
	else
		info->time = -1;

	info->units = APM_UNITS_MINS;

	up(&battery_class->sem);
	return;
}

static int __init apm_battery_init(void)
{
	printk(KERN_INFO "APM Battery Driver\n");

	old_apm_get_power_status = apm_get_power_status;
	apm_get_power_status = apm_battery_apm_get_power_status;
	return 0;
}

static void __exit apm_battery_exit(void)
{
	apm_get_power_status = old_apm_get_power_status;
	return;
}

module_init(apm_battery_init);
module_exit(apm_battery_exit);

MODULE_AUTHOR("Eugeny Boger");
MODULE_DESCRIPTION("APM driver for battery monitoring class");
MODULE_LICENSE("GPL");
