/*
 *  Universal battery monitor class
 *
 *  Copyright (c) 2007  Anton Vorontsov <cbou@mail.ru>
 *  Copyright (c) 2004  Szabolcs Gyurko
 *  Copyright (c) 2003  Ian Molton <spyro@f2s.com>
 *
 *  Modified: 2004, Oct     Szabolcs Gyurko
 *
 *  You may use this code as per GPL version 2
 *
 * All voltages, currents, capacities and temperatures in mV, mA, mAh and
 * tenths of a degree unless otherwise stated. It's driver's job to convert
 * its raw values to which this class operates. If for some reason driver
 * can't afford this requirement, then it have to create its own attributes,
 * plus additional "XYZ_units" for each of them.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/err.h>
#include <linux/battery.h>

/* If we have hwtimer trigger, then use it to blink charging LED */
#if defined(CONFIG_LEDS_TRIGGER_HWTIMER) ||                  \
               (defined(CONFIG_BATTERY_MODULE) &&            \
                defined(CONFIG_LEDS_TRIGGER_HWTIMER_MODULE))
	#define led_trigger_register_charging led_trigger_register_hwtimer
	#define led_trigger_unregister_charging led_trigger_unregister_hwtimer
#else
	#define led_trigger_register_charging led_trigger_register_simple
	#define led_trigger_unregister_charging led_trigger_unregister_simple
#endif

struct class *battery_class;

static void battery_external_power_changed(struct power_supplicant *pst,
                                           struct power_supply *psy)
{
	struct battery *bat = container_of(pst, struct battery, pst);
	pr_debug("%s\n", __FUNCTION__);
	if (bat->external_power_changed)
		bat->external_power_changed(bat);
	return;
}

int battery_is_external_power_supplied(struct battery *bat)
{
	pr_debug("%s\n", __FUNCTION__);
	return power_supplicant_am_i_supplied(&bat->pst);
}

void battery_status_changed(struct battery *bat)
{
	pr_debug("%s\n", __FUNCTION__);
	#ifdef CONFIG_LEDS_TRIGGERS
	switch(bat->get_status(bat))
	{
		case BATTERY_STATUS_FULL:
			led_trigger_event(bat->charging_trig, LED_OFF);
			led_trigger_event(bat->full_trig, LED_FULL);
			break;
		case BATTERY_STATUS_CHARGING:
			led_trigger_event(bat->charging_trig, LED_FULL);
			led_trigger_event(bat->full_trig, LED_OFF);
			break;
		default:
			led_trigger_event(bat->charging_trig, LED_OFF);
			led_trigger_event(bat->full_trig, LED_OFF);
			break;
	}
	#endif /* CONFIG_LEDS_TRIGGERS */
	return;
}

static char *status_text[] = {
	"Unknown", "Charging", "Discharging", "Not charging", "Full"
};

static ssize_t battery_show_status(struct device *dev,
                                   struct device_attribute *attr, char *buf)
{
	struct battery *bat = dev_get_drvdata(dev);
	int status = 0;
	if (bat->get_status) {
		status = bat->get_status(bat);
		if (status > 4)
			status = 0;
		return sprintf(buf, "%s\n", status_text[status]);
	}
	return 0;
}

/*
 * This is because the name "current" breaks the device attr macro.
 * The "current" word resolvs to "(get_current())" so instead of
 * "current" "(get_current())" appears in the sysfs.
 *
 * The source of this definition is the device.h which calls __ATTR
 * macro in sysfs.h which calls the __stringify macro.
 *
 * Only modification that the name is not tried to be resolved
 * (as a macro let's say).
 */

#define BATTERY_INT_ATTR(_name)                                         \
static ssize_t battery_show_##_name(struct device *dev,                 \
                                   struct device_attribute *attr,       \
                                   char *buf) {                         \
	struct battery *bat = dev_get_drvdata(dev);                     \
	return sprintf(buf, "%d\n", bat->get_##_name(bat));             \
}                                                                       \
static struct device_attribute dev_attr_##_name = {                     \
	.attr = { .name = #_name, .mode = 0444, .owner = THIS_MODULE }, \
	.show = battery_show_##_name,                                   \
	.store = NULL,                                                  \
}

static DEVICE_ATTR(status, 0444, battery_show_status, NULL);
BATTERY_INT_ATTR(min_voltage);
BATTERY_INT_ATTR(min_current);
BATTERY_INT_ATTR(min_capacity);
BATTERY_INT_ATTR(max_voltage);
BATTERY_INT_ATTR(max_current);
BATTERY_INT_ATTR(max_capacity);
BATTERY_INT_ATTR(temp);
BATTERY_INT_ATTR(voltage);
BATTERY_INT_ATTR(current);
BATTERY_INT_ATTR(capacity);

static int battery_create_attrs(struct battery *bat)
{
	int rc;

	#define create_bat_attr_conditional(name)                    \
	if(bat->get_##name) {                                        \
		rc = device_create_file(bat->dev, &dev_attr_##name); \
		if (rc) goto name##_failed;                          \
	}

	create_bat_attr_conditional(status);
	create_bat_attr_conditional(min_voltage);
	create_bat_attr_conditional(min_current);
	create_bat_attr_conditional(min_capacity);
	create_bat_attr_conditional(max_voltage);
	create_bat_attr_conditional(max_current);
	create_bat_attr_conditional(max_capacity);
	create_bat_attr_conditional(temp);
	create_bat_attr_conditional(voltage);
	create_bat_attr_conditional(current);
	create_bat_attr_conditional(capacity);

	#define remove_bat_attr_conditional(name)               \
	if(bat->get_##name)                                     \
		device_remove_file(bat->dev, &dev_attr_##name);

	goto success;

capacity_failed:     remove_bat_attr_conditional(current);
current_failed:      remove_bat_attr_conditional(voltage);
voltage_failed:      remove_bat_attr_conditional(temp);
temp_failed:         remove_bat_attr_conditional(max_capacity);
max_capacity_failed: remove_bat_attr_conditional(max_current);
max_current_failed:  remove_bat_attr_conditional(max_voltage);
max_voltage_failed:  remove_bat_attr_conditional(min_capacity);
min_capacity_failed: remove_bat_attr_conditional(min_current);
min_current_failed:  remove_bat_attr_conditional(min_voltage);
min_voltage_failed:  remove_bat_attr_conditional(status);
status_failed:
success:
	return rc;
}

static void battery_remove_attrs(struct battery *bat)
{
	remove_bat_attr_conditional(capacity);
	remove_bat_attr_conditional(current);
	remove_bat_attr_conditional(voltage);
	remove_bat_attr_conditional(temp);
	remove_bat_attr_conditional(max_capacity);
	remove_bat_attr_conditional(max_current);
	remove_bat_attr_conditional(max_voltage);
	remove_bat_attr_conditional(min_capacity);
	remove_bat_attr_conditional(min_current);
	remove_bat_attr_conditional(min_voltage);
	remove_bat_attr_conditional(status);
	return;
}

int battery_register(struct device *parent, struct battery *bat)
{
	int rc = 0;

	bat->dev = device_create(battery_class, parent, 0, "%s", bat->name);
	if(IS_ERR(bat->dev)) {
		rc = PTR_ERR(bat->dev);
		goto dev_create_failed;
	}

	dev_set_drvdata(bat->dev, bat);

	rc = battery_create_attrs(bat);
	if (rc)
		goto create_bat_attrs_failed;

	bat->pst.name = bat->name;
	bat->pst.power_supply_changed = battery_external_power_changed;
	rc = power_supplicant_register(&bat->pst);
	if (rc)
		goto power_supplicant_failed;

	#ifdef CONFIG_LEDS_TRIGGERS
	bat->charging_trig_name = kmalloc(strlen(bat->name) +
	                                  sizeof("-charging"), GFP_KERNEL);
	if (!bat->charging_trig_name) {
		rc = -ENOMEM;
		goto charging_trig_name_failed;
	}

	bat->full_trig_name = kmalloc(strlen(bat->name) +
	                              sizeof("-full"), GFP_KERNEL);
	if (!bat->full_trig_name) {
		rc = -ENOMEM;
		goto full_trig_name_failed;
	}

	strcpy(bat->charging_trig_name, bat->name);
	strcat(bat->charging_trig_name, "-charging");
	strcpy(bat->full_trig_name, bat->name);
	strcat(bat->full_trig_name, "-full");

	led_trigger_register_charging(bat->charging_trig_name,
	                              &bat->charging_trig);
	led_trigger_register_simple(bat->full_trig_name,
	                            &bat->full_trig);
	#endif /* CONFIG_LEDS_TRIGGERS */

	goto success;

#ifdef CONFIG_LEDS_TRIGGERS
full_trig_name_failed:
	kfree(bat->charging_trig_name);
charging_trig_name_failed:
#endif
	power_supplicant_unregister(&bat->pst);
power_supplicant_failed:
	battery_remove_attrs(bat);
create_bat_attrs_failed:
	device_unregister(bat->dev);
dev_create_failed:
success:
	return rc;
}

void battery_unregister(struct battery *bat)
{
	power_supplicant_unregister(&bat->pst);
	battery_remove_attrs(bat);
	device_unregister(bat->dev);

	#ifdef CONFIG_LEDS_TRIGGERS
	led_trigger_unregister_charging(bat->charging_trig);
	led_trigger_unregister_simple(bat->full_trig);
	kfree(bat->full_trig_name);
	kfree(bat->charging_trig_name);
	#endif

	return;
}

static int __init battery_class_init(void)
{
	battery_class = class_create(THIS_MODULE, "battery");

	if (IS_ERR(battery_class))
		return PTR_ERR(battery_class);

	return 0;
}

static void __exit battery_class_exit(void)
{
	class_destroy(battery_class);
	return;
}

EXPORT_SYMBOL_GPL(battery_register);
EXPORT_SYMBOL_GPL(battery_unregister);
EXPORT_SYMBOL_GPL(battery_status_changed);
EXPORT_SYMBOL_GPL(battery_is_external_power_supplied);

/* exported for the APM Power driver, APM emulation */
EXPORT_SYMBOL_GPL(battery_class);

subsys_initcall(battery_class_init);
module_exit(battery_class_exit);

MODULE_DESCRIPTION("Universal battery monitor class");
MODULE_AUTHOR("Ian Molton <spyro@f2s.com>, "
              "Szabolcs Gyurko, "
              "Anton Vorontsov <cbou@mail.ru>");
MODULE_LICENSE("GPL");
