/*
 * o2atom_cf.c -- O2 XDA Atom CompactFlash controller driver
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/platform_device.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/irq.h>

#include <pcmcia/ss.h>

#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/sizes.h>
#include <asm/arch/pxa-regs.h>

#define O2ATOM_CF_BASE	(0x14000000)
#define	CF_ATTR_PHYS	(O2ATOM_CF_BASE + (1 << 11))
#define	CF_IO_PHYS	(O2ATOM_CF_BASE)
#define	CF_MEM_PHYS	(O2ATOM_CF_BASE + (1 << 12))

/*--------------------------------------------------------------------------*/

static const char driver_name[] = "o2atom_vlio_cf";

struct o2atom_cf_socket {
	struct pcmcia_socket	socket;

	unsigned		present:1;

	struct platform_device	*pdev;
	struct o2atom_cf_data	*board;
	u_int			irq;
};

#define	SZ_2K			(2 * SZ_1K)

/*--------------------------------------------------------------------------*/

static int o2atom_cf_ss_init(struct pcmcia_socket *s)
{
        return 0;
}

static int o2atom_cf_get_status(struct pcmcia_socket *s, u_int *sp)
{
	struct o2atom_cf_socket	*cf;

	if (!sp)
		return -EINVAL;

	cf = container_of(s, struct o2atom_cf_socket, socket);

	/* NOTE: CF is always 3VCARD */
	*sp = SS_DETECT | SS_3VCARD | SS_READY | SS_POWERON;

	return 0;
}

static int
o2atom_cf_set_socket(struct pcmcia_socket *sock, struct socket_state_t *s)
{
	struct o2atom_cf_socket	*cf;

	cf = container_of(sock, struct o2atom_cf_socket, socket);

	/* toggle reset if needed */
        GPSR(97) = GPIO_bit(97);
	msleep(100);
        GPCR(97) = GPIO_bit(97);
	msleep(100);

	return 0;
}

static int o2atom_cf_ss_suspend(struct pcmcia_socket *s)
{
	return o2atom_cf_set_socket(s, &dead_socket);
}

static int o2atom_cf_set_io_map(struct pcmcia_socket *s, struct pccard_io_map *io)
{
	struct o2atom_cf_socket	*cf;

	cf = container_of(s, struct o2atom_cf_socket, socket);
	io->flags &= (MAP_ACTIVE | MAP_16BIT | MAP_AUTOSZ);

	if (!(io->flags & (MAP_16BIT|MAP_AUTOSZ))) {
		pr_info("%s: 8bit i/o bus\n", driver_name);
	} else {
		pr_info("%s: 16bit i/o bus\n", driver_name);
	}

	io->start = cf->socket.io_offset;
	io->stop = io->start + SZ_2K - 1;

	return 0;
}

/* pcmcia layer maps/unmaps mem regions */
static int
o2atom_cf_set_mem_map(struct pcmcia_socket *s, struct pccard_mem_map *map)
{
	struct o2atom_cf_socket	*cf;

	if (map->card_start)
		return -EINVAL;

	cf = container_of(s, struct o2atom_cf_socket, socket);

	map->flags &= MAP_ACTIVE|MAP_ATTRIB|MAP_16BIT;
	if (map->flags & MAP_ATTRIB)
		map->static_start = CF_ATTR_PHYS;
	else
		map->static_start = CF_MEM_PHYS;

	return 0;
}

static struct pccard_operations o2atom_cf_ops = {
	.init			= o2atom_cf_ss_init,
	.suspend		= o2atom_cf_ss_suspend,
	.get_status		= o2atom_cf_get_status,
	.set_socket		= o2atom_cf_set_socket,
	.set_io_map		= o2atom_cf_set_io_map,
	.set_mem_map		= o2atom_cf_set_mem_map,
};

/*--------------------------------------------------------------------------*/

static int __init o2atom_cf_probe(struct device *dev)
{
	struct o2atom_cf_socket	*cf;
	struct o2atom_cf_data	*board = dev->platform_data;
	struct platform_device	*pdev = to_platform_device(dev);
	struct resource		*io;
	int			status;

	io = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!io)
		return -ENODEV;

	cf = kcalloc(1, sizeof *cf, GFP_KERNEL);
	if (!cf)
		return -ENOMEM;

	cf->board = board;
	cf->pdev = pdev;
	dev_set_drvdata(dev, cf);

	pxa_gpio_mode(GPIO18_RDY_MD); // GPIO_ALT_FN_1_IN
        pxa_gpio_mode(GPIO33_nCS_5_MD); // GPIO_ALT_FN_2_OUT | GPIO_DFLT_HIGH
        pxa_gpio_mode(GPIO49_nPWE_MD);
        pxa_gpio_mode(GPIO54_pSKTSEL_MD); // nOE

        MSC2 = 0x52dc0000;
        (void)MSC2;

        MDREFR |= 0x02080000;

        pxa_gpio_mode(9 | GPIO_IN); // IREQn
	set_irq_type(IRQ_GPIO(9), IRQT_RISING);
        pxa_gpio_mode(97 | GPIO_OUT | GPIO_DFLT_LOW); // RESET
        pxa_gpio_mode(98 | GPIO_OUT | GPIO_DFLT_HIGH); // 3.3v LDO enable
        pxa_gpio_mode(99 | GPIO_OUT | GPIO_DFLT_HIGH); // PDn

	mdelay(100);
        GPSR(97) = GPIO_bit(97);
	mdelay(100);
        GPCR(97) = GPIO_bit(97);
	mdelay(100);

	cf->irq = IRQ_GPIO(9);
	cf->socket.pci_irq = IRQ_GPIO(9);

	/* pcmcia layer only remaps "real" memory not iospace */
	cf->socket.io_offset = (unsigned long) ioremap(CF_IO_PHYS, SZ_2K);
	if (!cf->socket.io_offset)
		goto fail1;

	if (!request_mem_region(io->start, io->end + 1 - io->start,
				driver_name))
		goto fail1;

	pr_info("%s", driver_name);

	cf->socket.owner = THIS_MODULE;
	cf->socket.dev.dev = dev;
	cf->socket.ops = &o2atom_cf_ops;
	cf->socket.resource_ops = &pccard_static_ops;
	cf->socket.features = SS_CAP_PCCARD | SS_CAP_STATIC_MAP
				| SS_CAP_MEM_ALIGN;
	cf->socket.map_size = SZ_2K;
	cf->socket.io[0].res = io;

	status = pcmcia_register_socket(&cf->socket);
	if (status < 0)
		goto fail2;

	return 0;

fail2:
	iounmap((void __iomem *) cf->socket.io_offset);
	release_mem_region(io->start, io->end + 1 - io->start);
fail1:
	free_irq(IRQ_GPIO(9), cf);
fail0a:
fail0:
	kfree(cf);
	return status;
}

static int __exit o2atom_cf_remove(struct device *dev)
{
	struct o2atom_cf_socket	*cf = dev_get_drvdata(dev);
	struct resource		*io = cf->socket.io[0].res;

	pcmcia_unregister_socket(&cf->socket);
	iounmap((void __iomem *) cf->socket.io_offset);
	release_mem_region(io->start, io->end + 1 - io->start);
	free_irq(cf->irq, cf);
	kfree(cf);
	return 0;
}

static struct device_driver o2atom_cf_driver = {
	.name		= (char *) driver_name,
	.bus		= &platform_bus_type,
	.probe		= o2atom_cf_probe,
	.remove		= __exit_p(o2atom_cf_remove),
	.suspend	= pcmcia_socket_dev_suspend,
	.resume		= pcmcia_socket_dev_resume,
};

/*--------------------------------------------------------------------------*/

static int __init o2atom_cf_init(void)
{
	return driver_register(&o2atom_cf_driver);
}
module_init(o2atom_cf_init);

static void __exit o2atom_cf_exit(void)
{
	driver_unregister(&o2atom_cf_driver);
}
module_exit(o2atom_cf_exit);

MODULE_DESCRIPTION("O2 XDA Atom CompactFlash Driver");
