/*
 *  Universal battery monitor class
 *
 *  Copyright (c) 2007  Anton Vorontsov <cbou@mail.ru>
 *  Copyright (c) 2004  Szabolcs Gyurko
 *  Copyright (c) 2003  Ian Molton <spyro@f2s.com>
 *
 *  Modified: 2004, Oct     Szabolcs Gyurko
 *
 *  You may use this code as per GPL version 2
 *
 * All voltages, currents, capacities and temperatures in mV, mA, mAh and
 * tenths of a degree unless otherwise stated. It's driver's job to convert
 * its raw values to which this class operates. If for some reason driver
 * can't afford this requirement, then it have to create its own attributes,
 * plus additional "XYZ_units" for each of them.
 */

#ifndef _LINUX_BATTERY_H
#define _LINUX_BATTERY_H

#include <linux/device.h>
#include <linux/external_power.h>
#include <linux/leds.h>

#define BATTERY_STATUS_UNKNOWN      0
#define BATTERY_STATUS_CHARGING     1
#define BATTERY_STATUS_DISCHARGING  2
#define BATTERY_STATUS_NOT_CHARGING 3
#define BATTERY_STATUS_FULL         4

/* 
 * For systems where the charger determines the maximum battery capacity
 * the min and max fields should be used to present these values to user
 * space. Unused/uknown fields can be NULL and will not appear in sysfs.
 */

struct battery {
	struct device *dev;
	char *name;

	/* For APM emulation, think legacy userspace. */
	int main_battery;

	/* executed in userspace, feel free to sleep */
	int (*get_min_voltage)(struct battery *bat);
	int (*get_min_current)(struct battery *bat);
	int (*get_min_capacity)(struct battery *bat);
	int (*get_max_voltage)(struct battery *bat);
	int (*get_max_current)(struct battery *bat);
	int (*get_max_capacity)(struct battery *bat);
	int (*get_temp)(struct battery *bat);
	int (*get_voltage)(struct battery *bat);
	int (*get_current)(struct battery *bat);
	int (*get_capacity)(struct battery *bat);
	int (*get_status)(struct battery *bat);

	/* drivers should not sleep inside it, you'll get there from ISRs */
	void (*external_power_changed)(struct battery *bat);

	/* private */
	struct power_supplicant pst;

	#ifdef CONFIG_LEDS_TRIGGERS
	struct led_trigger *charging_trig;
	char *charging_trig_name;
	struct led_trigger *full_trig;
	char *full_trig_name;
	#endif
};

/* 
 * This is recommended structure to specify static battery parameters.
 * Generic one, parametrizable for different batteries. Battery device
 * itself does bot use it, but that's what implementing most drivers,
 * should try reuse for consistency.
 */

struct battery_info {
	char *name;
	int min_voltage;
	int max_voltage;
	int min_current;
	int max_current;
	int min_capacity;
	int max_capacity;
	int main_battery;
};

extern void battery_status_changed(struct battery *bat);
extern int battery_is_external_power_supplied(struct battery *bat);
extern int battery_register(struct device *parent, struct battery *bat);
extern void battery_unregister(struct battery *bat);

/* For APM emulation, think legacy userspace. */
extern struct class *battery_class;

#endif
