/*****************************************************************************
 *
 * Microchip DeviceNet Stack (Router Object Header)
 *
 *****************************************************************************
 * FileName:        route.h
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.10.02 or higher
 * Linker:          MPLINK 03.20.01 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *
 * This file contains the Router object described in section 6-3 of 
 * volume 2 of the DeviceNet specification.
 *
 * The router provides several parameters to Explicit Message Handlers.
 * All of the parameters are globals defined in an access section for 
 * fast access. 
 *
 * Receive Buffer									Transmit Buffer
 * ---------------									---------------	
 * |   Header    | 									|   Header    |
 * |   Service   |									|   Service   |
 * |   ClassID   |									|    .....    | <- pOutBuf
 * | InstanceID  |									|    .....    |
 * | AttributeID | <- pInBuf, Ptr to rx data		|    .....    |
 * |    .....    |    aInBufLen, len of buf			|    .....    |
 * |    .....    |	  aInBufDataLen, len of data	|    .....    |
 * |    .....    |									|    .....    |
 * |    .....    | 									|    .....    |
 * |    .....    |									|    .....    |
 * ---------------									---------------		
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			04/03/03	...	
 * 
 *****************************************************************************/



typedef struct _ROUTE_INFO
{
	USINT		header;				// The header of the explicit message
	USINT		service;			// The service requested
#if (CLASS_WIDTH_16BIT)
 	UINT		classID;			// The class
#else 
	USINT		classID;
#endif
#if (INSTANCE_WIDTH_16BIT)
 	UINT		instanceID;			// The instance
#else
	USINT		instanceID;
#endif
	USINT		attributeID;		// The attribute
	USINT * 	pInBuf;				// Pointer to the received data
	USINT * 	pOutBuf;			// Pointer to the data buffer to transmit
	USINT 		inBufLen;			// The size of the receive buffer
	USINT		inBufDataLen;		// The amount of data in the receive buffer
	USINT		outBufLen;			// The size of the transmit buffer
	USINT		outBufDataLen;		// The amount of data in the transmit buffer
}ROUTE;


/*********************************************************************
 * Route variable made global 
 ********************************************************************/


extern NEAR ROUTE route;

/*********************************************************************
 * Function:        unsigned char RouteMessage(void)
 *
 * PreCondition:    
 *
 * Input:   		aHeader, aService, aClassID, aInstanceID         
 *					aAttributeID, pInBuf, pOutBuf, aInBufLen
 *					aInBufDataLen, aOutBufLen, aOutBufDataLen 
 *                  
 * Output:       	aHeader, aService, pInBuf, pOutBuf, aOutBufDataLen	
 *
 * Side Effects:    
 *
 * Overview:        This is the start of message routing. This function
 * 					decodes the class then calls the appropriate
 *					object to decode the instance, service, and if 
 *					necessary the attribute. If the class is not 
 *					defined in this object then an error is returned 
 *					in aService (= _ERROR_RESPONSE).
 *
 *					pOutBuf is loaded with the responce data and the
 *					length aOutBufDataLen is adjusted.
 *
 * Note:            Edit the case statement below to add more defined
 *					classes of objects to the list.
 ********************************************************************/
unsigned char RouteMessage(void);


/*********************************************************************
 * Function:        unsigned char _RouteExplMsgHandler(void)
 *
 * PreCondition:    
 *
 * Input:           
 *					
 *                  
 * Output:         
 *				
 *				
 *			
 *
 * Side Effects:    
 *
 * Overview:        Handler for explicit messaging 
 *
 * Note:            None
 ********************************************************************/
unsigned char _RouteExplMsgHandler(void);
   

/*********************************************************************
 * Function:        void RoutePutByte(USINT dataByte)
 *
 * PreCondition:    
 *
 * Input:           USINT - a byte to put into the output buffer
 *       
 * Output:       		
 *
 * Side Effects:    
 *
 * Overview:      	This function puts a byte in the output buffer.
 *					Plus it automatically adjusts the buffer count and
 *					pointer into the buffer. 
 *
 * Note:          	 
 ********************************************************************/
void RoutePutByte(USINT dataByte);
#define mRoutePutByte(dataByte)		RoutePutByte(dataByte)



/*********************************************************************
 * Function:        USINT RouteGetByte(void)
 *
 * PreCondition:    
 *
 * Input:           None.
 *       
 * Output:       	USINT - byte from the buffer.		
 *
 * Side Effects:    
 *
 * Overview:      	This function returns a byte from the router's 
 *					input buffer. Plus it auto adjusts to point the 
 *					the next byte in the buffer.   
 *
 * Note:            
 ********************************************************************/
USINT RouteGetByte(void);
#define mRouteGetByte()		RouteGetByte()



/*********************************************************************
 * Function:        unsigned char RouteTestValidInputDataLen(unsigned char expectedSize)
 *
 * PreCondition:    
 *
 * Input:           
 *       
 * Output:       		
 *
 * Side Effects:    
 *
 * Overview:      	This function tests the input buffer against 
 *					the expected size and automatically generates
 *					the necessary error packet if failed. 
 *
 * Note:          	 
 ********************************************************************/
unsigned char RouteTestValidInputDataLen(unsigned char expectedSize);
#define	mRouteTestValidInputDataLen(expSize) RouteTestValidInputDataLen(expSize)
#define mRouteTestNonValidInputDataLen(expSize) (!RouteTestValidInputDataLen(expSize))



/*********************************************************************
 * Function:        void RoutePutError(unsigned char errorCode)
 *
 * PreCondition:    
 *
 * Input:           
 *       
 * Output:       		
 *
 * Side Effects:    
 *
 * Overview:      	This function writes the requested error to the 
 *					buffer. 
 *
 * Note:          	 
 ********************************************************************/
void RoutePutError(unsigned char errorCode);
#define mRoutePutError(eCode)	RoutePutError(eCode)





/*********************************************************************
 * Function:        USINT RouteRxLen(void)
 *
 * PreCondition:    
 *
 * Input:           None.
 *       
 * Output:       	USINT - current length of the input buffer		
 *
 * Side Effects:    
 *
 * Overview:      	This function returns the current length of the
 *   				input buffer. 
 *
 * Note:          	Use this function in combination with RouteGetByte
 *					to read the contents of the input buffer. 
 ********************************************************************/
//#define mRouteRxLen()	(aInBufDataLen)
#define mRouteRxLen()	(route.inBufDataLen)
#define mRouteInLen()	(route.inBufDataLen)	

/*********************************************************************
 * Function:        USINT RouteTxLen(void)
 *
 * PreCondition:    
 *
 * Input:           
 *       
 * Output:       	USINT - the maximum size of the output buffer
 *
 * Side Effects:    
 *
 * Overview:      	This function returns the maximum size of the 
 *					output buffer.
 *
 * Note:          	Use this function in combination with RoutePutByte
 *					to write the contents of the output buffer.	 
 ********************************************************************/
//#define mRouteTxLen()	(aOutBufLen)
#define mRouteTxLen()	(route.outBufLen)
#define	mRouteOutLen()	(route.outBufLen)

/*********************************************************************
 * Function:        USINT RouteGetHeader(void)
 *
 * PreCondition:    
 *
 * Input:           
 *       
 * Output:       	USINT
 *
 * Side Effects:    
 *
 * Overview:      	This function returns the header of the Explicit
 *					message.
 *
 * Note:          		 
 ********************************************************************/
//#define mRouteGetHeader()	(aHeader)
#define mRouteGetHeader()	(route.header)



/*********************************************************************
 * Function:        USINT RouteGetServiceID(void)
 *
 * PreCondition:    
 *
 * Input:           
 *       
 * Output:       	USINT
 *
 * Side Effects:    
 *
 * Overview:      	This function returns the Service ID of the Explicit
 *					message.
 *
 * Note:          		 
 ********************************************************************/
//#define mRouteGetServiceID()	(aService)
#define mRouteGetServiceID()	(route.service)


/*********************************************************************
 * Function:        UINT RouteGetClassID(void)
 *
 * PreCondition:    
 *
 * Input:           
 *       
 * Output:       	UINT
 *
 * Side Effects:    
 *
 * Overview:      	This function returns the Class ID of the Explicit
 *					message.
 *
 * Note:          		 
 ********************************************************************/
//#define mRouteGetClassID()	(aClassID)
#if (CLASS_WIDTH_16BIT)
#define mRouteGetClassID()	(route.classID.word)
#else
#define mRouteGetClassID()	(route.classID)
#endif



/*********************************************************************
 * Function:        UINT RouteGetInstanceID(void)
 *
 * PreCondition:    
 *
 * Input:           
 *       
 * Output:       	UINT
 *
 * Side Effects:    
 *
 * Overview:      	This function returns the Instance ID of the 
 *					Explicit message.
 *
 * Note:          		 
 ********************************************************************/
//#define mRouteGetInstanceID()	(aInstanceID)
#if (INSTANCE_WIDTH_16BIT)
#define mRouteGetInstanceID()	(route.instanceID.word)
#else
#define mRouteGetInstanceID()	(route.instanceID)
#endif


/*********************************************************************
 * Function:        USINT RouteGetAttributeID(void)
 *
 * PreCondition:    
 *
 * Input:           
 *       
 * Output:       	USINT
 *
 * Side Effects:    
 *
 * Overview:      	This function returns the Attribute ID of the 
 *					Explicit message (if any).
 *
 * Note:          		 
 ********************************************************************/
//#define mRouteGetAttributeID()	(aAttributeID)
#define mRouteGetAttributeID()	(route.attributeID)




#define	mRouteGetInBufferPtr()			(route.pInBuf)
#define	mRouteGetOutBufferPtr()			(route.pOutBuf)
#define mRouteGetInBufferLength()		(route.inBufLen)
#define mRouteGetInBufferDataLength()	(route.inBufDataLen)
#define mRouteGetOutBufferLength()		(route.outBufLen)
#define mRouteGetOutBufferDataLength()	(route.outBufDataLen)



						  




/*********************************************************************
 * Function:        void RoutePutServiceID(USINT serviceCode)
 *
 * PreCondition:    
 *
 * Input:           USINT
 *       
 * Output:       	
 *
 * Side Effects:    
 *
 * Overview:      	This function sets the service code.
 *
 * Note:          	This is useful when the Explicit message handler
 *					must change the service to report an error to the
 *					client.
 ********************************************************************/
//#define mRoutePutServiceID(serviceCode)	(aService = serviceCode)
#define	mRoutePutServiceID(serviceCode)	(route.service = serviceCode)


#define	mRoutePutInBufferPtr(inPtr)				(route.pInBuf = inPtr)
#define	mRoutePutOutBufferPtr(outPtr)			(route.pOutBuf = outPtr)
#define mRoutePutInBufferLength(length)			(route.inBufLen = length)
#define mRoutePutInBufferDataLength(length)		(route.inBufDataLen = length)
#define mRoutePutOutBufferLength(length)		(route.outBufLen = length)
#define mRoutePutOutBufferDataLength(length)	(route.outBufDataLen = length)



	 
