package antichess;

import java.util.ArrayList;
import java.util.List;

/**
 * Extension of a ChessBoard with board-level moves modified for
 * Antichess.
 * <p>
 * Adds the additional functionality over ChessBoards of checking if
 * the player has won by having his opponent capture all of his pieces,
 * and disallowing moves if they do not capture any piece while some
 * other possible move does.
 * <p>
 * AntichessBoard also adds one more public end-game condition,
 * AntichessBoard.OUTOFPIECES (value = 4), to that set from ChessBoard.
 */
public class AntichessBoard extends ChessBoard implements Cloneable
{
	// Additional end conditions for Antichess
	/**
	 * End-game condition variable representing that some player
	 * has lost all of his pieces.
	 */
	public static final int OUTOFPIECES = 4;
	
	// Caching variable for existsACaptureMove
	// Values to meanings for existsACaptureMove
	//	-1	->	unset
	//	0	->	false
	//	1	->	true
	private int existsACaptureMove;
	
	/**
	 * Creates a new AntichessBoard and initializes it appropriately.
	 */
	public AntichessBoard() {
		super();
		existsACaptureMove = -1;
	}
	
	@Override
	public AntichessBoard clone() {
		AntichessBoard ab = new AntichessBoard();
		copyThis(ab);
		return ab;
	}
	
	/**
	 * Inheritable helper method for copying various
	 * data members from this Board to an empty Board.
	 * @param b the empty board into which this board
	 * should be copied.
	 */
	protected void copyThis(AntichessBoard ab) {
		super.copyThis(ab);
		ab.existsACaptureMove = this.existsACaptureMove;
	}
	
	@Override
	protected void switchPlayer() {
		existsACaptureMove = -1;
		super.switchPlayer();
	}
	
	@Override
	protected void checkForEndCondition() {
		if (getPieces(player).size() == 1) // The player is down to just his king
			endGame(OUTOFPIECES, player);
		else if (allLegalMoves().isEmpty() && isPlayerInCheck(player))
			endGame(CHECKMATE, player.otherPlayer());
		else
			endGame(NOTOVER, Player.NONE);
	}
	
	// Modified to allow null moves.
	@Override
	public void doMove(ChessMove m, String timestamp) throws IllegalMoveException {
		if (m == null) {
			pushMove(m, timestamp);
			switchPlayer();
			notifyObservers();
		} else super.doMove(m, timestamp);
	}
	
	/**
	 * @param m Move to check whether or not it involves a capture
	 * @return whether the given move involves a capture
	 */
	private boolean isACaptureMove(ChessMove m) {
		return m.getCapturedPiece() != null;
	}
	
	/**
	 * @return true iff there exists a capture move for the current player
	 */
	private boolean existsACaptureMove() {
		if (existsACaptureMove != -1) {
			if (existsACaptureMove == 0)
				return false;
			else
				return true;
		} else {
			List<Piece> pieces = getPieces(player);
			List<ChessMove> moves;

			for (Piece p : pieces) {
				moves = legalMovesInChess(p);
				for (ChessMove m : moves)
					if (isACaptureMove(m)) {
						existsACaptureMove = 1;
						return true;
					}
			}
			existsACaptureMove = 0;
			return false;
		}
	}
	
	/**
	 * Private method to test if a given move is legal in Chess.
	 * @param m the move to test
	 * @return true iff <code>m</code> is legal in Chess.
	 */
	private boolean isMoveLegalInChess(ChessMove m) {
		return super.isMoveLegal(m);
	}
	
	/**
	 * Private method to return a list of the legal moves for
	 * Chess for a given Piece p.
	 * @param p the Piece for which to get the legal Chess moves
	 * @return a list of ChessMoves representing the legal moves
	 * <code>p</code> may make in Chess
	 */
	private List<ChessMove> legalMovesInChess(Piece p) {
		assert p.getPlayer().equals(player) : p + " is the wrong player";
		List<ChessMove> legalMoves = new ArrayList<ChessMove>();
		List<ChessMove> possibleMoves = p.getMovementType().getMoves(p, this);
		
		for (ChessMove possibleMove : possibleMoves)
			if (isMoveLegalInChess(possibleMove))
				legalMoves.add(possibleMove);
		
		return legalMoves;
	}
	
	@Override
	public List<ChessMove> allLegalMoves() {
		List<ChessMove> allLegalMoves = new ArrayList<ChessMove>();
		List<Piece> pieces = getPieces(player);
		List<ChessMove> moves;
		
		for (Piece p : pieces) {
			moves = super.legalMoves(p);
			for (ChessMove m : moves) {
				if (isACaptureMove(m)) {
					if (!(existsACaptureMove == 1)) {
						existsACaptureMove = 1;
						allLegalMoves.clear();
					}
					allLegalMoves.add(m);
				} else {
					if (!(existsACaptureMove == 1))
						allLegalMoves.add(m);
				}
			}
		}
		
		if (existsACaptureMove == -1)
				existsACaptureMove = 0;
		
		return allLegalMoves;
	}
	
	@Override
	public List<ChessMove> legalMoves(Piece p) {
		List<ChessMove> legalChessMoves = super.legalMoves(p);
		if (!existsACaptureMove())
			return legalChessMoves;
		else {
			List<ChessMove> legalMoves = new ArrayList<ChessMove>();
			for (ChessMove m : legalChessMoves)
				if (isACaptureMove(m))
					legalMoves.add(m);
			return legalMoves;
		}
	}
	
	@Override
	public boolean isMoveLegal(ChessMove m) {
		if(super.isMoveLegal(m)) {
			if (isACaptureMove(m)) {
				existsACaptureMove = 1;
				return true;
			} else
				return !existsACaptureMove();
		}
		return false;
	}
	
}
