package antichess;

import antichess.viewgui.BoardView;

/**
 * This class handles the interactions between the Controller and the View
 * ChessPlayer is an implementation of the HumanPlayer interface.
 * @see HumanPlayer
 * 
 * 
 * @author nlharr
 * @specfield board   		//Board stores the board the HumanPlayer is interacting with
 * @specfield controller 	//controller specifies the GameController that this HumanPlayer is controlled by
 * @specfield boardView 	//BoardView that the HumanPlayer asks for moves
 * @specfield currentMove	//stores the currentMove to be updated to by the GameController
 * @specfield playerType	//Player 
 */
public class ChessPlayer implements HumanPlayer{  
	
	//Fields 
	private ChessBoard board;
	private GameController controller;
	private BoardView boardView;
	private Move currentMove;
	private Player playerType;
	
//	Abstraction Function
	//board = board
	//controller = controller
	//boardView = boardView
	//currentMove = currentMove if currentMove=null that means there is no currentMove
	//playerType = playerType
	
	//Representation Invariants
	//board != null
	//boardView != null
	//playerType != null
	
	/**
	 * Creates a new ChessPlayer
	 * @effects board = board, boardView = boardView, 
	 * 			currentMove = NONE, playerType = playerType  
	 */
	public ChessPlayer(ChessBoard board, BoardView boardView, Player playerType){
		this.board = board;
		this.currentMove = null;
		this.boardView = boardView;
		this.playerType = playerType;
		this.controller = null;
		checkRep();
	}
	
	private void checkRep(){
		if (boardView == null) throw new RuntimeException("boardView is null");
		if (playerType == null) throw new RuntimeException("playerType is null");
		if (board == null) throw new RuntimeException("board is null");
	}
	
	
	/**
	 * This function returns the currentMove and then sets the currentMove to null
	 * currentMove is null if there is no move cached.
	 * 
	 * @return currentMove
	 * @modifies currentMove
	 * @effects currentMove = null
	 */
	public synchronized Move getMove(GameController controller){
		this.controller = controller;
		if (currentMove == null){
			boardView.setAcceptingInput(this);
			return null;
		}else{
			Move tempMove = currentMove;
			currentMove = null;
			return tempMove;
		}
	}
	
	
	/**
	 * @returns playerType
	 */
	public synchronized Player getPlayerType(){
		return playerType;
	}
	
	/**
	 * Specific for ChessMove right now
	 * @requires startRows, startColumns, destinationRows, destinationColumnes are all not null 
	 * @modifies currentMove
	 * @effects if the move between the start and destination is legal
	 * 			it sets currentMove to that move
	 *
	 * @return true if the move between the start and destination is legal
	 * 			else false
	 */
	

	public synchronized boolean setMove(int[] startRows, int[] startColumns,
										int[] destinationRows, int[] destinationColumns){
		Piece startPiece = board.getPieceAt(startRows[0], startColumns[0]);
		if ((startPiece == null)||
				(!startPiece.getPlayer().equals(playerType))){
			return false;
		}
		Piece capturedPiece = board.getPieceAt(destinationRows[0], destinationColumns[0]);
		ChessMove newMove = new ChessMove(startPiece, destinationRows[0], 
									 destinationColumns[0], capturedPiece);
		//for debugging
		System.out.println(newMove.toString());
		if (board.isMoveLegal(newMove)){
			currentMove = newMove;
			return true;
		}else{
			return false;
		}
	}
	
	/**
	 * Notifies the controller that requested a move if it has a move
	 * @throws RuntimeException if no move has been requested so there is
	 * 			no controller to return a value to
	 */
	public synchronized void notifyHasMove(){
		if (controller == null){
			throw new RuntimeException("Controller not specified");
		}
		controller.notifyControllerHasMove();
	}
	

}
