// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess;

import java.io.*;
import java.util.*;
import javax.xml.*;
import javax.xml.parsers.*;
import javax.xml.transform.dom.*;
import javax.xml.validation.*;
import javax.xml.xpath.*;
import org.w3c.dom.*;
import org.xml.sax.*;

/**
 * The GameReader class reads XML game files for antichess games.
 *
 * @specfield board : ChessBoard     // The Board as of when the game was saved
 * @specfield timed : boolean   // Indicates whether the game was times
 * @specfield whiteTimer   : Timer // White's Timer as of when the game was saved
 * @specfield blackTimer   : Timer // Black's Timer as of when the game was saved
 **/
public class GameReader
{
	final private String ANTICHESS_RULESET = "6170-spring-2007";

	private AntichessBoard board;
	private boolean isTimed = false;
	private GameTimer whiteTimer;
	private GameTimer blackTimer;

	/**
	 * @effects constructs a new GameReader containing the data in the
	 * given XML stream
	 * @throws InvalidGameFileException if the stream is not a valid game (e.g. it does not validate)
	 */
	public GameReader(InputStream stream)
		throws InvalidGameFileException, IOException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		try {
			DocumentBuilder builder = factory.newDocumentBuilder();
			Document document = builder.parse(new InputSource(stream));
			loadDocument(document);
		} catch (SAXException sxe) {
			throw new InvalidGameFileException(sxe.toString());
		} catch (ParserConfigurationException pce) {
			throw new RuntimeException("Couldn't construct a parser: " + pce);
		}
	}
	
	/**
	 * @effects constructs a new GameReader containing the data in the
	 * given XML file
	 * @throws IOException if there is an IO error reading the file
	 * @throws InvalidGameFileException if the file is not valid game file (e.g. it does not validate)
	 */
	public GameReader(File gameFile)
		throws InvalidGameFileException, IOException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		try {
			DocumentBuilder builder = factory.newDocumentBuilder();
			Document document = builder.parse(gameFile);
			loadDocument(document);
		} catch (SAXException sxe) {
			throw new InvalidGameFileException(sxe.toString());
		} catch (ParserConfigurationException pce) {
			throw new RuntimeException("Couldn't construct a parser: " + pce);
		}
	}

	/**
	 * @effects constructs a new GameReader by reading the given string of XML
	 * @throws InvalidGameFileException if the file is not valid game file (e.g. it does not validate)
	 */
	public GameReader(String xmlData)
		throws InvalidGameFileException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		try {
			DocumentBuilder builder = factory.newDocumentBuilder();
			Document document = builder.parse(new InputSource(new StringReader(xmlData)));
			loadDocument(document);
		} catch (SAXException sxe) {
			throw new InvalidGameFileException(sxe.toString());
		} catch (ParserConfigurationException pce) {
			throw new RuntimeException("Couldn't construct a parser: " + pce);
		} catch (IOException ex) {
			// This just shouldn't happen. srsly.
			throw new RuntimeException("There was an IO error reading from a String.");
		}
	}

	/**
	 * @effects Load the given board with the data parsed from an XML
	 * &lt;pieces&gt; tag, including both the start and end tags, and
	 * set the specified player.
	 * @throws InvalidGameFileException if the XML is invalid
	 */
	public static void loadBoard(ChessBoard board, String xml, Player player)
		throws InvalidGameFileException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		try {
			DocumentBuilder builder = factory.newDocumentBuilder();
			Document document = builder.parse(new InputSource(new StringReader(xml)));
			List<Piece> pieces = parsePieces(document.getDocumentElement());
			board.loadGame(pieces, player, null);
		} catch (SAXException sxe) {
			throw new InvalidGameFileException(sxe.toString());
		} catch (ParserConfigurationException pce) {
			throw new RuntimeException("Couldn't construct a parser: " + pce);
		} catch (IOException ex) {
			// This just shouldn't happen. srsly.
			throw new RuntimeException("There was an IO error reading from a String.");
		}
	}

	/**
	 * @effects load a game from the specified DOM Document object
	 */
	private void loadDocument(Document document)
		throws InvalidGameFileException {
		Schema schema;
		try {
			schema = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI).
				newSchema(GameReader.class.getResource("antichess.xsd"));
		} catch(SAXException e) {
			throw new RuntimeException("Unable to read antichess.xsd: " + e);
		}
		// This is commented out for the moment since for some reason
		// the validator fails to load properly on some computers with
		// no useful error message.
		/* try {
			Validator validator = schema.newValidator();
			//validator.validate(new DOMSource(document));
		} catch(SAXException e) {
			throw new InvalidGameFileException("XML document does not validate: " + e);
		} catch(IOException e) {
			throw new InvalidGameFileException("IO Error parsing document: " + e);
			}*/
		
		Node root = getChildOfType(document, Node.ELEMENT_NODE);
		if(!root.getNodeName().equals("game"))
			throw new InvalidGameFileException("Expected a <game> node");
		String ruleset = getAttribute(root, "ruleset");
		if(!ruleset.equals(ANTICHESS_RULESET))
			throw new InvalidGameFileException("Unknown ruleset: " + ruleset);
		XPath xpath = XPathFactory.newInstance().newXPath();
		try {
			parseTime((Node)xpath.evaluate("/game/time", document, XPathConstants.NODE));
			MoveHistory<ChessMove> history = parseMoveHistory((Node)xpath.evaluate("/game/moveHistory", document, XPathConstants.NODE));
			List<Piece> pieces = parsePieces((Node)xpath.evaluate("/game/pieces", document, XPathConstants.NODE));
			Player player;
			player = history == null
				? Player.WHITE
				: history.getLastMove().getPlayer().otherPlayer();
			board = new AntichessBoard();
			board.loadGame(pieces, player, history);
			Object gameOver = xpath.evaluate("/game/gameOver", document, XPathConstants.NODE);
			if(gameOver != null)
				parseGameOver((Node)gameOver);
		} catch(XPathExpressionException e) {
			throw new RuntimeException("Internal error: " + e);
		} catch(Exception e) {
			e.printStackTrace();
			throw new InvalidGameFileException(e.toString());
		}
	}

	/**
	 * @effects Populates whiteTimer and blackTimer based on reading
	 * the given <time> XML node
	 */
	private void parseTime(Node time) {
		isTimed = "true".equals(getAttribute(time, "timed"));
		if(!isTimed) return;
		long initWhite = Long.parseLong(getAttribute(time, "initWhite"));
		long initBlack = Long.parseLong(getAttribute(time, "initBlack"));
		long curWhite = Long.parseLong(getAttribute(time, "currentWhite"));
		long curBlack = Long.parseLong(getAttribute(time, "currentBlack"));
		whiteTimer = new GameTimer(initWhite, curWhite, 500);
		blackTimer = new GameTimer(initBlack, curBlack, 500);
	}

	/**
	 * @return The MoveHistory represented by the given <moveHistory>
	 * XML element
	 */
	private MoveHistory<ChessMove> parseMoveHistory(Node history)
		throws InvalidGameFileException {
		AntichessBoard board = new AntichessBoard();
		/* We actually play the game on a board as we load it */
		board.newGame();
		NodeList children;
		MoveHistory<ChessMove> moveHistory = null;
		Node node;
		ChessMove m;

		children = history.getChildNodes();
		for(int i=0;i<children.getLength();i++) {
			node = children.item(i);
			if("move".equals(node.getNodeName())) {
				m = parseMove(node, board);
				try {
					board.doMove(m);
				} catch(IllegalMoveException e) {
					throw new InvalidGameFileException("Illegal move: " + m);
				}
				String timestamp = getAttribute(node, "time");
				if(timestamp == null)
					timestamp = "";
				if(moveHistory == null)
					moveHistory = new MoveHistory<ChessMove>(m, timestamp);
				else
					moveHistory = moveHistory.addMove(m, timestamp);
			}
		}
		return moveHistory;
	}

	/**
	 * @return the Move represented by the given <move> XML node on
	 * the given board
	 */
	private ChessMove parseMove(Node node, ChessBoard board)
		throws InvalidGameFileException {
		Piece piece;
		ChessMove move;
		Player player = parsePlayer(getAttribute(node, "side"));
		String value = getAttribute(node, "value");
		move = parseMove(value, board);
		piece = move.getPiece();
		if(piece == null || piece.getPlayer() != player)
			throw new InvalidGameFileException("Move " + value + " moves invalid piece: " + piece);
		return move;
	}

	/**
	 * @return a ChessMove by parsing the string description of a move
	 * (e.g e2-e4) on the given board.
	 * @throws IllegalArgumentException if the move is ill-formed, or
	 * tries to move a piece that doesn't exist
	 */
	public static ChessMove parseMove(String value, ChessBoard board) {
		Piece piece;
		int coords[];
		if(!value.matches("[a-h][1-8]-[a-h][1-8]"))
			throw new IllegalArgumentException("Bad move:" + value);
		coords = parseCoords(value.substring(0,2));
		piece = board.getPieceAt(coords[0],coords[1]);
		coords = parseCoords(value.substring(3));
		return new ChessMove(piece, coords[0], coords[1],
							 board.getPieceAt(coords[0], coords[1]));
	}

	/**
	 * @effects Parse a &lt;gameOver&gt node and set the game over
	 * state on this.board if appropriate.
	 */
	private void parseGameOver(Node node)
		throws InvalidGameFileException {
		Player winner = parsePlayer(getAttribute(node, "winner"));
		String description = getAttribute(node, "description");
		int reason;
		if(description.equals("piecesLost"))
			reason = AntichessBoard.OUTOFPIECES;
		else if(description.equals("checkmate"))
			reason = ChessBoard.CHECKMATE;
		else if(description.equals("stalemate"))
			reason = ChessBoard.STALEMATE;
		else if(description.equals("timeExpired"))
			reason = Board.OUTOFTIME;
		else
			throw new InvalidGameFileException("Bad game over reason: " + description);
		board.endGame(reason, winner);
	}

	/**
	 * @return a list of the Pieces on the board, parsed from the
	 * given <pieces> XML element
	 */
	private static List<Piece> parsePieces(Node pieces)
		throws InvalidGameFileException {
		List<Piece> pieceList = new ArrayList<Piece>();
		NodeList children;
		children = pieces.getChildNodes();
		for(int i=0;i<children.getLength();i++) {
			Node node = children.item(i);
			if("square".equals(node.getNodeName())) {
				pieceList.add(parsePiece(node));
			}
			
		}
		return pieceList;
	}

	/**
	 * @return the Piece represented by a given XML <square> node
	 */
	private static Piece parsePiece(Node node)
		throws InvalidGameFileException {
		Player player = parsePlayer(getAttribute(node, "side"));
		int[] coords = parseCoords(getAttribute(node, "id"));
		PieceType type = parsePieceType(getAttribute(node, "piece"));
		return new Piece(player, type, coords[0], coords[1]);
	}

	/**
	 * @return the first child of <tt>node</tt> with type
	 * <tt>type</tt>, or <tt>null</tt> if none exists.
	 */
	private Node getChildOfType(Node node, int type) {
		NodeList children = node.getChildNodes();
		for(int i = 0;i < children.getLength(); i++) {
			Node child = children.item(i);
			if(child.getNodeType() == type)
				return child;
		}
		return null;
	}

	/**
	 * @return the value of the named attribute for <tt>node</tt>, or
	 * <tt>null</tt> if it does not exist.
	 */
	private static String getAttribute(Node node, String name) {
		NamedNodeMap map = node.getAttributes();
		if(map == null) return null;
		Node attr =  map.getNamedItem(name);
		if(attr == null) return null;
		return attr.getNodeValue();
	}

	/**
	 * @return the Player represented by the given string
	 * @throws InvalidGameFileException if the string is neither "white" nor "black"
	 */
	private static Player parsePlayer(String player)
		throws InvalidGameFileException {
		if(player.equals("white"))
			return Player.WHITE;
		if(player.equals("black"))
			return Player.BLACK;
		throw new InvalidGameFileException("Bad player specification: " + player);
	}

	/**
	 * @return a two-element array of [row, column] representing the
	 * coordinates specified in coords
	 * @requires coords matches "[a-g]\\d"
	 */
	private static int[] parseCoords(String coords) {
		int[] c = new int[2];
		c[1] = Character.codePointAt(coords, 0)
			- Character.codePointAt("a", 0);
		c[0] = Integer.parseInt(coords.substring(1)) - 1;
		return c;
	}

	/**
	 * @return the PieceType for the given named piece
	 */
	private static PieceType parsePieceType(String name)
		throws InvalidGameFileException {
		PieceType type = PieceTypeFactory.getNamedType(name);
		if(type == null)
			throw new InvalidGameFileException("No such piece type: " + name);
		return type;
	}

	/**
	 * @return this.board
	 */
	public ChessBoard getBoard() {
		return board;
	}

	/**
	 * @return this.timed
	 */
	public boolean isTimed() {
		return isTimed;
	}

	/**
	 * @return this.whiteTimer
	 */
	public GameTimer getWhiteTimer() {
		return whiteTimer;
	}
	/**
	 * @return this.blackTimer
	 */
	public GameTimer getBlackTimer() {
		return blackTimer;
	}

}
