package antichess;

import java.util.ArrayList;
import java.util.Timer;
import java.util.TimerTask;

/**
 * The GameTimer stores a time that ticks down when GameTimer
 * is running.  GameTimer also stores a list of GameTimerListeners
 * that it can notify of its time.  
 * 
 * 
 * @author nlharr
 * @specfield startTime     : long //
 * @specfield timeLeft   	: long //The number of millseconds left on the timer
 * @specfield listenerList	: sequence // This GameTimer's listeners
 * @specfield isRunning 	: boolean // Whether the clock is running
 * @specfield timeInterval  : long // the interval between updates on the clock 
 */

public class GameTimer extends TimerTask{
	//Fields
	private final long startTime;
	private long currentTime; 	//stores the time on the clock
	private long tempTime; 		//stores the the time of the previous call of the system time
	private boolean isRunning;
	private ArrayList<GameTimerListener> listenerList;
	private Timer updateTimer;
	//Abstraction Function
	//startTime = startTime
	//timeLeft = currentTime
	//isRunning = isRunning
	//listenerList = listenerList
	//timeInterval = the interval stored in the updateTimer
	
	
	//Representation Invariants
	//currentTime >= 0;
	//listenerList != null
	//none of the elements of listenerList are null
	//timeInterval > 0;
	
	/**
	 * Creates a GameTimer with a start time and current time of startTime
	 * @param startTime is the start time measured in milliseconds
	 * 		  interval is the interval between updates to the listeners of this timer
	 * 		  if interval is 0 the listeners are not updated
	 * @throws IllegalArgumentException if interval < 0 or startTimer < 0
	 */
	public GameTimer(long startTime, long interval){
		this(startTime, startTime, interval);
	}

	/**
	 * Creates a GameTimer with a start time of startTime, but a current time of currentTime
	 * @param startTime   : the start time measured in milliseconds
	 * @param currentTime : the remaining time in milliseconds 
	 * @param interval    : the time between updates if interval is 0 the listeners are not updated
	 * @throws IllegalArgumentException if interval < 0 or startTimer < 0 or currentTime < 0
	 */
	public GameTimer(long startTime, long currentTime, long interval) {
		if ((interval < 0) || (startTime < 0 ) || (currentTime < 0)) {
			throw new IllegalArgumentException("Interval is less than zero in GameTimer"+startTime+" "+interval);
		}
		this.startTime = startTime;
		this.currentTime = currentTime;
		this.tempTime = System.currentTimeMillis();
		this.isRunning = false;
		this.listenerList = new ArrayList<GameTimerListener>();
		if (interval>0){
			this.updateTimer = new Timer();
			updateTimer.scheduleAtFixedRate(this, 0, interval);
		}else{
			this.updateTimer = null;
		}
		checkRep();
	}
	
	//enforces representation invariants
	private void checkRep(){
		if (currentTime<0) throw new RuntimeException("currentTime in GameTimer is less than 0");
		if (listenerList == null) throw new IllegalArgumentException("listenerList in GameTimer is null");
		for (GameTimerListener lt : listenerList){
			if (lt == null){
				throw new RuntimeException("Listener is null");
			}
		}
	}
	
	
	/**
	 * Starts the timer
	 * @modifies isRunning
	 * @effects isRunning = true
	 */
	public void start(){
		if (!isRunning){
			tempTime = System.currentTimeMillis();
			isRunning = true;
		}
		
	}
	
	/**
	 * Stops the timer
	 * @modifies isRunning, timeLeft
	 * @effects isRunning = false, 
	 * 			if isRunning
	 * 				timeleft -=the interval since the first start() after the last stop()
	 * 			
	 */
	public void stop(){
		if (isRunning){
			isRunning = false;
			currentTime -= System.currentTimeMillis()-tempTime;
			if (currentTime<0) currentTime = 0;
		}
		notifyTimeUpdate();
		checkRep();
	}

	/**
	 * @return startTime
	 */
	public long getStartTime() {
		return startTime;
	}
	
	/**
	 * @return timeLeft
	 */
	public long getTime(){
		long tim;
		if (isRunning){
			tim = currentTime-(System.currentTimeMillis()-tempTime);
		}else{
			tim = currentTime;
		}		
		if (tim < 0){
			tim = 0;
		}
		return tim;
	}
	
	/**
	 * @requires listener != null
	 * @modifies listenerList
	 * @effects adds listener to listenerList
	 */
	synchronized public void addTimerListener(GameTimerListener listener){
		listenerList.add(listener);
		checkRep();
	}

	/**
	 * @return isRunning
	 */
	synchronized public boolean isRunning(){
		return isRunning;
	}
	
	/**
	 * Calls refreshTimer on all the listeners
	 */
	synchronized public void notifyTimeUpdate(){
		long timeTemp = getTime();
		for (GameTimerListener lt : listenerList){
			lt.refreshTimer(timeTemp);
		}
	}
	
	/**
	 * Refreshes the time for all the listeners
	 */
	synchronized public void run(){
		if(isRunning){
			notifyTimeUpdate();
		}
		
	}
	
	
	/**
	 * @modifies timeLeft
	 * @effects if (!isRunning) timeLeft = newTime
	 * @throws IllegalArgumentException if newTime < 0
	 * 		
	 */
	synchronized public void setTime(long newTime){
		if (newTime<0){
			throw new IllegalArgumentException("GameTimer: setTime newTimer is < 0");
		}else if(!isRunning){
			currentTime = newTime;
			notifyTimeUpdate();
		}
		checkRep();
	}
	
	
}
