package antichess;

import java.util.*;

/**
 * A linked-list implementation of the history of moves. MoveHistory's are immutable.
 * 
 */
public class MoveHistory<M extends Move>
{

	// FIELDS
	private M lastMove;
	private String lastMoveTime;
	private MoveHistory<M> history;
	
	public MoveHistory(M lastMove) {
		this(lastMove, "", null);
	}
	
	/**
	 * Creates a new MoveHistory with one Move
	 * @requires <code>lastMove</code> is a valid Move of type M
	 * @param lastMove
	 * @param timestamp
	 */
	public MoveHistory(M lastMove, String timestamp) {
		this(lastMove, timestamp, null);
	}
	
	private MoveHistory(M lastMove, String timestamp, MoveHistory<M> history) {
		this.lastMove = lastMove;
		this.lastMoveTime = timestamp;
		this.history = history;
		checkRep();
	}
	
	private void checkRep() {
		assert(lastMove == null || lastMove instanceof Move);
	}
	
	/**
	 * Adds a move to the MoveHistory. No timestamp is associated
	 * with the move.
	 * 
	 * @param move the new Move to add to the MoveHistory
	 * @return the new MoveHistory with <code>move</code> added to the end of the old MoveHistory
	 */
	public MoveHistory<M> addMove(M move) {
		return addMove(move, "");
	}
	
	/**
	 * Adds a move with the given timestamp to the MoveHistory.
	 * 
	 * @param move the new Move to add to the MoveHistory
	 * @param timestamp 
	 * @return the new MoveHistory with <code>move</code> added to the end of the old MoveHistory
	 */
	public MoveHistory<M> addMove(M move, String timestamp) {
		return new MoveHistory<M>(move, timestamp, this);
	}
	
	/**
	 * Removes the last move in the MoveHistory.
	 * 
	 * @return The MoveHistory that has the last move removed from the end of the old MoveHistory
	 */
	public MoveHistory<M> removeLastMove() {
		return history;
	}
	
	/**
	 * @return the last move in the MoveHistory
	 */
	public M getLastMove() {
		return lastMove;
	}
	
	/**
	 * @return the time the last move in the MoveHistory
	 * was made
	 */
	public String getLastMoveTime() {
		return lastMoveTime;
	}
	
	/**
	 * @return a list of the moves in the MoveHistory from the earliest Move to the latest Move
	 */
	public List<M> getHistory() {
		List<M> moves = new ArrayList<M>();
		
		for (MoveHistory<M> cur = this; cur != null; cur = cur.removeLastMove())
			moves.add(0, cur.getLastMove());
		
		return moves;
	}
	
	public List<String> getMoveTimes() {
		List<String> timestamps = new ArrayList<String>();
		
		for (MoveHistory<M> cur = this; cur != null; cur = cur.removeLastMove())
			timestamps.add(0, cur.getLastMoveTime());
		
		return timestamps;
	}
}
