// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess;

import antichess.movegen.*;

/**
 * PieceTypeFactory contains static methods for generating singleton
 * instances of PieceTypes for every type of Chess piece.
 **/
public class PieceTypeFactory
{
	/**
	 * Private constructor. PieceTypeFactory cannot be instantiated.
	 **/
	private PieceTypeFactory() {}

	/**
	 * ChessPieceType is a concrete implementation of PieceType for
	 * chess pieces
	 **/
	private static class ChessPieceType implements PieceType {
		MoveGenerator moveGenerator;
		String name;
		ChessPieceType(MoveGenerator g, String name) {
			moveGenerator = g;
			this.name = name;
		}

		public MoveGenerator getMovement() {
			return moveGenerator;
		}

		public String getName() {
			return name;
		}

		/* Because ChessPieceType objects are singletons, pointer
		 * equality suffices as an equality check.
		 */
		public boolean equals(Object o) {
			return super.equals(o);
		}
	}

	/**
	 * The PieceType representing a pawn
	 **/
	private static PieceType pawn;

	/**
	 * The PieceType representing a rook
	 **/
	private static PieceType rook;

	/**
	 * The PieceType representing a knight
	 **/
	private static PieceType knight;

	/**
	 * The PieceType representing a bishop
	 **/
	private static PieceType bishop;

	/**
	 * The PieceType representing a queen
	 **/
	private static PieceType queen;

	/**
	 * The PieceType representing a king
	 **/
	private static PieceType king;

	/**
	 * @return a PieceType with name == "Pawn" representing the type
	 * of a pawn
	 **/
	public static PieceType getPawnType() {
		return pawn;
	}

	/**
	 * @return a PieceType with name == "Rook" representing the type
	 * of a rook
	 **/
	public static PieceType getRookType() {
		return rook;
	}

	/**
	 * @return a PieceType with name == "Knight" representing the type
	 * of a knight
	 **/
	public static PieceType getKnightType() {
		return knight;
	}

	/**
	 * @return a PieceType with name == "Bishop" representing the type
	 * of a bishop
	 **/
	public static PieceType getBishopType() {
		return bishop;
	}

	/**
	 * @return a PieceType with name == "Queen" representing the type
	 * of a queen
	 **/
	public static PieceType getQueenType() {
		return queen;
	}

	/**
	 * @return a PieceType with name == "King" representing the type
	 * of a king
	 **/
	public static PieceType getKingType() {
		return king;
	}

	/**
	 * @return a PieceType with the given name (compared
	 * case-insensitively), or <code>null</code> if none exists
	 */
	public static PieceType getNamedType(String name) {
		name = name.toLowerCase();
		if(name.equals("pawn"))
			return getPawnType();
		if(name.equals("rook"))
			return getRookType();
		if(name.equals("knight"))
			return getKnightType();
		if(name.equals("bishop"))
			return getBishopType();
		if(name.equals("queen"))
			return getQueenType();
		if(name.equals("king"))
			return getKingType();
		return null;
	}

	static {
		pawn = new ChessPieceType(new PawnMoveGenerator(), "Pawn");
		rook = new ChessPieceType(new LinearMoveGenerator(false, true), "Rook");
		knight = new ChessPieceType(new JumpMoveGenerator(new int[]{1,-1, 1,-1, 2,-2, 2,-2},
														  new int[]{2, 2,-2,-2, 1, 1,-1,-1}),
									"Knight");
		bishop = new ChessPieceType(new LinearMoveGenerator(true, false), "Bishop");
		queen = new ChessPieceType(new LinearMoveGenerator(true, true), "Queen");
		king = new ChessPieceType(new JumpMoveGenerator(new int[]{ 1, 1, 1, 0, 0,-1,-1,-1},
														new int[]{ 1, 0,-1, 1,-1, 1, 0,-1}),
								  "King");
	}
}
