// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess.movegen;

import antichess.*;
import java.util.*;

/**
 * A JumpMoveGenerator is a MoveGenerator for pieces that ``jump'' in
 * some pattern, regardless of whether there are pieces in the way.
 * For example, knights move with a Jump movement type.
 *
 * @specfield directions : list  // A list of (delta row, delta column) pairs
 **/
public class JumpMoveGenerator extends ChessMoveGenerator
{
    /* AF: directions[i] = (deltaRow[i],deltaColumn[i])
     */
    private final int[] deltaRow;
    private final int[] deltaColumn;

    /**
     * @effects Creates a JumpMoveGenerator with a set of moves
     * specified as a pair of lists of the change in rows and change
     * in columns for each direction of moves.
     * @requires dr.length == dc.length
     **/
    public JumpMoveGenerator(int[] dr, int[] dc) {
        deltaRow = new int[dr.length];
        deltaColumn = new int[dc.length];
        System.arraycopy(dr, 0, deltaRow, 0, dr.length);
        System.arraycopy(dc, 0, deltaColumn, 0, dc.length);
    }

    public List<ChessMove> getMoves(Piece piece, Board board) {
        List<ChessMove> moves = new ArrayList<ChessMove>();
        int dr, dc;
        int row, col;
        Piece otherPiece;
        for(int i=0;i<deltaRow.length;i++) {
            dr = deltaRow[i];
            dc = deltaColumn[i];
            row = piece.getRow() + dr;
            col = piece.getColumn() + dc;
            if(row < 0 || row >= board.getRows()
               || col < 0 || col >= board.getColumns())
                continue;
            otherPiece = board.getPieceAt(row, col);
            if(otherPiece == null || otherPiece.getPlayer() != piece.getPlayer())
                moves.add(new ChessMove(piece, row, col, otherPiece));
        }
        return moves;
    }

    public boolean isChessMoveLegal(ChessMove cm, Board board) {
        int dr, dc;
        int row, col;
        Piece piece = cm.getPiece();
        boolean ok = false;
        for(int i=0;i<deltaRow.length;i++) {
            dr = deltaRow[i];
            dc = deltaColumn[i];
            row = piece.getRow() + dr;
            col = piece.getColumn() + dc;
            if(row == cm.getRow() && col == cm.getColumn()) {
                ok = true;
                break;
            }
        }
        return ok;
    }

}
