package antichess.test;

import antichess.*;
import java.util.*;
import junit.framework.TestCase;

public class MoveHistoryTests extends TestCase
{
	private Piece whitePiece, blackPiece;
	
	public MoveHistoryTests(String name) {
		super(name);
		
		PieceType pieceType = new NullPieceType();
		whitePiece = new Piece(Player.WHITE, pieceType, 0, 0);
		blackPiece = new Piece(Player.BLACK, pieceType, 7, 5);
	}
	
	public void testConstructor() {
		checkConstructor(new ChessMove(whitePiece, 1, 1));
		checkConstructor(new ChessMove(blackPiece, 0, 0, whitePiece));
		checkConstructor(new ChessMove(whitePiece, -1, -1));
		checkConstructor(new ChessMove(blackPiece, 0, 5));
		
		checkConstructor(new ChessMove(whitePiece, 1, 1), "123456");
		checkConstructor(new ChessMove(blackPiece, 0, 0, whitePiece), "987654321");
		checkConstructor(new ChessMove(whitePiece, -1, -1), "11111111");
		checkConstructor(new ChessMove(blackPiece, 0, 5), "140000000000000000");
	}
	
	public void testAddMove() {
		ChessMove cm = new ChessMove(whitePiece, 1, 1);
		MoveHistory<ChessMove> mh = new MoveHistory<ChessMove>(cm);
		List<ChessMove> expected = new ArrayList<ChessMove>();
		expected.add(cm);
		List<String> expectedTimes = new ArrayList<String>();
		expectedTimes.add("");
		
		cm = new ChessMove(blackPiece, 6, 3);
		mh = mh.addMove(cm);
		expected.add(cm);
		expectedTimes.add("");
		checkMoveHistory(mh, expected, expectedTimes);
		
		cm = new ChessMove(whitePiece.moveTo(1, 1), 5, 4);
		mh = mh.addMove(cm, "1516171819");
		expected.add(cm);
		expectedTimes.add("1516171819");
		checkMoveHistory(mh, expected, expectedTimes);
		
		cm = null;
		mh = mh.addMove(null, "12");
		expected.add(null);
		expectedTimes.add("12");
		checkMoveHistory(mh, expected, expectedTimes);
		
		cm = null;
		mh = mh.addMove(null);
		expected.add(null);
		expectedTimes.add("");
		checkMoveHistory(mh, expected, expectedTimes);
		
		cm = new ChessMove(blackPiece.moveTo(6, 3), 5, 4, whitePiece.moveTo(1, 1).moveTo(5, 4));
		mh = mh.addMove(cm, "");
		expected.add(cm);
		expectedTimes.add("");
		checkMoveHistory(mh, expected, expectedTimes);
		
	}
	
	public void testRemoveLastMove() {
		ChessMove cm = new ChessMove(whitePiece, 1, 1);
		MoveHistory<ChessMove> mh = new MoveHistory<ChessMove>(cm);
		List<ChessMove> expected = new ArrayList<ChessMove>();
		expected.add(cm);
		List<String> expectedTimes = new ArrayList<String>();
		expectedTimes.add("");
		
		cm = new ChessMove(blackPiece, 6, 3);
		mh = mh.addMove(cm);
		expected.add(cm);
		expectedTimes.add("");
		
		cm = new ChessMove(whitePiece.moveTo(1, 1), 5, 4);
		mh = mh.addMove(cm, "");
		expected.add(cm);
		expectedTimes.add("");
		
		cm = null;
		mh = mh.addMove(null, "1.60254309");
		expected.add(null);
		expectedTimes.add("1.60254309");
		
		cm = new ChessMove(blackPiece.moveTo(6, 3), 5, 4, whitePiece.moveTo(1, 1).moveTo(5, 4));
		mh = mh.addMove(cm);
		expected.add(cm);
		expectedTimes.add("");
		checkMoveHistory(mh, expected, expectedTimes);
		
		mh = mh.removeLastMove();
		expected.remove(expected.size()-1);
		expectedTimes.remove(expectedTimes.size()-1);
		checkMoveHistory(mh, expected, expectedTimes);
		
		mh = mh.removeLastMove();
		expected.remove(expected.size()-1);
		expectedTimes.remove(expectedTimes.size()-1);
		checkMoveHistory(mh, expected, expectedTimes);
		
		mh = mh.removeLastMove();
		expected.remove(expected.size()-1);
		expectedTimes.remove(expectedTimes.size()-1);
		checkMoveHistory(mh, expected, expectedTimes);
		
		mh = mh.removeLastMove();
		expected.remove(expected.size()-1);
		expectedTimes.remove(expectedTimes.size()-1);
		checkMoveHistory(mh, expected, expectedTimes);
	}
	
	public void testImmutable() {
		ChessMove cm = new ChessMove(whitePiece, 1, 1);
		MoveHistory<ChessMove> mh = new MoveHistory<ChessMove>(cm);
		List<ChessMove> expected = new ArrayList<ChessMove>();
		expected.add(cm);
		List<String> expectedTimes = new ArrayList<String>();
		expectedTimes.add("");
		
		cm = new ChessMove(blackPiece, 6, 3);
		mh.addMove(cm);
		checkMoveHistory(mh, expected, expectedTimes);
		
		mh = mh.addMove(cm);
		expected.add(cm);
		expectedTimes.add("");
		cm = new ChessMove(whitePiece.moveTo(1, 1), 6, 3, blackPiece.moveTo(6, 3));
		mh.addMove(cm, "12133.1414");
		checkMoveHistory(mh, expected, expectedTimes);
		
		mh = mh.addMove(cm, "12133.1414");
		expected.add(cm);
		expectedTimes.add("12133.1414");
		mh.addMove(new ChessMove(blackPiece.moveTo(6, 3), -1, -1));
		checkMoveHistory(mh, expected, expectedTimes);
	}
	
	private void checkMoveHistory(MoveHistory<ChessMove> mh, List<ChessMove> expected, List<String> expectedTimes) {
		assertEquals("MoveHistory does not contain the expected lastMove",
				expected.get(expected.size()-1), mh.getLastMove());
		assertEquals("MoveHistory does not contain the expected lastMoveTime",
				expectedTimes.get(expectedTimes.size()-1), mh.getLastMoveTime());
		
		assertTrue("MoveHistory does not contain the same number of timestamps as moves",
				mh.getHistory().size() == mh.getMoveTimes().size());
		
		List<ChessMove> history = mh.getHistory();
		List<String> timestamps = mh.getMoveTimes();
		
		assertEquals("MoveHistory does not contain the expected history",
				expected, history);
		assertEquals("MoveHistory does not contain the expected timestamps",
				expectedTimes, timestamps);
	}
	
	private void checkConstructor(ChessMove initialMove, String timestamp) {
		MoveHistory<ChessMove> mh = null;
		try {
			mh = new MoveHistory<ChessMove>(initialMove, timestamp);
		} catch (Exception e) {
			fail("Threw exception rather than taking valid initial Move.");
			return;
		}
		List<ChessMove> expected = new ArrayList<ChessMove>();
		expected.add(initialMove);
		List<String> expectedTimes = new ArrayList<String>();
		expectedTimes.add(timestamp);
		
		checkMoveHistory(mh, expected, expectedTimes);
	}
	
	private void checkConstructor(ChessMove initialMove) {
		MoveHistory<ChessMove> mh = null;
		try {
			mh = new MoveHistory<ChessMove>(initialMove);
		} catch (Exception e) {
			fail("Threw exception rather than taking valid initial Move.");
			return;
		}
		List<ChessMove> expected = new ArrayList<ChessMove>();
		expected.add(initialMove);
		List<String> expectedTimes = new ArrayList<String>();
		expectedTimes.add("");
		
		checkMoveHistory(mh, expected, expectedTimes);
	}
}
