
package antichess.viewgui;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.File;
import java.util.ArrayList;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.KeyStroke;
import javax.swing.border.EtchedBorder;

import antichess.BoardObserver;
import antichess.ChessBoard;
import antichess.ControllerMaster;
import antichess.GameClock;
import antichess.Player;

/**
 * ChessGUI is a GUI for displaying a game of Chess, Antichess, or any
 * other chess game that uses a ChessBoard.
 * @author nlharr
 *
 * @specfield clock				//GameClock for the game being displayed by the GUI
 * @specfield boardView			//ChessBoardView that is displaying the current game
 * @specfield master 			//ControllerMaster that is running the current game
 * @specfield board				//ChessBoard that this ChessGUI is displaying 
 */

public class ChessGUI implements BoardObserver, ActionListener, Runnable {
	//Fields
	private GameClock clock;
	private ChessBoardView view;
	private JFrame mainFrame;
	private ControllerMaster master;
	private JFileChooser fileChooser;
	private ChessBoard currentBoard;
	
	//A boolean that stores whether we have displayed a game over message for this
	//board
	private boolean hasNotifiedGameOver;
	
	//AbstractionFunction
	//clock = clock
	//boardView = view
	//master = master
	//board = currentBoard
	
	//Representation Invariants
	//clock != null
	//view != null
	//master != null
	//currentBoard != null 
	
	private void checkRep(){
		if (clock == null) throw new RuntimeException("clock is null");
		if (view == null) throw new RuntimeException("view is null");
		if (master == null) throw new RuntimeException("master is null");
		if (currentBoard == null) throw new RuntimeException("currentBoard is null");
	}
	
	
	//private constructor for use with the factory method
	private ChessGUI(ChessBoardView view, GameClock clock, 
			ControllerMaster master, ChessBoard board){
		this.view = view;
		this.clock = clock;
		this.master = master;
		this.fileChooser = new JFileChooser();
		this.currentBoard = board;
		this.hasNotifiedGameOver = false;
		currentBoard.addObserver(this);
		checkRep();
	}
	
	
	/**
	 * Closes the GUI
	 *
	 */
	public void destroy(){
		mainFrame.dispose();
	}
	
	/**
	 * @return boardView
	 */
	public BoardView getBoardView(){
		return view;
	}
	
	/**
	 * @modifies board, clock
	 * @effects board = newBoard, clock = newClock
	 */
	public void switchViewedGame(ChessBoard newBoard, GameClock clock){
		this.clock = clock;
		
		currentBoard.removeObserver(this);
		this.view.switchViewedBoard(newBoard);
		
		hasNotifiedGameOver = false;
		
		currentBoard = newBoard;
		newBoard.addObserver(this);
		
		mainFrame.setContentPane(this.getContentPane());
		mainFrame.setVisible(true);
		
		checkRep();
	}
	
	
	/**
	 * ChessGUI is a BoardObserver so this method is called whenever
	 * the currentBoard deems it necesary.  This method handles switching the
	 * text color for the White and Black labels, and displaying a dialogue for 
	 * the end of the game.
	 */
	
	public void postRefresh(){
		//Code for bringing up a dialogue if the game has ended
		if (currentBoard.isGameOver()){
			if (!hasNotifiedGameOver){
				if (currentBoard.getWinner().equals(Player.NONE)){
					JOptionPane.showMessageDialog(mainFrame,"Game Over: Stalemate");
	                
	            }else{
	            	JOptionPane.showMessageDialog(mainFrame, "Game Over: "+currentBoard.getWinner().toString().substring(0, 1)+
	            			currentBoard.getWinner().toString().toLowerCase().substring(1)+" Wins");
	            }
				//keeps track of whether we have opened a dialogue box before
				hasNotifiedGameOver = true;
			}
		}
		
		if (currentBoard.getPlayer().equals(Player.WHITE)){
			whiteLabel.setText("<html><font size=+2 color=blue>White</font>");
			blackLabel.setText("<html><font size=+2>Black</font>");
		}else{
			whiteLabel.setText("<html><font size=+2>White</font>");
			blackLabel.setText("<html><font size=+2 color=blue>Black</font>");
		}
	}
	
	
	/**
	 * This generates the menu bar used by the Chess GUI
	 * 
	 * @return the menu bar for the ChessGUI
	 */
	public JMenuBar getMenu(){
            JMenuBar menuBar;
            JMenu menu;
            JMenuItem menuItem;

            //Create the menu bar.
            menuBar = new JMenuBar();

            //adds the option of loading a Database
            menu = new JMenu("Options");
            menuBar.add(menu);

            //adds the New Game option
            menuItem = new JMenuItem("New Game",
                                     KeyEvent.VK_N);
            menuItem.setAccelerator(KeyStroke.getKeyStroke(
                    KeyEvent.VK_N, ActionEvent.CTRL_MASK));
            
            menuItem.getAccessibleContext().setAccessibleDescription(
                    "Brings up dialogue for a new game");
            menuItem.setActionCommand("NewGame");
            menuItem.addActionListener(this);
            menu.add(menuItem);

            //adds the save game option
            menuItem = new JMenuItem("Save Game",
                                     KeyEvent.VK_S);
            menuItem.setAccelerator(KeyStroke.getKeyStroke(
                    KeyEvent.VK_S, ActionEvent.CTRL_MASK));
            
            menuItem.getAccessibleContext().setAccessibleDescription(
                    "Brings up dialogue for saving the game");
            menuItem.setActionCommand("SaveGame");
            menuItem.addActionListener(this);
            menu.add(menuItem);

            //adds the load game option
            menuItem = new JMenuItem("Load Game",
                                     KeyEvent.VK_L);
            menuItem.setAccelerator(KeyStroke.getKeyStroke(
                    KeyEvent.VK_L, ActionEvent.CTRL_MASK));
            
            menuItem.getAccessibleContext().setAccessibleDescription(
                    "Brings up dialogue for loading the game");
            menuItem.setActionCommand("LoadGame");
            menuItem.addActionListener(this);
            menu.add(menuItem);
            
            
            //adds the Quit option
            menuItem = new JMenuItem("Exit",
                KeyEvent.VK_Q);
            menuItem.setAccelerator(KeyStroke.getKeyStroke(
                    KeyEvent.VK_Q, ActionEvent.CTRL_MASK));
            menuItem.getAccessibleContext().setAccessibleDescription(
                    "Exits the program");
            menuItem.addActionListener(this);
            menu.add(menuItem);
            return menuBar;
	}
	
	//These JLabels are outside the method because their text value
	//is modified in the postRefresh() method
	JLabel whiteLabel;
	JLabel blackLabel;

	/**
	 * @return the content plane for the GUI
	 */
	public Container getContentPane(){
		TimerLabel whiteTimer = new TimerLabel(clock, Player.WHITE);
		whiteLabel = new JLabel();
		TimerLabel blackTimer = new TimerLabel(clock, Player.BLACK);
		blackLabel = new JLabel();
		
		//assigns the proper text to the label
		if (currentBoard.getPlayer().equals(Player.WHITE)){
			whiteLabel.setText("<html><font size=+2 color=blue>White</font>");
			blackLabel.setText("<html><font size=+2>Black</font>");
		}else{
			whiteLabel.setText("<html><font size=+2>White</font>");
			blackLabel.setText("<html><font size=+2 color=blue>Black</font>");
		}
		
		
		JPanel whitePanel = new JPanel(new GridBagLayout());
		JPanel blackPanel = new JPanel(new GridBagLayout());
		GridBagConstraints c = new GridBagConstraints();
		c.weighty = .5;
		c.ipady = 6;
		c.ipadx = 20;
		c.gridwidth = GridBagConstraints.REMAINDER;
		whitePanel.add(whiteLabel, c);
		blackPanel.add(blackLabel, c);
		c.gridy = 1;
		whitePanel.add(whiteTimer, c);
		blackPanel.add(blackTimer, c);
		whitePanel.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.RAISED));
		blackPanel.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.RAISED));
		
		
		JLabel historyLabel = new JLabel("History");
		MoveHistoryView history = new MoveHistoryView(currentBoard, 10, 16);
		
		JScrollPane scrollPane = 
		    new JScrollPane(history,
		                    JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
		                    JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
		
		JPanel historyPanel = new JPanel();
		historyPanel.setLayout(new BoxLayout(historyPanel, BoxLayout.Y_AXIS));
		historyPanel.add(historyLabel);
		historyPanel.add(scrollPane);
		
		JPanel pan = new JPanel(new GridBagLayout());
		
		c = new GridBagConstraints();
		
		
	    c.gridwidth = 10;
	    c.gridheight = 3;
	    c.weightx = .5;
	    c.weighty = .5;
	    c.gridx = 0;
	    c.gridy = 0;
	    pan.add(view, c);
		
	    c.weightx = .5;
	    c.weighty = .1;
	    c.gridwidth = GridBagConstraints.REMAINDER;
	    c.gridheight = 1;
	    c.gridx = 11;
	    c.gridy = 0;
	    pan.add(whitePanel, c);
	    c.gridheight = 1;
	    c.gridy = 1;
	    pan.add(blackPanel, c);
	    c.gridy = 2;
	    pan.add(historyPanel, c);
		return pan;
	}
	
	
	/**
	 * Handles these events for the ChessGUI:
	 * Exiting
	 * Saving
	 * Loading
	 * Starting a new game
	 * 
	 */

	public void actionPerformed(ActionEvent e) {
		if (e.getActionCommand().equals("Exit")){
			System.exit(0);
		}else if (e.getActionCommand().equals("NewGame")){
			NewGameWindow.createNewGameWindow(master);
		}else if (e.getActionCommand().equals("SaveGame")){
			int returnVal = fileChooser.showSaveDialog(mainFrame);
			if (returnVal == JFileChooser.APPROVE_OPTION) {
				File file = fileChooser.getSelectedFile();
				try{
					master.save(file);
				}catch(Exception ex){
					JOptionPane.showMessageDialog(mainFrame, "Failed to Save Game");
				}
			}
		}else if (e.getActionCommand().equals("LoadGame")){
			int returnVal = fileChooser.showOpenDialog(mainFrame);

			if (returnVal == JFileChooser.APPROVE_OPTION) {
				File file = fileChooser.getSelectedFile();
				try{
					Object[] options = {"White",
                    "Black"};
					int n = JOptionPane.showOptionDialog(mainFrame,
						    "What player would you like to be?",
						    "Player Selection",
						    JOptionPane.YES_NO_OPTION,
						    JOptionPane.QUESTION_MESSAGE,
						    null,     //don't use a custom Icon
						    options,  //the titles of buttons
						    options[0]); //default button title
					if (n == JOptionPane.YES_OPTION){
						System.err.println("Player is White");
						master.load(file, Player.WHITE);
					}else if (n == JOptionPane.NO_OPTION){
						System.err.println("Player is Black");
						master.load(file, Player.BLACK);
					}
					
					
				}catch(Exception ex){
					JOptionPane.showMessageDialog(mainFrame, "Failed to Load Game "+ ex.getMessage());
				}
			}
		}
	 
	}
	
	 
	 /**
	  * Creates a ChessGUI suitable for Chess or Antichess.  
	  * 
	  * For the ChessGUI returned:
	  * board = @param board
	  * clock = @param gameClock
	  * master = @param master
	  * boardView = a ChessBoardView generated with the board.
	  * @return a ChessGUI with the specfields above.
	  */
	
	 public static ChessGUI createGUI(ChessBoard board, GameClock gameClock, ControllerMaster master){
//		initializes the BoardView
			ArrayList<String> chessNameList = new ArrayList<String>();
			chessNameList.add("RAntiking");
			chessNameList.add("RBishop");
			chessNameList.add("RKing");
			chessNameList.add("RKnight");
			chessNameList.add("RPawn");
			chessNameList.add("RQueen");
			chessNameList.add("RRook");
			chessNameList.add("WAntiking");
			chessNameList.add("WBishop");
			chessNameList.add("WKing");
			chessNameList.add("WKnight");
			chessNameList.add("WPawn");
			chessNameList.add("WQueen");
			chessNameList.add("WRook");
			chessNameList.add("board");
			
			ImageMap chessMap = new ImageMap("images");
			chessMap.loadImageList(chessNameList);
			ChessBoardView boardView = new ChessBoardView(board, chessMap, 50, 50, 400, 400);
			
		 	
		 	
		 	//Create and set up the content pane and menu bar
		 	ChessGUI gui = new ChessGUI(boardView, gameClock, master, board);
	        return gui;
	 }
	 
	/**
     * Shows this GUI.
     */
    public void showGUI() {
//    	Create and set up the window
        JFrame.setDefaultLookAndFeelDecorated(true);
        mainFrame = new JFrame("AntiChess");
        mainFrame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        mainFrame.setContentPane(this.getContentPane());
        mainFrame.setJMenuBar(this.getMenu());
        //Display the window
        mainFrame.setSize(700, 600);
        mainFrame.setResizable(false);
        mainFrame.setVisible(true);
    }
	
    /**
     * Calls showGUI().  Used to start a thread for the GUI
     */
	public void run(){
		this.showGUI();
	}
}