#!/usr/bin/perl
use strict;
use warnings;

use FLAP;
use AIMUtils;

use TTTGame;
use TTT;

use POE;
use POE::Component::Client::TCP;
use POE::Filter::Stream;

my $host="toc.oscar.aol.com";
my $auth_host="login.oscar.aol.com";
my $port=5190;
my $handle=AIMUtils::normalizeSN("AdiumTTT");
my $password=undef;

die("You must set a password in TOC.pl") unless defined($password);

use constant WANT_FLAP_SIGNON	=>	0;
use constant WANT_SIGNON		=>	1;
use constant WANT_CONFIG		=>	2;
use constant CONNECTED			=>	3;

POE::Component::Client::TCP->new(
	RemoteAddress	=>	$host,
	RemotePort		=>	$port,
	Filter			=>	"POE::Filter::Stream",

	Connected		=> 	\&connected,
	ConnectError	=>	\&connect_error,
	Disconnected	=>	\&disconnected,

	ServerInput		=>	\&input,

	Alias			=>	"TOC",

	InlineStates	=>
		{
			"toc_packet_in"	=>	\&toc_packet_in,
			"ttt_message"	=>	\&ttt_message,
			"game_over"		=>	\&game_over,
			"send_im"		=>	\&send_im
		}
);

$poe_kernel->run();

sub connected
{
	my ($kernel,$heap) = @_[KERNEL, HEAP];
	$heap->{FLAP} = FLAP->new;
	$heap->{GAMES} = {};
	$heap->{STATE} = WANT_FLAP_SIGNON;
	$heap->{server}->put("FLAPON\r\n\r\n");
	print "Sent FLAPON\n";
}

sub connect_error
{
	die("Unable to connect to $host:$port");
}

sub disconnected
{
}

sub input
{
	my ($kernel,$heap,$input) = @_[KERNEL, HEAP, ARG0];
	$heap->{FLAP}->addData($input);
	my $packet;
	while($packet = $heap->{FLAP}->nextPacket)
	{
		$kernel->yield("toc_packet_in",$packet);
	}
}

sub toc_packet_in
{
	my ($kernel,$heap,$packet) = @_[KERNEL, HEAP, ARG0];
	if($packet->{FRAMETYPE} == $FLAP::FT_SIGNON)
	{
		die "Got unexpected FLAP signon" unless $heap->{STATE} == WANT_FLAP_SIGNON;
		my $flap_version = unpack("N",$packet->{DATA});
		die "Bad FLAP version: $flap_version" unless $flap_version == 1;
		$heap->{server}->put($heap->{FLAP}->signonPacket($handle));
		$heap->{server}->put($heap->{FLAP}->dataPacket(
				toc_command("toc_signon",$auth_host,$port,$handle,roast($password),
								"english", "TOC.pl v0.01")));

		print "Sent signon" . "\n";
		$heap->{STATE} = WANT_SIGNON;
	}
	elsif($packet->{FRAMETYPE} == $FLAP::FT_KEEP_ALIVE)
	{
		print "Got keep-alive.\n";
	}
	else
	{
		die "Bad frametype" if($packet->{FRAMETYPE} != $FLAP::FT_DATA);
		my ($cmd,$args) = split /:/, $packet->{DATA},2;
		if($cmd eq "SIGN_ON")
		{
			die("Unexpected SIGN_ON") if($heap->{STATE} != WANT_SIGNON);
			$heap->{STATE} = WANT_CONFIG;
			print "Got SIGN_ON: ", $args, "\n";
		}
		elsif($cmd eq "CONFIG")
		{
			print "Got configuration.\n";
			#Add ourself to our list, since AIM won't log us on unless we add
			#at least one buddy
			$heap->{server}->put($heap->{FLAP}->dataPacket("toc_add_buddy $handle"));
			$heap->{server}->put($heap->{FLAP}->dataPacket("toc_init_done"));
			$heap->{STATE} = CONNECTED;
		}
		elsif($cmd eq "IM_IN")
		{
			my ($from,$auto,$msg) = split /:/, $args,3;
			$msg =~ s/<.*?>//g;			#strip HTML
			$msg =~ s/^\s+//g;			#and trim whitespace
			$msg =~ s/\s+$//g;
			if($msg =~ m{\[TTT/(.*)]:(.*)})
			{
				my ($type,$game_msg) = ($1,$2);
				$kernel->yield("ttt_message",$from,$type,$game_msg);
			}
		}
		else
		{
			print "Got command: ", $packet->{DATA}, "\n";
		}
	}

}

sub ttt_message
{
	my ($kernel,$heap,$from,$type,$msg) = @_[KERNEL,HEAP,ARG0,ARG1,ARG2];
	$from = AIMUtils::normalizeSN($from);
	if($type eq "Invite")
	{
		return if $heap->{GAMES}->{$from};
		my $game = TTTGame->new($from,$msg=~/^X/?$TTT::X:$TTT::O);
		$heap->{GAMES}->{$from} = $game;
	}
	else
	{
		return unless $heap->{GAMES}->{$from};
		$kernel->post($from,"ttt_message",$type,$msg);
	}
}

sub game_over
{
	my ($kernel,$heap,$who) = @_[KERNEL,HEAP,ARG0];
	delete $heap->{GAMES}->{$who};
}

sub send_im
{
	my ($heap,$to,$msg,$auto) = @_[HEAP, ARG0 .. $#_];
	$to = AIMUtils::normalizeSN($to);
	$heap->{server}->put($heap->{FLAP}->dataPacket(
							 toc_command("toc_send_im",$to,$msg,$auto?"auto":"")));
}

sub roast
{
	my $password = shift;
	my $roastString = substr("Tic/Toc" x 10,0,length $password);
	$password ^= $roastString;
	return "0x" . unpack("H*",$password);
}

sub encodeArg
{
	my $arg = shift;
	#Escape various characters
	$arg =~ s/([\[\]{}()\\\$'"])/\\$1/g;
	$arg = '"' . $arg . '"' if $arg =~ / /;
	return $arg;
}

sub toc_command
{
	my $cmd = shift;
	my $arg;
	foreach $arg (@_)
	{
		$cmd .= " " . encodeArg($arg);
	}
	print "COMMAND: $cmd\n";
	return $cmd;
}
