/*****************************************************************************
 *
 * Microchip DeviceNet Stack (Bit-Strobed Messaging Connection Object Source)
 *
 *****************************************************************************
 * FileName:        conn3.c
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.20.00 or higher
 * Linker:          MPLINK 03.40.00 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *
 * This file contains Bit-Strobe messaging support for the Connection Object 
 * described in Section 5-4 and Chapter 7 of Volume 1 of the DeviceNet 
 * specification. Additional support is in the UsrConn.c file.
 * 
 *
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			04/28/03	...	
 * 
 *****************************************************************************/

#include	"dnet.def"			// Global definitions file

#include 	"typedefs.h"

#if SUPPORT_BIT_STROBED

#include	"conn.h"			// Connection prototypes and symbols

#include	"services.h"		// Service codes
#include	"errors.h"			// Error codes
#include	"class.h"			// Class codes

#include	"route.h"			// Global symbols defined by the router
#include	"dnet.h"			// DeviceNet prototypes and symbols
#include	"CAN.h"



#pragma	udata
/*********************************************************************
 * Connection related variables
 ********************************************************************/
CONN_STROBE 	uConn3;



/*********************************************************************
 * Function:        unsigned char _Conn3Create(void)
 *
 * PreCondition:    The CAN (or other) driver must be prepared to 
 *					accept some filter settings.
 *
 * Input:       	none	
 *                  
 * Output:      	unsigned char handle to the connection	
 *
 * Side Effects:    none
 *
 * Overview:        This function creates a connection 
 *					in the predefined set and returns a 
 *					handle to the connection. The connection is 
 *					placed into a 'configuring' state. 
 *					
 *					This function will call out to the users code
 *					for additional resource allocation. If the 
 *					user returns NULL then the allocation will not
 *					occur.
 *
 * Note:            This function is not called directly by 
 *					application code.
 ********************************************************************/
unsigned char _Conn3Create()
{
	if (UsrConnCreateEvent(3))
	{
		//Initialize the connection attributes
		uConn3.attrib.state = _STATE_CONFIGURING;
		uConn3.attrib.transportClass.byte = 0x83;
		uConn3.attrib.produced_cid.bytes.LSB = (uDNet.MACID >> 3) | (0xE << 3);
		uConn3.attrib.produced_cid.bytes.MSB = uDNet.MACID << 5;
		uConn3.attrib.consumed_cid.bytes.LSB = uDNet.AllocInfo.MasterMACID | 0x80; 
		uConn3.attrib.consumed_cid.bytes.MSB = 0x00;
		uConn3.attrib.expected_packet_rate.word = 0;
		
		_existentFlags.bits.strobe = 1;
		
		// Setup the pointer and other info for the receiving side
		uConn3.rx.lenMax = uConn3.attrib.consumed_con_size.bytes.LSB;
						
		// Setup the pointer and other info for the transmitting side
		uConn3.tx.lenMax = uConn3.attrib.produced_con_size.bytes.LSB;
						
		//Issue a request to start receiving the CID
		CANSetFilter(uConn3.attrib.consumed_cid.word);
		return (3);
	}
	else return (0);
}


/*********************************************************************
 * Function:        unsigned char _Conn3Close()
 *
 * PreCondition:    The connection should have already been open.
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char instance that closed
 *
 * Side Effects:    none
 *
 * Overview:        Closes the specified connection. Users code is 
 *					notified to release any used resources.
 *
 * Note:            none
 ********************************************************************/
unsigned char _Conn3Close()
{	
	// Transition to the non-existent state
	uConn3.attrib.state = _STATE_NON_EXISTENT;
	
	_establishFlags.bits.strobe = 0;
	_existentFlags.bits.strobe = 0;
	
	// Issue a request to the driver to stop receiving the message
	CANClrFilter(uConn3.attrib.consumed_cid.word);
	
	UsrConnCloseEvent(3);

	return(1);
}



/*********************************************************************
 * Function:        void _Conn3TimerEvent(void)
 *
 * PreCondition:    None
 *
 * Input:       	None	
 *                  
 * Output:      	None
 *
 * Side Effects:    None
 *
 * Overview:        Update timer and process any timer events. If the 
 *					timer overflows then the state of the connection
 *					is changed.
 *
 * Note:            None
 ********************************************************************/
void _Conn3TimerEvent(void)
{
	// Process the watchdog if the packet rate is other than 0
	if (uConn3.attrib.expected_packet_rate.word)
	{
		// Adjust the time
		uConn3.timer.word -= TICK_RESOLUTION;

		// If the wdt expires then change the state of the connection
		if (uConn3.timer.word == 0) 
		{
			uConn3.attrib.state = _STATE_TIMED_OUT;
			_establishFlags.bits.strobe = 0;
		}
	}	   
}	   





/*********************************************************************
 * Function:        void _Conn3RxEvent(void)
 *
 * PreCondition:    none
 *
 * Input:       	none
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process received data for this connection.
 *
 * Note:            This event occures when data has been received
 *					for this connection instance.
 ********************************************************************/
void _Conn3RxEvent(void)
{
	// Get the length of the message
	uConn3.rx.len = CANGetRxCnt();

 	// Copy the message to the connection buffer
 	CANGetRxDataTyp0(uConn3.rx.pMsg);
 	//*((_MSG_DATA *)(uConn3.rx.pMsg)) = *((_MSG_DATA *)CANGetRxDataPtr());
     
 	// Reset the connection wdt
 	uConn3.timer.word = uConn3.attrib.expected_packet_rate.word;	
		
 	// Indicate message has been received (located in conn.c)	
 	_rxFlag.bits.strobe = 1;
	
	// Notify the application
	UsrConnRxDataEvent(3);
	
  	// Release the hardware to continue receiving
   	CANRead();	
}




/*********************************************************************
 * Function:        void _Conn3TxOpenEvent(void)
 *
 * PreCondition:    A transmit request must have been made.
 *
 * Input:       	none	
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process open transmit que event
 *
 * Note:            This event occurs when the buffer is available 
 *					for this connection instance to transmit. A 
 *					transmit request must have been made to enter 
 *					this function.
 ********************************************************************/
void _Conn3TxOpenEvent(void)
{
 	// Copy the message to the hardware buffer
 	CANPutTxDataTyp0(uConn3.tx.pMsg);
	//*((_MSG_DATA *)CANGetTxDataPtr()) = *((_MSG_DATA *)(uConn3.tx.pMsg));

	// Set the produced CID
	CANPutTxCID(uConn3.attrib.produced_cid.word);

	// Set the length of the message
	CANPutTxCnt(uConn3.tx.len);

	// Request the hardware to queue the message to send
	CANSend(3);
	
	// Clear the transmit flag to open access to the write buffer
	// Notice the queue is cleared regardless of actually sending data
	_txFlag.bits.strobe = 0;		
}


/*********************************************************************
 * Function:        void _Conn3TxEvent(void)
 *
 * PreCondition:    Data must have been queued to transmit.
 *
 * Input:       	none	
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process data for this connection.
 *
 * Note:            This event occurs when the buffer has successfully
 *					placed the requested data on the bus.
 ********************************************************************/
void _Conn3TxEvent(void)
{
	// Set flag indicating data has been placed on the bus
	_txFinFlags.bits.strobe = 1;
	
	// Notify the application
	UsrConnTxDataEvent(3);
}





/*********************************************************************
 * Function:        unsigned char _Conn3ExplicitEvent(void)
 *
 * PreCondition:    none
 *
 * Input:       	none	
 *                  
 * Output:      	unsigned char success of the request
 *
 * Side Effects:    none
 *
 * Overview:        Handle explicit messaging for this instance
 *
 * Note:            None
 ********************************************************************/
unsigned char _Conn3ExplicitEvent(void)
{
	switch(mRouteGetServiceID())
   	{
   		case SRVS_GET_ATTRIB_SINGLE:
   			return (_Conn3GetAttrib());
   		case SRVS_SET_ATTRIB_SINGLE:
   			return (_Conn3SetAttrib());
   	   	default:
   			mRoutePutError(ERR_SERVICE_NOT_SUPPORTED);
   			break;
   	}
		
	return (1);
}


/*********************************************************************
 * Function:        unsigned char _Conn3GetAttrib()
 *
 * PreCondition:    none
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char success of the request
 *
 * Side Effects:    none
 *
 * Overview:        Handle explicit messaging
 *
 * Note:            None
 ********************************************************************/
unsigned char _Conn3GetAttrib(void)
{
	UINT	work;
	USINT 	i;

	switch (mRouteGetAttributeID())
	{
		case	_ATTRIB_STATE:
			mRoutePutByte(uConn3.attrib.state);
			break;
		case	_ATTRIB_INSTANCE_TYPE:
			mRoutePutByte(1);
			break;
		case	_ATTRIB_CLASS_TRIGGER:
			mRoutePutByte(uConn3.attrib.transportClass.byte);
			break;
		case 	_ATTRIB_PRODUCED_CID:
			work.word = (uConn3.attrib.produced_cid.word >> 5);
			mRoutePutByte(work.bytes.LSB);
			mRoutePutByte(work.bytes.MSB);
			break;
		case	_ATTRIB_CONSUMED_CID:
			work.word = (uConn3.attrib.consumed_cid.word >> 5);
			mRoutePutByte(work.bytes.LSB);
			mRoutePutByte(work.bytes.MSB);
			break;	
		case 	_ATTRIB_INITIAL_COMM_CHAR:
			mRoutePutByte(0x02);
			break;
		case	_ATTRIB_PRODUCED_CONN_SIZE:
			mRoutePutByte(uConn3.attrib.produced_con_size.bytes.LSB);
			mRoutePutByte(uConn3.attrib.produced_con_size.bytes.MSB);
			break;
		case	_ATTRIB_CONSUMED_CONN_SIZE:
			mRoutePutByte(uConn3.attrib.consumed_con_size.bytes.LSB);
			mRoutePutByte(uConn3.attrib.consumed_con_size.bytes.MSB);
			break;
		case	_ATTRIB_EXPECTED_RATE:
			mRoutePutByte(uConn3.attrib.expected_packet_rate.bytes.LSB);
			mRoutePutByte(uConn3.attrib.expected_packet_rate.bytes.MSB);
			break;
		case	_ATTRIB_WDT_ACTION:
			mRoutePutByte(0x00);
			break;
		case	_ATTRIB_PRODUCED_CONN_PATH_LEN:
			mRoutePutByte(uConn3.attrib.produced_path_len.bytes.LSB);
			mRoutePutByte(uConn3.attrib.produced_path_len.bytes.MSB);
			break;
		case	_ATTRIB_PRODUCED_CONN_PATH:
			if (mRouteOutLen() > uConn3.attrib.produced_path_len.bytes.LSB)  
			{
				for (i = 0; i < (uConn3.attrib.produced_path_len.bytes.LSB); i++)
				{
					mRoutePutByte(uConn3.attrib.produced_path[i]);
				}
			}
			else
			{
   				mRoutePutError(ERR_REPLY_TOO_LARGE);
			}
			break;
		case	_ATTRIB_CONSUMED_CONN_PATH_LEN:
			mRoutePutByte(uConn3.attrib.consumed_path_len.bytes.LSB);
			mRoutePutByte(uConn3.attrib.consumed_path_len.bytes.MSB);
			break;
		case	_ATTRIB_CONSUMED_CONN_PATH:
			if (mRouteOutLen() > uConn3.attrib.consumed_path_len.bytes.LSB)  
			{
				for (i = 0; i < (uConn3.attrib.consumed_path_len.bytes.LSB); i++)
				{
					mRoutePutByte(uConn3.attrib.consumed_path[i]);
				}
			}
			else
			{
				mRoutePutError(ERR_REPLY_TOO_LARGE);
			}
			break;
		default:
			mRoutePutError(ERR_ATTRIB_NOT_SUPPORTED);
			break;
	}
	return(1);
}




/*********************************************************************
 * Function:        unsigned char _Conn3SetAttrib(void)
 *
 * PreCondition:    none
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char success of the request
 *
 * Side Effects:    none
 *
 * Overview:        Handle explicit messaging
 *
 * Note:            None
 ********************************************************************/
unsigned char _Conn3SetAttrib(void)
{
	unsigned char 	work;
	
	// Ignore the first byte (it is actually the attribute ID) 
	mRouteGetByte();

	switch (mRouteGetAttributeID())
	{	
		case	_ATTRIB_EXPECTED_RATE:
			// Continue only if there is sufficient data
			if (mRouteTestValidInputDataLen(2))
			{
				if (uConn3.attrib.state == _STATE_CONFIGURING)
				{ 
					uConn3.attrib.state = _STATE_ESTABLISHED;
					_establishFlags.bits.strobe = 1;
				}

				// Read in the requested packet rate
				uConn3.attrib.expected_packet_rate.bytes.LSB = mRouteGetByte();
				uConn3.attrib.expected_packet_rate.bytes.MSB = mRouteGetByte();
	
				// Get the ls bits
				work = uConn3.attrib.expected_packet_rate.bytes.LSB & (TICK_RESOLUTION - 1);
	
				// Remove the ls bits from desired resolution
				uConn3.attrib.expected_packet_rate.bytes.LSB &= (~(TICK_RESOLUTION - 1));
				
				// Round up if necessary
				if (work) uConn3.attrib.expected_packet_rate.word += (TICK_RESOLUTION);
	
				// Return the value actually used
				mRoutePutByte(uConn3.attrib.expected_packet_rate.bytes.LSB);
				mRoutePutByte(uConn3.attrib.expected_packet_rate.bytes.MSB);
	
				// Set the timer 4x (section 5-4.4.2)
				uConn3.timer.word = uConn3.attrib.expected_packet_rate.word << 2; 
			}	
			break;
		case	_ATTRIB_CLASS_TRIGGER:
			if (uConn3.attrib.state == _STATE_CONFIGURING)
			{
				if (mRouteTestValidInputDataLen(1))
				{ 
					// To be handled by the app
					UsrConnSetAttribEvent(3);
				}
			}
			else
			{
				mRoutePutError(ERR_OBJECT_STATE_CONFLICT);
			}
			break;
		case	_ATTRIB_PRODUCED_CONN_PATH:
			if (uConn3.attrib.state == _STATE_CONFIGURING)
			{
				if (mRouteTestValidInputDataLen(uConn3.attrib.produced_path_len.bytes.LSB))
				{
					// To be handled by the app
					UsrConnSetAttribEvent(3);
				}
			}
			else
			{
				mRoutePutError(ERR_OBJECT_STATE_CONFLICT);
			}
			break;
		case	_ATTRIB_CONSUMED_CONN_PATH:
			if (uConn3.attrib.state == _STATE_CONFIGURING)
			{
				if (mRouteTestValidInputDataLen(uConn3.attrib.consumed_path_len.bytes.LSB))
				{
					// To be handled by the app
					UsrConnSetAttribEvent(3);
				}
			}
			else
			{
				mRoutePutError(ERR_OBJECT_STATE_CONFLICT);
			}
			break;
		case	_ATTRIB_PRODUCED_CONN_SIZE:
			if (uConn2.attrib.state == _STATE_CONFIGURING)
			{
				if (mRouteTestValidInputDataLen(2))
				{
					// To be handled by the app
					UsrConnSetAttribEvent(3);
				}
			}
			else
			{
				mRoutePutError(ERR_OBJECT_STATE_CONFLICT);
			}
			break;
		case	_ATTRIB_WDT_ACTION:
		case	_ATTRIB_CONSUMED_CONN_SIZE:
		case	_ATTRIB_STATE:
		case	_ATTRIB_INSTANCE_TYPE:
		case 	_ATTRIB_PRODUCED_CID:
		case	_ATTRIB_CONSUMED_CID:
		case 	_ATTRIB_INITIAL_COMM_CHAR:
		case	_ATTRIB_PRODUCED_CONN_PATH_LEN:
		case	_ATTRIB_CONSUMED_CONN_PATH_LEN:
			mRoutePutError(ERR_ATTRIB_NOT_SETTABLE);
			break;
			
		default:
			mRoutePutError(ERR_ATTRIB_NOT_SUPPORTED);
			break;
	}
	return(1);
}

// SUPPORT_BIT_STROBED
#endif 	
