// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess;

import java.util.*;

/**
 * A Board represents an arbitrary 2-dimensional game board that is somehow
 * populated by pieces that are allowed to move in specific waves.
 * <p>
 * Board defines some game-over conditions for all Boards built off of this one:
 * NOTOVER, with the value 0, and OUTOFTIME, with value 1. Subclasses of this
 * Board should not set a different end condition that matches this one's
 * integer value.
 * <p>
 *  
 * @specfield player : Player      // The Player whose move it is
 * @specfield pieces : Set&ltPieces&gt // The Pieces on the board
 * @specfiled observers : Set&ltBoardObservers&gt	// Observers that are notified when something on the Board changes
 **/
public abstract class Board <M extends Move>
{

	/**
	 * Default Board termination condition, indicating
	 * that the game is not over
	 */
	public static final int NOTOVER = 0;
	/**
	 * End-game condition that signifies that the
	 * game has run out of time for some player
	 */
	public static final int OUTOFTIME = 1;
	
	// Abstraction Function
	// For all Boards b:
	// 	player(b) => b.player
	// 	observers(b) => b.observers
	
	// Representation Invariant
	// For all Boards b:
	// 	b.gameOverReason matches the value
	// 		of one of the end-game
	// 		condition variables set
	// 		depending on the game.
	
	// FIELDS
	protected Player player;
	protected Player winner;
	protected int gameOverReason;
    protected List<BoardObserver> observers;
    
    /**
     * Initializes some variables for the Board assuming
     * it is being created for a new game,
     * including observers.
     */
    // Some variables, like player, must be initialized in the constructors
    // of subclasses of Board.
    public Board() {
    	winner = Player.NONE;
    	gameOverReason = NOTOVER;
    	observers = new ArrayList<BoardObserver>();
    }

    /**
     * Creates a clone of this Board
     */
	public abstract Board<M> clone();
    
	/**
	 * Inheritable helper method for copying various
	 * data members from this Board to an empty Board.
	 * @param b the empty board into which this board
	 * should be copied.
	 */
    protected void copyThis(Board<M> b) {
    	// Copy winner, gameOverReason, other end-game variables
    	b.winner = this.winner;
    	b.gameOverReason = this.gameOverReason;
    }
    
	/**
	 * @return the number of rows on the board
	 */
	public abstract int getRows();
	
	/**
	 * @return the number of columns on the board
	 */
	public abstract int getColumns();
	
	/**
	 * @return this.player
	 **/
	public Player getPlayer() {
		return player;
	}

	/**
	 * @return a Piece p in this.pieces with p.row == row and p.column
	 * == column or <tt>null</tt> if no such piece exists
	 **/
	public abstract Piece getPieceAt(int row, int column);

	/**
	 * @return an interator over this.pieces in an unspecified order
	 **/
	public abstract List<Piece> getPieces();

	/**
	 * @return an interator over {p in this.pieces | p.player ==
	 * player} in an unspecified order
	 **/
	public abstract List<Piece> getPieces(Player player);

	/**
	 * @return an iterator over all legal moves for the current player
	 * from the current position
	 */
	public abstract List<M> allLegalMoves();

	/**
	 * @require p.player == this.player
	 * @return an iterator over all legal moves for the specified
	 * Piece
	 **/
	public abstract List<M> legalMoves(Piece p);

	/**
	 * @return true iff the given move is an allowed move for the
	 * current player
	 **/
	public abstract boolean isMoveLegal(M m);

	/**
	 * @modifies this
	 * @effects the current board to rearrange the state of the
	 * Pieces according to the given Move
	 * @param m the Move that describes how to modify the Board
	 * @throws IllegalMoveException if the move is illegal
	 */
	public void doMove(M m) throws IllegalMoveException {
		doMove(m, "");
	}
	
	/**
	 * @modifies this
	 * @effects modifies the current board to rearrange the state
	 * of the Pieces according to the given Move, and if that
	 * move has an associated time (such as in timed games), logs
	 * that time.
	 * @param m the Move that describes how to modify the Board
	 * @param timestamp the time that Move <code>m</code> was made
	 * @throws IllegalMoveException if the move is illegal
	 */
	public abstract void doMove(M m, String timestamp)
		throws IllegalMoveException;
	
	/**
	 * @modifies this
	 * @effects undoes the last Move performed on the Board, restoring
	 * it to its previous condition before that Move was executed.
	 */
	public abstract void undoLastMove();
	
	/**
	 * @return true if the game is over; false otherwise
	 */
	public boolean isGameOver() {
		return gameOverReason != NOTOVER;
	}
	
	/**
	 * @return the Player that has won the game
	 */
	// I lose
	public Player getWinner() {
		return winner;
	}
	
	/**
	 * @return the termination condition for this game
	 */
	public int getGameOverReason() {
		return gameOverReason;
	}
	
	/**
	 * Terminates the current game, specifying the cause of
	 * game termination and the winner.  
	 * @param gameOverReason the game termination condition.
	 * This value should match the value of one of the end-
	 * game conditions available for a particular game.
	 * @param winner the winner of the game 
	 * @modifies this
	 * @effects sets the end-game variables on the board
	 * to those values specified.
	 */
	public void endGame(int gameOverReason, Player winner) {
		this.gameOverReason = gameOverReason;
		this.winner = winner;
	}
	
    // Methods for the Observers
    /**
     * Adds an observer to the list of BoardObservers
     * @param bo the BoardObserver to add to this Board
     * 
     * @modifies this.observers
     * @effects adds <code>bo</code> to the list of observers
     */
    public void addObserver(BoardObserver bo) {
    	observers.add(bo);
    }
    
    /**
     * Removes an observer from the list of BoardObservers.
     * @param bo the BoardObserver to remove from this Board
     * 
     * @modifies this.observers
     * @effects removes <code>bo</code> from the list of observers
     * if <code>bo</code> is present in the list of observers;
     * otherwise nothing
     */
    public void removeObserver(BoardObserver bo) {
    	observers.remove(bo);
    }
    
    /**
     * Updates all observers in the list of observers that the Board has
     * been changed.
     *
     */
    public void notifyObservers() {
    	for (BoardObserver bo : observers)
    		bo.postRefresh();
    }
    
}
