package antichess;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import antichess.ai.AntichessAI;
import antichess.viewgui.BoardView;
import antichess.viewgui.ChessGUI;

/**
 * ControllerMaster provides a facade for the controller that other classes
 * can use to cause the controller to perform the basic actions relating to 
 * saving, loading and starting new games.
 * 	
 * The currentGameController notifies the ControllerMaster of it's current state,
 * whether the currentGameController is in the middle of the run method or not.
 * This state is defined as: isRunning.  If the ControllerMaster wants the 
 * GameController to pause first it checks if the current GameController is running,
 * if it isnt' the ControllerMaster simply sets awaitingPause to true.
 * 
 * 
 * @author nlharr
 *
 * @specfield description				//GameDescriptor
 * @specfield currentGameContoller		//GameController
 * @specfield awaitingPause				//boolean
 * @specfield isGameControllerRunning	//boolean
 */


public class ControllerMaster extends ClassLoader{
	//Fields
	private Board currentBoard;
	private GameClock currentClock;
	private ChessGUI currentChessGUI;
	private String currentGUI;
	private GameDescriptor oldDescriptor;
	private GameController currentGameController;
	private boolean controllerPaused;
	private boolean awaitingPause;
	
	private ControllerMaster(){
		controllerPaused = true;
		awaitingPause = false;
		currentChessGUI = null;
		currentGUI = "NONE";
		currentBoard = null;
		currentClock = null;
		oldDescriptor = null;
	}
	
	public static ControllerMaster createControllerMaster(){
		return new ControllerMaster();
	}
	
	/**
	 * Saves the current game to the the file specified by filename
	 * 
	 */
	public void save(File writeFile) throws IOException {

			if (currentGUI.equals("Chess")||currentGUI.equals("Antichess")){
				if (currentBoard == null || currentClock == null){
					throw new RuntimeException("Unable to save unitiated game");
				}
				pauseCurrentGameController();
				assert(!isGameControllerRunning());
				
				GameWriter writer;
				ChessBoard tempBoard;
				//casting because I'm a bad person
				if (currentBoard instanceof ChessBoard){
					tempBoard = (ChessBoard) currentBoard;
				}else{
					throw new RuntimeException("currentBoard is not a chessboard");
				}
				//gets the writer
				try{
					if (!(currentClock.hasPlayer(Player.WHITE))||!(currentClock.hasPlayer(Player.BLACK))){
						writer = new GameWriter(tempBoard, null, null);
					}else{
						writer = new GameWriter(tempBoard, currentClock.getTimer(Player.WHITE),
														   currentClock.getTimer(Player.BLACK));
					}
				}catch (Exception ex){
					(new Thread(currentGameController)).start();
					throw new IOException(ex.getMessage());
				
				}
				//attempts to write the file
				try{
					writer.writeGame(writeFile);
				}catch (Exception ex){
					(new Thread(currentGameController)).start();
					throw new IOException(ex.getMessage());
				}
				
				(new Thread(currentGameController)).start();
				
			}
			
		
	}
	
	/**
	 * Stops the current game and loads a game specified by fileName
	 * It for players it loads a human and an AI the human color is specified by humanPlayer
	 *
	 */
	public void load(File readFile, Player humanPlayer) throws IOException, InvalidGameFileException{
		if (oldDescriptor == null || currentGUI.equals("NONE")){
			throw new RuntimeException("No previous game has been initiliazed");
		}
		if (currentGUI.equals("Chess")){
			pauseCurrentGameController();
			assert(!isGameControllerRunning());
			
			GameReader reader;
			try{
				reader = new GameReader(readFile);
			}catch (IOException ex){
				(new Thread(currentGameController)).start();
				throw new IOException(ex.getMessage());
			}catch (InvalidGameFileException exc){
				(new Thread(currentGameController)).start();
				throw new InvalidGameFileException(exc.getMessage());
			}
			
			Board board = reader.getBoard();
			currentBoard = board;
			ChessBoard cboard;
			if (board instanceof ChessBoard){
				cboard = (ChessBoard) board;
			}else{
				(new Thread(currentGameController)).start();
				throw new RuntimeException("Improper board type for loading");
			}
			
			
			
			GameClock clock;
			//initializes the clock
			if (reader.isTimed()){
				Player[] players = {Player.WHITE, Player.BLACK};
				long[] startTimes = {reader.getWhiteTimer().getStartTime(),
									 reader.getBlackTimer().getStartTime()};
				long[] currentTimes =  {reader.getWhiteTimer().getTime(),
						 				reader.getBlackTimer().getTime()};
				clock = new GameClock(players, startTimes, currentTimes);
			}else{
				long[] startTimes = {};
				long[] currentTimes = {};
				Player[] players = {};
				clock = new GameClock(players, startTimes, currentTimes);
			}
			
			currentChessGUI.switchViewedGame(cboard, clock);
			
			
			//initializes the GamePlayers
			GamePlayer whitePlayer;
			GamePlayer blackPlayer;
			if (humanPlayer.equals(Player.WHITE)){
				whitePlayer = createPlayer("Human",
									   cboard, currentChessGUI.getBoardView(), Player.WHITE);
				blackPlayer = createPlayer("AI",
									   cboard, currentChessGUI.getBoardView(), Player.BLACK);
			}else{
				whitePlayer = createPlayer("AI",
						   cboard, currentChessGUI.getBoardView(), Player.WHITE);
	blackPlayer = createPlayer("Human",
						   cboard, currentChessGUI.getBoardView(), Player.BLACK);
			}
			ArrayList<GamePlayer> playerList = new ArrayList<GamePlayer>();
			playerList.add(whitePlayer);
			playerList.add(blackPlayer);
			
			currentGameController = GameController.createGameController(board, clock, playerList, this);
			(new Thread(currentGameController)).start();
		
		}
	}
	
	/**
	 * Starts a new game based of the format specified in GameDescriptor
	 * 
	 */
	public void startNewGame(GameDescriptor newGame){
		//First we stop the old game controller
		pauseCurrentGameController();
		assert(!isGameControllerRunning());
		
		
		//First we check for the type of game
		if ((newGame.getGameTypeName().equals("Chess"))||
				(newGame.getGameTypeName().equals("Antichess"))){
			oldDescriptor = newGame;
			
			//we make sure the GameDescriptor has a proper format for Chess or Antichess
			if ((newGame.getPlayers().size()!=2)||
					(newGame.getPlayerDescriptions().size() != 2)||
					(newGame.getPlayerTimes().length != 2)){
				throw new IllegalArgumentException("Input Sizes for Chess are wrong");
			}
			
			if ((!newGame.getPlayers().get(0).equals(Player.WHITE))||
					(!newGame.getPlayers().get(1).equals(Player.BLACK))){
				throw new IllegalArgumentException("Player order for Chess is wrong");
			}
			
			ChessBoard board;
			//initiliazes the Board
			if (newGame.getGameTypeName().equals("Chess")){
				board = new ChessBoard();
			}else{
				board = new AntichessBoard();
			}
			board.newGame();
			currentBoard = board;
			
			GameClock clock = getClock(newGame);

			if(currentGUI.equals("Chess")){
					currentChessGUI.switchViewedGame(board, clock);
			}else{
				currentChessGUI = ChessGUI.createGUI(board, clock, this);
				javax.swing.SwingUtilities.invokeLater(currentChessGUI);
				currentGUI = "Chess";
			}
			//initializes the GamePlayers
			GamePlayer whitePlayer;
			GamePlayer blackPlayer;
			whitePlayer = createPlayer(newGame.getPlayerDescriptions().get(0),
									   board, currentChessGUI.getBoardView(), Player.WHITE);
			blackPlayer = createPlayer(newGame.getPlayerDescriptions().get(1),
									   board, currentChessGUI.getBoardView(), Player.BLACK);

			ArrayList<GamePlayer> playerList = new ArrayList<GamePlayer>();
			playerList.add(whitePlayer);
			playerList.add(blackPlayer);
			
			currentGameController = GameController.createGameController(board, clock, playerList, this);
		
			
		}
		
		(new Thread(currentGameController)).start();
		
		
	}

	/**
	 * Return a GamePlayer of the given named type (as in
	 * GameDescriptor), for the given board and board view, playing on
	 * the named side
	 */
	private GamePlayer createPlayer(String type, ChessBoard board, BoardView view, Player side) {
		if (type.equals("Human")){
			return new ChessPlayer(board, view, side);
		}else if (type.equals("AI")){
			if(board instanceof AntichessBoard)
				return new AIPlayer<ChessMove,AntichessBoard>((AntichessBoard)board, side, new AntichessAI());
			else
				return RandomAI.createPlayer(board, side);
		}else{
			throw new IllegalArgumentException("Improper player description for Chess");
		}
	}	

	
	/**
	 * Return a GameClock of the type described in the GameDescriptor
	 */
	public GameClock getClock(GameDescriptor newGame){
		//initializes the clock
		
		//this dense little block filters out all the times that have 0 as there start
		//value so that those players games count as untimed
		ArrayList<Player> playersClock = new ArrayList<Player>();
		ArrayList<Long> timesClock = new ArrayList<Long>();
		for (int i=0; i<newGame.getPlayerTimes().length; i++){
			if (newGame.getPlayerTimes()[i]!=0){
				playersClock.add(newGame.getPlayers().get(i));
				timesClock.add(Long.valueOf(newGame.getPlayerTimes()[i]));
			}
		}
		//adds the times to an array
		long[] timesClockProper = new long[timesClock.size()];
		for (int i=0; i<timesClock.size(); i++){
			timesClockProper[i]=timesClock.get(i);
		}
		currentClock = new GameClock(playersClock.toArray(new Player[playersClock.size()]), timesClockProper);
		return currentClock;
	}
	
	
	//handle communication for the GameController

	/**
	 * @modifies isGameControllerRunning
	 * @effects isGameControllerRunning = false
	 */
	public synchronized void pauseCurrentGameController(){
		awaitingPause = true;
		
		while (isGameControllerRunning()){
			try {
		        this.wait();
		      } catch (InterruptedException e) {}
		}
		
		awaitingPause = false;
	}
	
	
	
	
	
	/**
	 * Notifies that the currentGameController is paused
	 * 
	 * @requieres that only the currentGameController calls this method.
	 * 			  that means when calling this method one should only use 'this' as the parameter
	 * 			  and this method should only be called in the GameController class
	 * 
	 * @modifies isGameControllerRunning
	 * @effects isGameControllerRunning = false
	 */
	public synchronized void setGameControllerPaused(GameController controller){
		//referential equality is purposeful
		if (controller != currentGameController){
			throw new RuntimeException("Not the current controller is attempting " +
										"to set pause for the currentGameController");
		}
		
		controllerPaused = true;
		this.notify();
		System.out.println("Master knows GameController Paused");
	}
	/**
	 * Notifies that the currentGameController is running
	 * 
	 * @requieres that only the currentGameController calls this method.
	 * 			  that means when calling this method one should only use 'this' as the parameter
	 * 			  and this method should only be called in the GameController class
	 * 
	 * @modifies isGameControllerRunning
	 * @effects isGameControllerRunning = true
	 */
	public synchronized void setGameControllerRunning(GameController controller){
		//referential equality is purposeful
		if (controller != currentGameController){
			throw new RuntimeException("Not the current controller is attempting " +
										"to set pause for the currentGameController");
		}
		controllerPaused = false;
		System.out.println("Master knows GameController Running");
	}
	
	public synchronized boolean isGameControllerRunning(){
		return !controllerPaused;
	}
	
	
	public synchronized boolean isAwaitingPause(){
		return awaitingPause;
	}
	
	
	
	
	
	
	
	public static void main(String[] args){
		ControllerMaster controller = ControllerMaster.createControllerMaster();
		ArrayList<Player> playerList = new ArrayList<Player>();
		playerList.add(Player.WHITE);
		playerList.add(Player.BLACK);
		
		ArrayList<String> descriptions = new ArrayList<String>();
		descriptions.add("Human");
		descriptions.add("AI");
		long[] times = {300000, 300000};
		
		controller.startNewGame(new GameDescriptor("Antichess", playerList, descriptions,
				times, false));

	} 
}
