package antichess;

import antichess.ai.*;

/**
 * A MachinePlayer is a player of Antichess that uses artificial
 * intelligence to decide on its moves.
 */
public class MachinePlayer
{
	/**
	 * The game of antichess this <code>MachinePlayer</code> is
	 * playing
	 */
	private AntichessBoard board;

	/**
	 * The side this <code>MachinePlayer</code> represents
	 */
	private Player player;

	/**
	 * The <code>GameAI</code> this <code>MachinePlayer</code> uses to
	 * decide its moves.
	 */
	private GameAI<ChessMove, AntichessBoard> ai;
	
	/**
	 * @param isWhite
	 * @param level
	 * @effects Creates a new MachinePlayer with the normal starting
	 * board (as described in the assignment). If isWhite is true, the
	 * player plays as white, else as black. level is an optional,
	 * arbitrary integer which can be used to affect the algorithm the
	 * machine player uses. The tournament referee will always request
	 * a level 0 MachinePlayer.
	 */
	public MachinePlayer(boolean isWhite, int level)
	{
		player = isWhite ? Player.WHITE : Player.BLACK;
		board = new AntichessBoard();
		board.newGame();
		ai = new AntichessAI();
	}
	
	/**
	 * @param isWhite
	 * @param level
	 * @param bd
	 * @requires bd is a String containing a valid &ltpieces&gt
	 * element, including the start and end tag of &ltpieces&gt, as
	 * defined in the assignment.
	 * @effects Creates a new MachinePlayer in the same way described
	 * by the constructor MachinePlayer(boolean, int), except that the
	 * initial board configuration is set to be the one described by
	 * the String bd.
	 */
	public MachinePlayer(boolean isWhite, int level, String bd)
	{
		player = isWhite ? Player.WHITE : Player.BLACK;
		try {
			board = new AntichessBoard();
			// We're not here given whose turn it is. However, we will
			// always be given a move by the other player before we're
			// asked to make a move (even though it may be a null
			// move), so we load with it being the other player's
			// turn.
			GameReader.loadBoard(board, bd, player.otherPlayer());
		} catch(InvalidGameFileException e) {
			throw new RuntimeException("Bad board description: " + e);
		}
		ai = new AntichessAI();
	}
	
	/**
	 * @return a String representing the name of the MachinePlayer
	 * (this is needed for the tournament.)
	 */
	public String getName()
	{
		return "Ze Nihilists";
	}
	
	/**
	 * @param opponentMove
	 * @param timeLeft
	 * @param opponentTimeLeft
	 * @requires opponentMove be either a String representing a valid
	 * move by the opponent on the board stored in this (in the
	 * "standard string format" defined by the assignment), or the
	 * empty String. Also, timeLeft > 0 && opponentTimeLeft > 0.
	 * @modifies this
	 * @return a valid next move for this player in a String of the
	 * appropriate format, given the opponent's move, the time left
	 * for this player, and the time left for the opponent. (Both
	 * times are in milliseconds.) If opponentMove is the empty
	 * String, then the board for this player should be considered up
	 * to date (as would be the case if this player is asked to make
	 * the first move of the game). NOTE: This procedure may run
	 * greater than timeLeft, but this would mean losing the game.
	 */
	public String makeMove(String opponentMove, int timeLeft, int opponentTimeLeft)
	{
		if(opponentMove.equals("")) {
			/* If we get an empty move, but it's not our turn, it's a
			 * null move. If it is our turn, then we just started */
			if(board.getPlayer() != player)
				try {
					board.doMove(null);
				} catch(IllegalMoveException e) {
					throw new RuntimeException("Board threw an exception making a null move");
				}
		} else {
			ChessMove other = GameReader.parseMove(opponentMove, board);
			try {
				board.doMove(other);
			} catch(IllegalMoveException e) {
				throw new RuntimeException("Board claimed " + opponentMove + " illegal");
			}
		}
		ChessMove move = ai.findMove(board, timeLeft, opponentTimeLeft);
		if(move == null)
			return "";
		else
			return move.humanReadableString();
	}
}
