// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess.movegen;

import antichess.*;
import java.util.*;

/**
 * ChessMoveGenerator is an abstract implementation of MoveGenerator
 * for generating ChessMoves. 
 **/
public abstract class ChessMoveGenerator implements MoveGenerator<ChessMove>
{
    public abstract List<ChessMove> getMoves(Piece piece, Board board);

	/**
	 * <p> Ensures that the given move is a <tt>ChessMove</tt>, that
	 * its destination coordinates are within the bounds of the Board,
	 * and that the Piece on the board at the destination square is
	 * either <tt>null</tt>, or belongs to the other player than the
	 * Move's player. In addition, it ensures that
	 * <tt>move.capturedPiece</tt> is in fact the move located at
	 * (move.row, move.column)</p>
	 *
	 * <p> If these conditions are satisfied, it calls
	 * <tt>isMoveLegal(ChessMove, Board)</tt>; If not, returns
	 * false. </p>
	 **/
	public boolean isMoveLegal(ChessMove move, Board board){
		if(move.getRow() < 0 || move.getRow() >= board.getRows()
		   || move.getColumn() < 0 || move.getColumn() >= board.getColumns())
			return false;
		Piece otherPiece = board.getPieceAt(move.getRow(), move.getColumn());

		if(otherPiece != null) {
			if(otherPiece.getPlayer() == move.getPiece().getPlayer()
			   || !otherPiece.equals(move.getCapturedPiece()))
				return false;
		} else if(move.getCapturedPiece() != null) {
			return false;
		}

		return isChessMoveLegal(move, board);
	}

	/**
	 * Check that the given move is legal for this piece type.
	 * @requires 0 <= move.row < board.rows && 0 <= move.column <
	 * board.columns && board.pieceAt(move.row, move.col) == null ||
	 * board.pieceAt(move.row, move.col).player != move.piece.player
	 * && board.pieceAt(move.row, move.col) = move.capturedPlayer
	 */
	public abstract boolean isChessMoveLegal(ChessMove move, Board board);

}
