// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess.test;

import java.util.*;
import junit.framework.*;
import antichess.*;

/**
 * Tests that the various pieces generate the correct moves.
 **/
public class MoveGeneratorTests extends TestCase
{
	public MoveGeneratorTests(String name) {
		super(name);
	}

	/**
	 * Convert an array of Pieces into a List. Why Java doesn't have
	 * something like this natively is a damn good question.
	 */
	private List<Piece> pieces(Piece... pieces) {
		List<Piece> c =  new ArrayList<Piece>();
		for (Piece p : pieces)
			c.add(p);
		return c;
	}

	/**
	 * @return a white pawn at the specified location
	 */
	private Piece white(int row, int col) {
		return new Piece(Player.WHITE,
						 PieceTypeFactory.getPawnType(),
						 row, col);
	}

	/**
	 * @return a black pawn at the specified location
	 */
	private Piece black(int row, int col) {
		return new Piece(Player.BLACK,
						 PieceTypeFactory.getPawnType(),
						 row, col);
	}

	/**
	 * @return a white king at the specified location
	 */
	private Piece whiteKing(int row, int col) {
		return new Piece(Player.WHITE,
						 PieceTypeFactory.getKingType(),
						 row, col);
	}

	/**
	 * @return a black king at the specified location
	 */
	private Piece blackKing(int row, int col) {
		return new Piece(Player.BLACK,
						 PieceTypeFactory.getKingType(),
						 row, col);
	}

	private void testMoves(List<Move> moveList,
								  Piece piece, Board board,
								  int[][] moves) {
		ChessMove cm;
		int count = 0;
		boolean ok;
		for(Move m : moveList) {
			count++;
			assertTrue("Get a ChessMove", m instanceof ChessMove);
			cm = (ChessMove)m;
			assertEquals(m.getPiece(), piece);
			assertEquals(cm.getCapturedPiece(),
						 board.getPieceAt(cm.getRow(), cm.getColumn()));
			ok = false;
			for(int[] pos : moves) {
				if(pos[0] == cm.getRow() &&
				   pos[1] == cm.getColumn())
					ok = true;
			}
			if(!ok)
				assertTrue( piece + " returned illegal move " + m, false);
		}
		assertEquals(piece + " returned wrong number of moves", moves.length, count);
	}

	private <M extends Move> void testPiece(Board<M> board,
											Piece piece,
											int[][] expectedMoves) {
		testMoves(piece.getMovementType().getMoves(piece, board),
						 piece, board, expectedMoves);
		List<M> m = piece.getMovementType().getMoves(piece, board);
		for(M move : m) {
			assertTrue("Move " + move + " should be legal",
					   piece.getMovementType().isMoveLegal(move, board));
		}
	}

	private void testPiece(Piece piece, int[][] expectedMoves) {
		ChessBoard board = new ChessBoard();
		List<Piece> pieces = new ArrayList<Piece>();
		pieces.add(piece);
		pieces.add(new Piece(Player.WHITE,
							 PieceTypeFactory.getKingType(),
							 1,0));
		pieces.add(new Piece(Player.BLACK,
							 PieceTypeFactory.getKingType(),
							 6,7));
		board.loadGame(pieces, Player.WHITE, null);
		testPiece(board, piece, expectedMoves);
	}

	public void testRook() {
		Piece rook;
		int [][] expectedMoves;

		rook = new Piece(Player.WHITE,
							   PieceTypeFactory.getRookType(),
							   3, 4);
		expectedMoves = new int[][]
			{
				{3, 0}, {3, 1}, {3, 2}, {3, 3}, {3, 5}, {3, 6}, {3, 7},
				{0, 4}, {1, 4}, {2, 4}, {4, 4}, {5, 4}, {6, 4}, {7, 4},
			};
		testPiece(rook, expectedMoves);

		rook = new Piece(Player.BLACK,
						 PieceTypeFactory.getRookType(),
						 0,7);
		expectedMoves = new int[][]
			{
				{0, 0}, {0, 1}, {0, 2}, {0, 3}, {0, 4}, {0, 5}, {0, 6},
				{1, 7}, {2, 7}, {3, 7}, {4, 7}, {5, 7}
			};
		testPiece(rook, expectedMoves);
	}

	public void testKnight() {
		Piece knight;
		int [][] expectedMoves;

		knight = new Piece(Player.WHITE,
						   PieceTypeFactory.getKnightType(),
						   3, 4);
		expectedMoves = new int[][]
			{
				{2, 6}, {4, 6},
				{1, 5}, {5, 5},
				{1, 3}, {5, 3},
				{4, 2}, {2, 2}
			};
		testPiece(knight, expectedMoves);

		knight = new Piece(Player.BLACK,
						   PieceTypeFactory.getKnightType(),
						   0,0);
		expectedMoves = new int[][]
			{
				{2, 1}, {1, 2}
			};
		testPiece(knight, expectedMoves);
	}

	public void testBishop() {
		Piece bishop;
		int [][] expectedMoves;

		bishop = new Piece(Player.WHITE,
						   PieceTypeFactory.getBishopType(),
						   1, 2);
		expectedMoves = new int[][]
			{
				{0, 1}, {2, 3}, {3, 4}, {4, 5}, {5, 6}, {6, 7},
				{2, 1}, {3, 0}, {0, 3}
			};
		testPiece(bishop, expectedMoves);

		bishop = new Piece(Player.BLACK,
						   PieceTypeFactory.getBishopType(),
						   7,0);
		expectedMoves = new int[][]
			{
				{6, 1}, {5, 2}, {4, 3}, {3, 4}, {2, 5}, {1, 6}, {0, 7},
			};
		testPiece(bishop, expectedMoves);
	}

	public void testQueen() {
		Piece queen;
		int [][] expectedMoves;

		queen = new Piece(Player.WHITE,
						  PieceTypeFactory.getQueenType(),
						  5, 5);
		expectedMoves = new int[][]
			{
				{5, 0}, {5, 1}, {5, 2}, {5, 3}, {5, 4}, {5, 6}, {5, 7},
				{0, 5}, {1, 5}, {2, 5}, {3, 5}, {4, 5}, {6, 5}, {7, 5},
				{0, 0}, {1, 1}, {2, 2}, {3, 3}, {4, 4}, {6, 6}, {7, 7},
				{6, 4}, {7, 3}, {4, 6}, {3, 7},
			};
		testPiece(queen, expectedMoves);
	}

	public void testKing() {
		Piece king;
		int [][] expectedMoves;

		king = new Piece(Player.WHITE,
						 PieceTypeFactory.getKingType(),
						 3, 3);
		expectedMoves = new int[][]
			{
				{2, 2}, {2, 3}, {2, 4},
				{3, 2}, {3, 4},
				{4, 2}, {4, 3}, {4, 4},
			};
		testPiece(king, expectedMoves);

		king = new Piece(Player.BLACK,
						   PieceTypeFactory.getKingType(),
						   4,7);
		expectedMoves = new int[][]
			{
				{5, 6}, {4, 6}, {3, 6}, {5, 7}, {3, 7},
			};
		testPiece(king, expectedMoves);
	}

	public void testPawn() {
		Piece pawn;
		int [][] expectedMoves;

		pawn = new Piece(Player.WHITE,
						 PieceTypeFactory.getPawnType(),
						 1, 4);
		expectedMoves = new int[][] {{2, 4}, {3, 4}};
		testPiece(pawn, expectedMoves);

		pawn = new Piece(Player.WHITE,
						 PieceTypeFactory.getPawnType(),
						 4, 1);
		expectedMoves = new int[][] {{5, 1}};
		testPiece(pawn, expectedMoves);

		pawn = new Piece(Player.BLACK,
						 PieceTypeFactory.getPawnType(),
						 6, 4);
		expectedMoves = new int[][] {{5, 4}, {4, 4}};
		testPiece(pawn, expectedMoves);

		pawn = new Piece(Player.BLACK,
						 PieceTypeFactory.getPawnType(),
						 5, 6);
		expectedMoves = new int[][] {{4, 6}};
		testPiece(pawn, expectedMoves);
	}

	public void testPawnCaptures() {
		ChessBoard board;
		Piece white, black;
		int [][] expectedMoves;

		board = new ChessBoard();

		white = new Piece(Player.WHITE,
						 PieceTypeFactory.getPawnType(),
						 1,4);
		black = new Piece(Player.BLACK,
						 PieceTypeFactory.getPawnType(),
						 3,4);
		board.loadGame(pieces(white, black,
							  black(2,3),
							  black(2,4),
							  black(2,5),
							  whiteKing(7,7),
							  blackKing(0,7)),
					   Player.WHITE, null);

		expectedMoves = new int[][] {{2,3}, {2,5}};
		testPiece(board, white, expectedMoves);
		expectedMoves = new int[][] {};
		testPiece(board, black, expectedMoves);
	}

	public void testQueenCaptures() {
		ChessBoard board;
		Piece queen;
		int [][] expectedMoves;

		board = new ChessBoard();

		queen = new Piece(Player.WHITE,
						  PieceTypeFactory.getQueenType(),
						  4,2);
		board.loadGame(pieces(queen, white(6,0), black(4,1),
							  white(3,1), white(3,2),
							  black(3,3), white(4,5),
							  white(6,2),
							  whiteKing(0,0),
							  blackKing(7,7)),
					   Player.WHITE, null);
		expectedMoves = new int[][] {
			{5,1}, {4,1}, {4,3}, {3, 3}, {4, 4}, {5,3}, {6, 4}, {7,5}, {5, 2}
		};
		testPiece(board, queen, expectedMoves);
	}

	public void testKnightCaptures() {
		ChessBoard board;
		Piece knight;
		int [][] expectedMoves;

		board = new ChessBoard();

		knight = new Piece(Player.BLACK,
						   PieceTypeFactory.getKnightType(),
						   5,5);
		board.loadGame(pieces(knight,
							  white(7,4), black(7,6),
							  white(6,7), black(4,7),
							  white(3,6), black(3,4),
							  white(4,3), black(6,3),
							  whiteKing(0,0),
							  blackKing(7,7)),
					   Player.BLACK, null);
		expectedMoves = new int[][] {
			{7,4}, {6,7}, {3,6}, {4,3}
		};
		testPiece(board, knight, expectedMoves);
	}
}

