/**
 ** io_mac.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


/* Serial and console library routines for MAC */

unsigned long clock();

#define IO_MODULE
#define MAC

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "util.h"

typedef struct iostream_struct {
    Int stream;
    Int istream;
    Int ostream;
} IOStream;

#include "io.h"

Int io_delay_factor= 1;
Int io_hex_on_serial= 0;

char *io_default_serial_name = "modem";
Int io_debug= 0;

void io_set_delay_factor(Int f)
{
    if (f < 1) f= 1;
    if (f > 100) f= 100;
    io_delay_factor= f;
}

void io_set_hex_on_serial(Int hex)
{
    io_hex_on_serial= hex;
}

void io_make_stdin_unbuffered(void)
{
}

void io_sigint_interrupt(void)
{
}

/*******************/
/* SERIAL ROUTINES */
/*******************/

IOStream *io_open_serial(char *name)
{
    Int e1, e2;
    IOStream *ret= malloc(sizeof(IOStream));
    if (!stricmp(name, "printer")) {
    	e1= OpenDriver("\005.BOut", &ret->ostream);
    	e2= OpenDriver("\004.BIn", &ret->istream);
    }
    else if (!stricmp(name, "modem")) {
    	e1= OpenDriver("\005.AOut", &ret->ostream);
    	e2= OpenDriver("\004.AIn", &ret->istream);
    }
    else {
    	return 0;
    }
    
    if (!ret->istream) {
    	printf("e1: %d, e2: %d\n", e1, e2);
    	exit(1);
    }
    
    if (io_debug) printf("driver # %d,%d\n", ret->istream, ret->ostream);
    return ret;
}

IOStream *io_open_stdin(void)
{
    IOStream *ret= malloc(sizeof(IOStream));
    ret->stream= 0;
    return ret;
}

Int io_mac_getbaud(long baud)
{
    switch (baud) {
      case 300:	return 380;
      case 600:   return 189;
      case 1200:  return 94;
      case 2400:  return 46;
      case 4800:  return 22;
      case 9600:  return 10;
      case 19200: return 4;
      case 38400: return 0;
      default:
	printf("Illegal baud rate %ld in getbaud\n", baud);
	exit(1);
    }
}

void io_serial_init(long baud, IOStream *s)
{
    Int init= io_mac_getbaud(baud) + 0 + 3072 + 16384;
    if (io_debug) printf("SerReset(%d, %d)\n", s->istream, init);
    SerReset(s->istream, init);
    if (io_debug) printf("SerReset(%d, %d)\n", s->ostream, init);
    SerReset(s->ostream, init);
    if (io_debug) printf("serial init finished\n");
    io_discard_input(s);
}

Int io_getchar(IOStream *s, Int timeout)
{
    IOStream *streams[2];
    Int which;
    if (io_debug) printf("enter io_getchar\n");
    streams[0]= s;
    streams[1]= 0;
    return io_getchar_multiple_inputs(streams, &which, timeout);
}

void io_putchar(Int c, IOStream *s)
{
    long to_write= 1;
    unsigned char buf= c;
    if (s->ostream) {
    	if (io_debug) printf("FSWrite(%d, %ld, [%d])\n", 
			     s->ostream, to_write, (Int) buf);
    	FSWrite(s->ostream, &to_write, &buf);
    	if (to_write != 1) {
	    die(("Error on write in io_putchar"));
    	}
    }
    else {
    	die(("Illegal output stream in io_putchar\n"));
    }
}

Int io_getchar_nowait(IOStream *s)
{
    unsigned char buf;
    if (io_debug) printf("entering io_getchar_nowait\n");
    if (s->istream) {
	long pending;
	if (io_debug) printf("SerGetBuf(%d,&pending)", s->istream);
	SerGetBuf(s->istream, &pending);
	if (io_debug) printf(" -> pending = %ld\n", pending);
	if (pending > 0) {
	    pending= 1;
	    FSRead(s->istream, &pending, &buf);
	    if (pending != 1) die(("error reading serial\n"));
	    if (io_debug) printf("returning char %d\n", (Int) buf);
	    return 0xff & (Int) buf;
	} else {
	    if (io_debug) printf("no char, returning EOF\n");
	    return EOF;
	}
    }
    else if (s->stream == 0) {
	if (io_debug) printf("calling getchar()\n");
	return getchar();
    }
    else {
	printf("bad input stream in getchar_nowait\n");
	exit(1);
    }
}

Int io_getchar_multiple_inputs(IOStream **inputs, Int *which, Int timeout)
{
    Int i, j= 0;
    Int c;
    unsigned long first_tick= clock();
    unsigned long last_tick= first_tick + 1 + timeout/17;
    if (io_debug) printf("get serial until %ld\n", last_tick);
    
    do {
	for (j= 0; inputs[j]; j++) {
	    c= io_getchar_nowait(inputs[j]);
	    if (c != EOF) {
		*which= j;
		return c;
	    }
	}
	if (io_debug) printf("getchar %ld\n", clock());
    } while (clock() <= last_tick);
    return EOF;
}

void io_discard_input(IOStream *s)
{
    while (io_getchar(s, 0) != EOF);
}

void msleep(Int msec)
{
    unsigned long first_tick_count= clock();
    unsigned long last_tick_count= first_tick_count + 1 + msec / 17L;
    if (io_debug) printf("msleep till %ld\n", last_tick_count);
    while (clock() <= last_tick_count) 
      if (io_debug) printf("msleep %ld\n", clock());
}

#ifdef STAND_ALONE
NOPROTO main()
{
    IOStream *kbd= io_open_stdin();
    IOStream *ser= io_open_serial(io_default_serial_name);
    IOStream *streams[3];
    Int i;
    
    streams[0]= kbd;
    streams[1]= ser;
    streams[2]= 0;
    
    for (i= 0; i< 3; i++) {
	Int c, which;
	
	c= io_getchar_multiple_inputs(streams, &which, 5000);
	printf("Got a char %d >%c< from %d\n", c, c, which);
    }
    
}
#endif









