/*
 * Copyright (c) 1980, 1987, 1988 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
/*
 *  sccsid[] = "@(#)login.c	5.40 (Berkeley) 5/9/89";
 *
 *  Ported to HP/UX by Michael Glad <glad@daimi.dk>
 *  Ported to Linux by <poe@daimi.aau.dk>
 *  Ported to Athena by Sal Valente <svalente@athena.mit.edu>
 */

#define __USE_BSD_SIGNAL

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/stat.h>
#include <time.h>
#include <termios.h>
#include <signal.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include <setjmp.h>

#include <sys/file.h>
#include <sys/syslog.h>
#include <sys/resource.h>
#include <sys/sysmacros.h>
#include <utmp.h>
#include <paths.h>
#include <getopt.h>
#include "pathnames.h"

#define TTYGRPNAME      "other"
#ifndef MAXPATHLEN
#define MAXPATHLEN	1024
#endif

/*
 * This bounds the time given to login.  Not a define so it can
 * be patched on machines where it's too small.
 */
int	timeout = 300;

struct	passwd *pwd;
int	failures;
char	*hostname, *username, *tty;
char	thishost[100];

int	errorprtflag = 0;
int	remote_login;

#ifdef ATHENA
extern int kerberos_user;
extern int hesiod_user;
extern int afs_user;

int athena_login (struct passwd **pwdp, char *username, char *password);
void athena_get_homedir (struct passwd *pwd);
void athena_init_wgfile (void);
void athena_do_fork (struct passwd *pwd);
#endif

void getloginname (void);
void timedout(int sig);
void init_tty (char *ttyn);
void opentty(const char * tty);
int  rootterm(char *ttyn);
void motd(void);
void sigint(int sig);
void checknologin(void);
void write_utmp (char *username, char *hostname, char *tty);
void dolastlog(int quiet);
void badlogin(char *name);
void checktty(char *user, char *tty);
void sleepexit(int eval);

int main(int argc, char **argv)
{
	extern int optind;
	extern char *optarg, **environ;
	struct group *gr;
	register int ch;
	register char *p;
	int ask, fflag, hflag, pflag, cnt, good;
	int quietlog, passwd_req;
	char *domain, *salt, *ttyn, *pp, *termenv;
	char tbuf[MAXPATHLEN + 2], tname[sizeof(_PATH_TTY) + 10];

	signal(SIGALRM, timedout);
	alarm((unsigned int)timeout);
	signal(SIGQUIT, SIG_IGN);
	signal(SIGINT, SIG_IGN);

	setpriority(PRIO_PROCESS, 0, 0);
#ifdef HAVE_QUOTA
	quota(Q_SETUID, 0, 0, 0);
#endif

	/*
	 * -p is used by getty to tell login not to destroy the environment
 	 * -f is used to skip a second login authentication 
	 * -h is used by other servers to pass the name of the remote
	 *    host to login so that it may be placed in utmp and wtmp
	 */
	gethostname(tbuf, sizeof(tbuf));
	strncpy(thishost, tbuf, sizeof(thishost)-1);
	domain = strchr(tbuf, '.');

	username = NULL;
	hostname = NULL;
	tty = NULL;
	pwd = NULL;
	remote_login = 0;
	failures = 0;

	fflag = hflag = pflag = 0;
	passwd_req = 1;
	while ((ch = getopt(argc, argv, "fh:p")) != EOF)
		switch (ch) {
		case 'f':
			fflag = 1;
			break;

		case 'h':
			if (getuid()) {
				fprintf(stderr,
				    "login: -h for super-user only.\n");
				exit(1);
			}
			hflag = 1;
			if (domain && (p = strchr(optarg, '.')) &&
			    strcasecmp(p, domain) == 0)
				*p = 0;
			hostname = optarg;
			remote_login = 1;
			break;

		case 'p':
			pflag = 1;
			break;
		case '?':
		default:
			fprintf(stderr,
			    "usage: login [-fp] [username]\n");
			exit(1);
		}
	argc -= optind;
	argv += optind;
	if (*argv) {
		username = *argv;
		ask = 0;
	} else
		ask = 1;

	for (cnt = sysconf(_SC_OPEN_MAX); cnt > 2; cnt--)
		close(cnt);

	ttyn = ttyname(0);
	if (ttyn == NULL || *ttyn == '\0') {
		sprintf(tname, "%s??", _PATH_TTY);
		ttyn = tname;
	}

#ifndef DEBUG
	setpgrp();
	init_tty(ttyn);
#endif

	tty = ttyn;
	if (! strncmp (ttyn, "/dev/", 5))
		tty = ttyn + 5;

	/* preserve TERM even without -p flag */
	termenv = getenv("TERM");
	if (! termenv) termenv = "dumb";

	/* destroy environment unless user has requested preservation */
	if (!pflag) {
		environ = (char**)malloc(sizeof(char*));
		environ[0] = NULL;
	}

	openlog("login", LOG_ODELAY, LOG_AUTH);

	for (cnt = 0; 1; ask++) {
		if (ask) {
			fflag = 0;
			getloginname();
		}

		checktty(username, tty);

		strcpy(tbuf, username);
		if ((pwd = getpwnam(username)) != NULL)
			salt = pwd->pw_passwd;
		else
			salt = "xx";

		/* if user not super-user, check for disabled logins */
		if (pwd == NULL || pwd->pw_uid)
			checknologin();

		/*
		 * Disallow automatic login to root; if not invoked by
		 * root, disallow if the uid's differ.
		 */
		if (fflag && pwd) {
			int uid = getuid();

			passwd_req = pwd->pw_uid == 0 ||
			    (uid && uid != pwd->pw_uid);
		}

		/*
		 * If trying to log in as root, but with insecure terminal,
		 * refuse the login attempt.
		 */
		if (pwd && pwd->pw_uid == 0 && !rootterm(tty)) {
			fprintf(stderr,
			    "%s login refused on this terminal.\n",
			    pwd->pw_name);

			if (hostname)
				syslog(LOG_NOTICE,
				    "LOGIN %s REFUSED FROM %s ON TTY %s",
				    pwd->pw_name, hostname, tty);
			else
				syslog(LOG_NOTICE,
				    "LOGIN %s REFUSED ON TTY %s",
				     pwd->pw_name, tty);
			continue;
		}

		/*
		 * If no pre-authentication and a password exists
		 * for this user, prompt for one and verify it.
		 */
		if (!passwd_req || (pwd && !*pwd->pw_passwd))
			break;

  		setpriority(PRIO_PROCESS, 0, -4);
		pp = getpass("Password: ");
		setpriority(PRIO_PROCESS, 0, 0);
#ifndef ATHENA
		p = crypt(pp, salt);
		good = (pwd && !strcmp(p, pwd->pw_passwd));
#else
		good = athena_login (&pwd, username, pp);
#endif
		memset(pp, 0, strlen(pp));
		if (good) break;

		if (! errorprtflag) printf("Login incorrect\n");
		failures++;
		badlogin(username); /* log ALL bad logins */

		/* we allow 10 tries, but after 3 we start backing off */
		if (++cnt > 3) {
			if (cnt >= 10) {
				sleepexit(1);
			}
			sleep((unsigned int)((cnt - 3) * 5));
		}
	}

	/* committed to login -- turn off timeout */
	alarm((unsigned int)0);

#ifdef HAVE_QUOTA
	if (quota(Q_SETUID, pwd->pw_uid, 0, 0) < 0 && errno != EINVAL) {
		switch(errno) {
		case EUSERS:
			(void)fprintf(stderr,
		"Too many users logged on already.\nTry again later.\n");
			break;
		case EPROCLIM:
			(void)fprintf(stderr,
			    "You have too many processes running.\n");
			break;
		default:
			perror("quota (Q_SETUID)");
		}
		sleepexit(0);
	}
#endif

#ifdef ATHENA
	if (kerberos_user || hesiod_user || afs_user)
	    athena_do_fork (pwd);
	athena_init_wgfile ();
	if (! kerberos_user)
	    puts ("Warning: no Kerberos tickets obtained.");
	athena_get_homedir (pwd);
#endif

#ifdef ATHENA
	/* 
	  The effective uid is used under AFS for access.
	  NFS uses euid and uid for access checking
	 */
	seteuid(pwd->pw_uid);
#endif
	if (chdir(pwd->pw_dir) < 0) {
		printf("No directory %s!\n", pwd->pw_dir);
		if (chdir("/"))
			exit(0);
		pwd->pw_dir = "/";
		printf("Logging in with home = \"/\".\n");
	}
#ifdef ATHENA
	seteuid(getuid());
#endif
	quietlog = access(_PATH_HUSHLOGIN, 004) == 0;

#ifndef DEBUG
	write_utmp(username, hostname, tty);
	dolastlog(quietlog);
#endif
	
	chown(ttyn, pwd->pw_uid,
	    (gr = getgrnam(TTYGRPNAME)) ? gr->gr_gid : pwd->pw_gid);
	chmod(ttyn, 0622);
	setgid(pwd->pw_gid);

	initgroups(username, pwd->pw_gid);

#ifdef HAVE_QUOTA
	quota(Q_DOWARN, pwd->pw_uid, (dev_t)-1, 0);
#endif

	if (*pwd->pw_shell == '\0')
		pwd->pw_shell = _PATH_BSHELL;

        setenv("HOME", pwd->pw_dir, 0);      /* legal to override */
        if(pwd->pw_uid) setenv("PATH", _PATH_DEFPATH, 1);
        else setenv("PATH", _PATH_DEFPATH_ROOT, 1);
	setenv("SHELL", pwd->pw_shell, 1);
	setenv("TERM", termenv, 1);

        /* mailx will give a funny error msg if you forget this one */
        sprintf(tbuf,"%s/%s",_PATH_MAILDIR,pwd->pw_name);
        setenv("MAIL",tbuf,0);

        /* LOGNAME is not documented in login(1) but
	   HP-UX 6.5 does it. We'll not allow modifying it.
	 */
	setenv("LOGNAME", pwd->pw_name, 1);

	if (pwd->pw_uid == 0)
		if (hostname)
			syslog(LOG_NOTICE, "ROOT LOGIN ON %s FROM %s",
			    tty, hostname);
		else
			syslog(LOG_NOTICE, "ROOT LOGIN ON %s", tty);

	if (!quietlog) {
		struct stat st;

		motd();
		sprintf(tbuf, "%s/%s", _PATH_MAILDIR, pwd->pw_name);
		if (stat(tbuf, &st) == 0 && st.st_size != 0)
			printf("You have %smail.\n",
			    (st.st_mtime > st.st_atime) ? "new " : "");
	}

	signal(SIGALRM, SIG_DFL);
	signal(SIGQUIT, SIG_DFL);
	signal(SIGINT, SIG_DFL);
	signal(SIGTSTP, SIG_IGN);
	signal(SIGHUP, SIG_DFL);

	tbuf[0] = '-';
	strcpy(tbuf + 1, (p = strrchr(pwd->pw_shell, '/')) ?
	    p + 1 : pwd->pw_shell);

	/* discard permissions last so can't get killed and drop core */
#ifndef DEBUG
	if(setuid(pwd->pw_uid) < 0 && pwd->pw_uid) {
	    syslog(LOG_ALERT, "setuid() failed");
	    exit(1);
	}
#endif

	execlp(pwd->pw_shell, tbuf, (char *)0);
	perror ("login: no shell");
	exit(0);
}

void getloginname (void)
{
    int ch;
    char *p;
    static char nbuf[UT_NAMESIZE + 1];

    while (1) {
	printf("\n%s login: ", thishost);
	fflush(stdout);
	for (p = nbuf; (ch = getchar()) != '\n'; ) {
	    if (ch == EOF) {
		badlogin(username);
		exit(0);
	    }
	    if (p < nbuf + UT_NAMESIZE)
		*p++ = ch;
	}
	if (p > nbuf)
	    if (nbuf[0] == '-')
		fprintf(stderr, "login names may not start with '-'.\n");
	    else {
		*p = '\0';
		username = nbuf;
		break;
	    }
    }
}

void timedout(int sig)
{
    struct termios ti;

    fprintf(stderr, "Login timed out after %d seconds\n", timeout);
    /* reset echo */
    tcgetattr (0, &ti);
    ti.c_lflag |= ECHO;
    tcsetattr (0, TCSANOW, &ti);
}

void init_tty (char *ttyn)
{
    struct termios tt, ttt;

    tcgetattr(0, &tt);
    ttt = tt;
    ttt.c_cflag &= ~HUPCL;

    if((chown(ttyn, 0, 0) == 0) && (chmod(ttyn, 0622) == 0)) {
	tcsetattr(0,TCSAFLUSH,&ttt);
	signal(SIGHUP, SIG_IGN); /* so vhangup() wont kill us */
	vhangup();
	signal(SIGHUP, SIG_DFL);
    }

    setsid();

    /* re-open stdin,stdout,stderr after vhangup() closed them */
    /* if it did, after 0.99.5 it doesn't! */
    opentty(ttyn);
    tcsetattr(0,TCSAFLUSH,&tt);
}

void opentty(const char * tty)
{
    int i;
    int fd = open(tty, O_RDWR);

    for (i = 0 ; i < fd ; i++)
      close(i);
    for (i = 0 ; i < 3 ; i++)
      dup2(fd, i);
    if (fd >= 3)
      close(fd);
}

int rootterm(char *ttyn)
{
    FILE *fp;
    char buf[100];
    int len;

    fp = fopen (_PATH_SECURETTY, "r");
    if (! fp) return (1);

    /* read each line in /etc/securetty, if a line matches our ttyline
       then root is allowed to login on this tty, and we should return
       true. */
    while (fgets (buf, sizeof (buf), fp) != NULL) {
	len = strlen (buf);
	if (buf[len - 1] == '\n') buf[len - 1] = 0;
	if (! strcmp (buf, tty)) return (1);
    }
    fclose (fp);
    return (0);
}

jmp_buf motdinterrupt;

void motd(void)
{
	register int fd, nchars;
	void (*oldint)(), sigint();
	char tbuf[8192];

	if ((fd = open(_PATH_MOTDFILE, O_RDONLY, 0)) < 0)
		return;
	oldint = signal(SIGINT, sigint);
	if (setjmp(motdinterrupt) == 0)
		while ((nchars = read(fd, tbuf, sizeof(tbuf))) > 0)
			write(fileno(stdout), tbuf, nchars);
	signal(SIGINT, oldint);
	close(fd);
}

void sigint(int sig)
{
	longjmp(motdinterrupt, 1);
}

void checknologin(void)
{
	register int fd, nchars;
	char tbuf[1024];

	if ((fd = open(_PATH_NOLOGIN, O_RDONLY, 0)) >= 0) {
		while ((nchars = read(fd, tbuf, sizeof(tbuf))) > 0)
			write(fileno(stdout), tbuf, nchars);
		sleepexit(0);
	}
}

void write_utmp (char *username, char *hostname, char *tty)
{
    struct utmp ut;
    char *ttyabbrev;
    int wtmp;

    memset((char *)&ut, 0, sizeof(ut));
    ut.ut_type = USER_PROCESS;
    ut.ut_pid = getpid();
    strncpy(ut.ut_line, tty, sizeof(ut.ut_line));
    ttyabbrev = tty + sizeof("tty") - 1;
    strncpy(ut.ut_id, ttyabbrev, sizeof(ut.ut_id));
    time(&ut.ut_time);
    strncpy(ut.ut_user, username, sizeof(ut.ut_user));

    /* fill in host and ip-addr fields when we get networking */
    if (hostname)
	strncpy(ut.ut_host, hostname, sizeof(ut.ut_host));

    utmpname(_PATH_UTMP);
    setutent();
    pututline(&ut);
    endutent();

    if((wtmp = open(_PATH_WTMP, O_APPEND|O_WRONLY)) >= 0) {
	flock(wtmp, LOCK_EX);
	write(wtmp, (char *)&ut, sizeof(ut));
	flock(wtmp, LOCK_UN);
	close(wtmp);
    }
    /* fix_utmp_type_and_user(username, ttyn, LOGIN_PROCESS); */
}

struct  lastlog {
    long ll_time;
    char ll_line[12];
    char ll_host[16];
};

void dolastlog(int quiet)
{
    struct lastlog ll;
    int fd;

    if ((fd = open(_PATH_LASTLOG, O_RDWR, 0)) >= 0) {
	lseek(fd, (off_t)pwd->pw_uid * sizeof(ll), SEEK_SET);
	if (!quiet) {
	    if (read(fd, (char *)&ll, sizeof(ll)) == sizeof(ll) &&
		ll.ll_time != 0) {
		printf("Last login: %.*s ",
		       24-5, ctime(&ll.ll_time));
		if (*ll.ll_host != '\0')
		    printf("from %.*s\n",
			   sizeof(ll.ll_host), ll.ll_host);
		else
		    printf("on %.*s\n",
			   sizeof(ll.ll_line), ll.ll_line);
	    }
	    lseek(fd, (off_t)pwd->pw_uid * sizeof(ll), SEEK_SET);
	}
	memset((char *)&ll, 0, sizeof(ll));
	time(&ll.ll_time);
	strncpy(ll.ll_line, tty, sizeof(ll.ll_line));
	if (hostname)
	    strncpy(ll.ll_host, hostname, sizeof(ll.ll_host));
	write(fd, (char *)&ll, sizeof(ll));
	close(fd);
    }
}

void badlogin(char *name)
{
	if (failures == 0)
		return;

	if (hostname)
		syslog(LOG_NOTICE, "%d LOGIN FAILURE%s FROM %s, %s",
		    failures, failures > 1 ? "S" : "", hostname, name);
	else
		syslog(LOG_NOTICE, "%d LOGIN FAILURE%s ON %s, %s",
		    failures, failures > 1 ? "S" : "", tty, name);
}

void checktty(char *user, char *tty)
{
    FILE *f;
    char buf[256];
    char *ptr;
    char devname[50];
    struct stat stb;

    /* no /etc/usertty, default to allow access */
    if(!(f = fopen(_PATH_USERTTY, "r"))) return;

    while(fgets(buf, 255, f)) {

	/* strip comments */
	for(ptr = buf; ptr < buf + 256; ptr++) 
	  if(*ptr == '#') *ptr = 0;

	strtok(buf, " \t");
	if(strncmp(user, buf, 8) == 0) {
	    while((ptr = strtok(NULL, "\t\n "))) {
		if(strncmp(tty, ptr, 10) == 0) {
		    fclose(f);
		    return;
		}
		if(strcmp("PTY", ptr) == 0) {
#ifdef linux
		    sprintf(devname, "/dev/%s", ptr);
		    /* VERY linux dependent, recognize PTY as alias
		       for all pseudo tty's */
		    if((stat(devname, &stb) >= 0)
		       && major(stb.st_rdev) == 4 
		       && minor(stb.st_rdev) >= 192) {
			fclose(f);
			return;
		    }
#endif
		}
	    }
	    /* if we get here, /etc/usertty exists, there's a line
	       beginning with our username, but it doesn't contain the
	       name of the tty where the user is trying to log in.
	       So deny access! */
	    fclose(f);
	    printf("Login on %s denied.\n", tty);
	    badlogin(user);
	    sleepexit(1);
	}
    }
    fclose(f);
    /* users not mentioned in /etc/usertty are by default allowed access
       on all tty's */
}

void sleepexit(int eval)
{
	sleep((unsigned int)5);
	exit(eval);
}
