/*
 * KCLL -  Ken and Chee's Limey Lisp  
 * All Rights to the code and any products created with this code is 
 * hereby granted. 
 *
 * I.E. You can do whatever the hell you want with this code. 
 * The only restriction is that this copyright notice not be modified.
 */
/* 
  Limey Byte code compiler
*/

#include "comp.h"
#include "obj.h"
#include "vector.h"
#include "cont.h"
#include "error.h"
#include "lbc.h"

#define MAXOBJECTS 1000		/* Maximum no. objects in a single lambda */
#define MAXCODESIZE 5000	/* Maximum no. byte code instructions in a single lambda */
#define MAXLABELS 1000		/* Maximum no. of lables */
#define MAXLABELREFS 2000	/* Maximum no. of label references */

static LLObj *objects[MAXOBJECTS];
static int numobjects;
static unsigned char bytecode[MAXCODESIZE]; /* Indexed by PC */
static int pc;			/* Instruction pointer */
static int labels[MAXLABELS];	/* Indexed by label, holds PC that label resolves to */
static int numlabels;
static int labelrefs[MAXLABELREFS]; /* Just a list of PC's where label references are */
static int numlabelrefs;

typedef unsigned char Instr;

#define DISCARD (Instr) 0210	/* Discard top of stack */
#define DUP (Instr) 0211	/* Duplicate top of stack */
#define CONSTANT (Instr) 0300	/* Push object #(opcode - 0300) from objects vector  */
#define CONSTANT2 (Instr) 0201	/* Push object #(following-short) from objects vector */
#define VARREF (Instr) 010	/* Push value of variable onto stack */
#define VARSET (Instr) 020	/* Set value of variable */
#define VARBIND (Instr) 030	/* Add new variable to current environment */
#define COMPILE (Instr) 0170	/* Invoke compiler on lambda or delay expression on top of stack */
#define CREATE_ENV (Instr) 0171	/* Introduce a new environment for VARBIND */
#define RESTORE_ENV (Instr) 0172 /* Toss current environment (switch to parent) */
#define PUSH_ENV (Instr) 0173	/* Push the current environment onto the stack */
#define CALL_WCC (Instr) 0174	/* Call with current continuation */
#define RETURN (Instr) 0175	/* Return to parent continuation */
#define RETURN_C (Instr) 0176	/* RETURN and exit C function llrun_byte_code as well */
#define CALL (Instr) 040	/*  */
#define CAR (Instr) 0100	/* Yes, these are going to be special forms ! */
#define CDR (Instr) 0101	/*  */
#define CONS (Instr) 0102	/*  */

#define GOTO (Instr) 0202	/*  */
#define GOTO_IF_NIL (Instr) 0203 /*  */
#define GOTO_IF_NOT_NIL (Instr) 0204 /*  */
#define GOTO_IF_NIL_ELSE_POP (Instr) 0205 /*  */
#define GOTO_IF_NOT_NIL_ELSE_POP (Instr) 0206 /*  */


static void compile_form();	
static void compile_forms();

static int add_object (obj)
     LLObj *obj;
{
  if (numobjects == MAXOBJECTS)
    llerror(LLCOMP_OUT_OF_SPACE);
  objects[numobjects] = obj;
  return numobjects++;
}

static int obj_to_index (obj)
     LLObj *obj;
{
  int n;
  
  for (n = 0; n < numobjects; n++)
    if (llcompare(LLEQ, objects[n], obj) != NIL)
      return n;
  n = add_object(obj);
  return n;
}

static int add_instruction(b)
     Instr b;
{
  if (pc == MAXCODESIZE)
    llerror(LLCOMP_OUT_OF_SPACE);
  bytecode[pc] = b;
  return pc++;
}

static void add_narged_instruction (b, n)
     Instr b;
     int n;
{
  if (n < 6) 
    add_instruction(b+n);
  else if (n < 0x100) {
    add_instruction(b+6);
    add_instruction((Instr) n);
  } else {
    add_instruction(b+7);
    add_instruction((Instr) (n & 0xff));
    add_instruction((Instr) (n >> 8));
  }
}

/* Allocates a new label */
static int new_label()
{
  if (numlabels == MAXLABELS)
    llerror(LLCOMP_OUT_OF_SPACE);
  return numlabels++;
}

/* When you've discovered where a label exists */
static void set_label(label)
     int label;
{
  labels[label] = pc;
}

/* Add a reference to a label at current pc */
static void add_label_ref(label)
     int label;
{
  if (numlabelrefs == MAXLABELREFS)
    llerror(LLCOMP_OUT_OF_SPACE);
  labelrefs[numlabelrefs++] = pc;
  add_instruction((Instr) (label & 0xff));
  add_instruction((Instr) (label << 8));
}

static void add_goto (opcode, label)
     Instr opcode;
     int label;
{
  add_instruction(opcode);
  add_label_ref(label);
}

static void compile_constant(form) 
     LLObj *form;
{
  int n;

  n = obj_to_index(form);
  if (n + CONSTANT > 0xff) {
    add_instruction(CONSTANT2);
    add_instruction((Instr) n & 0xff);
    add_instruction((Instr) n >> 8);
  } else
    add_instruction((Instr) n + CONSTANT);
}

#define compile_varref(form)  add_narged_instruction(VARREF,  obj_to_index((form)))
#define compile_varset(form)  add_narged_instruction(VARSET,  obj_to_index((form)))
#define compile_varbind(form) add_narged_instruction(VARBIND, obj_to_index((form)))
#define compile_call(nargs)   add_narged_instruction(CALL,    (nargs))


/* This is very, very different than llcompile_lambda below.
   llcompile_lambda takes a lambda expression and produces byte
   code for it.  We can not do that here, because this lambda
   (or delay) is __imbedded__ in the form being compiled --- it 
   cannot be compiled until the outer lambda's byte code is being 
   interpreted! */
static void compile_compile(form)
     LLCons *form;
{
  compile_constant((LLObj *) form);
  add_instruction(COMPILE);
}

static void compile_define(orig_form)
     LLCons *orig_form;
{
  LLCons *todefine;
  LLCons* form;

  form = (LLCons *) llccdr(orig_form);
  todefine = (LLCons *) llccar(form);
  if (llis_cons(todefine)) {	/* (define (func argn ...) forms) */
    compile_form (llcmake_cons (llcstring_to_sym("lambda"), 
				llcmake_cons (llccdr(todefine), llccdr(form))));
    todefine = (LLCons *) llccar(todefine);
  } else 
    compile_form(llccar(llccdr(form)));
  if(llsymcmp(llccar(orig_form), "set!"))
    compile_varset((LLObj *) todefine);
  else 
    compile_varbind((LLObj *) todefine);
  compile_constant(todefine);
}

static void compile_while(theform)
     LLCons *theform;
{
  LLObj *form, *test;
  int start_label, done_label;

  form = llccdr(theform);
  if (llclength(form) < 1)
    llerror(LLBAD_FORM);
  test = llccar(form);
  form = llccdr(form);

  set_label(start_label = new_label());
  compile_form(test);
  add_goto(GOTO_IF_NIL_ELSE_POP, done_label = new_label());
  compile_forms(form);
  add_instruction(DISCARD);	/* This is never used, I guess */
  add_goto(GOTO, start_label);
  set_label(done_label);
}

static void compile_if(form)
     LLCons *form;
{
  int else_label, done_label;
  LLCons *else_part, *then_part;

  form = (LLCons *) llccdr(form);
  if (llclength(form) < 2) 
    llerror(LLBAD_FORM);
  compile_form(llccar(form));
  form = (LLCons *) llccdr(form);
  then_part = (LLCons *) llccar(form);
  else_part = (LLCons *) llccdr(form);
  if (else_part == (LLCons *) NIL) { /* No else part */
    add_goto(GOTO_IF_NIL_ELSE_POP, (done_label = new_label()));
    compile_form(then_part);
  } else {
    add_goto(GOTO_IF_NIL, (else_label = new_label()));
    compile_form(then_part);
    add_goto(GOTO, (done_label = new_label()));
    set_label(else_label);
    compile_forms(else_part);
  }
  set_label(done_label);
}

static void compile_cond(form) 
     LLCons *form;
{
  LLCons *cond, *forms;
  LLObj *test;
  int last_label, next_label;

  form = (LLCons *) llccdr(form); /* get past cond atom */
  if (llclength(form) < 0)	/* An empty COND is ok... */
    llerror(LLBAD_FORM);		/* But it's got to be a list! */
  last_label = new_label();
  while ((form != (LLCons *) NIL) &&
	 (lltypecheck((cond = (LLCons *) llccar(form)), llcons_t)) &&
	 (!(llis_sym(llccar(cond)) && llsymcmp(llccar(cond), "else")))) {
    if (llclength(cond) < 1) llerror(LLBAD_FORM);
    test = (LLObj *) llccar(cond);
    forms = (LLCons *) llccdr(cond);
    compile_form(test);
    switch (llclength(forms)) {
    case 0:
      add_goto(GOTO_IF_NOT_NIL_ELSE_POP, last_label);
      break;
    case -1:
      llerror(LLBAD_FORM);
    default:
      add_goto(GOTO_IF_NIL, (next_label = new_label()));
      compile_forms(forms); 
      add_goto(GOTO, last_label);
      set_label(next_label);
    }
    form = (LLCons *) llccdr(form);
  }
  if (form != (LLCons *) NIL) {	/* there's an else clause */
    forms = (LLCons *) llccdr(cond);
    if(llclength(forms) < 1) llerror(LLBAD_FORM); /* One form is required */
    compile_forms(forms);
  } else 
    compile_constant(NIL);
  set_label(last_label);
}

static void push_things(list)	/* Needs to push in REVERSE order... sigh */
     LLCons *list;
{
  if (list != (LLCons *) NIL) {
    push_things((LLCons *) llccdr(list));
    compile_form(llccar(list));
  }
}

static void compile_optimized_funcall(form, bcode, nargs)
     LLCons *form;
     Instr bcode;
     int nargs;
{
  form = (LLCons *) llccdr(form);
  if (llclength(form) != nargs)
    llerror(LLWRONG_NARGS);
  push_things(form);
  add_instruction(bcode);
}

static void compile_funcall(form, length)
     LLCons *form;
     int length;
{
  push_things(form);
  compile_call(length - 1);
}

static void compile_let_bindings(bindings)
     LLCons *bindings;		/* Assumed valid */
{
  LLObj *binding;

  if (bindings != (LLCons *) NIL) {
    binding = (LLObj *) llccar(bindings); 
    if (llis_sym(binding))	/* No value supplied --- 
				   NIL by default */
      compile_constant(NIL);
    else if (llclength(binding) == 2) { 
      compile_form (llccar(llccdr(binding)));
      binding = (LLObj *) llccar(binding);
    } else
      llerror(LLBAD_FORM);
				/* Now, get the rest of 
				   the bindings on the stack */
    compile_let_bindings((LLCons *) llccdr(bindings));
				/* Finally, do the varbind */
    compile_varbind(binding);
  }
}

static void compile_letstar_bindings(bindings)
     LLCons *bindings;
{
  LLObj *binding;

  while (bindings != (LLCons *) NIL) {
    binding = (LLObj *) llccar(bindings); 
    if (llis_sym(binding))	/* No value supplied --- 
				   NIL by default */
      compile_constant(NIL);
    else if (llclength(binding) == 2) { 
      compile_form (llccar(llccdr(binding)));
      binding = (LLObj *) llccar(binding);
    } else
      llerror(LLBAD_FORM);
				/* Do the varbind FIRST... */
    compile_varbind(binding);
				/* Now, do the rest of the bindings */
    bindings = (LLCons *) llccdr(bindings);
  }
}  

static void compile_letrec_bindings(bindings)
     LLCons *bindings;
{
  LLObj *binding;
  LLCons *bindingstart;

  bindingstart = bindings;
				/* First, bind them all to NIL */
  while (bindings != (LLCons *) NIL) {
    compile_constant(NIL);
    binding = (LLObj *) llccar(bindings); 
    if (!llis_sym(binding))
      if (llclength(binding) != 2 || (!llis_sym(binding = llccar(binding))))
	llerror(LLBAD_FORM);
    compile_varbind(binding);
    bindings = (LLCons *) llccdr(bindings);
  }
  bindings = bindingstart;
				/* Now, actually evaluate the 
				   initializers */
  while (bindings != (LLCons *) NIL) {
    binding = (LLObj *) llccar(bindings);
    if (llis_sym(binding))	/* No value supplied --- 
				   NIL by default */
      compile_constant(NIL);
    else {
      compile_form (llccar(llccdr(binding)));
      binding = (LLObj *) llccar(binding);
    }
    compile_varset(binding);
    bindings = (LLCons *) llccdr(bindings);
  }
}

typedef enum lettype {
  LET, LETSTAR, LETREC 
  } LLLettype;

static void compile_let(form, type)
     LLCons *form;
     LLLettype type;
{
  LLCons *bindings;

  if (llclength(form) < 2)
    llerror(LLBAD_FORM);
  bindings = (LLCons *) llccar(form = (LLCons *) llccdr(form));
  form = (LLCons *) llccdr(form);
  if (llclength(bindings) < 0)	/* Ensure LIST or NIL */
    llerror(LLBAD_FORM);
  add_instruction(CREATE_ENV);
  switch (type) {
  case LET: compile_let_bindings(bindings); break;
  case LETSTAR: compile_letstar_bindings(bindings); break;
  case LETREC: compile_letrec_bindings(bindings); break;
  }
  compile_forms(form);		/* List of forms... */
  add_instruction(RESTORE_ENV);	/* Throw away environment */
}

static void compile_constream(form)
{
  llccdr(llccdr(form)) = (LLObj*)llcmake_cons(llcstring_to_sym("delay"),
				      llccdr(llccdr(llccdr(form))));
  puts("test1");
  printf("%d is the length\n", llclength(llccdr(form)));
/*  llprint_obj(llccdr(form), s_stdout);*/
  compile_optimized_funcall
    (llccdr(form), CONS, 2);
}

static void compile_the_environment(form)
     LLCons *form;
{
  if (llclength(form) != 1)
    llerror(LLBAD_FORM);
  add_instruction(PUSH_ENV);
}

static void compile_call_with_current_continuation(form)
     LLCons *form;
{
  if (llclength(form) != 2)
    llerror(LLBAD_FORM);
  compile_form(llccar(llccdr(form)));
  add_instruction(CALL_WCC);
}

static void compile_cons(form)
     LLCons *form;
{
  LLSym *magic;			/* First word in a cons is the magic word */
  int n;
  
  n = llclength(form);
  if (n < 1)
    llerror(LLBAD_FORM);
  if (llis_sym(llccar(form))) {
    magic = (LLSym *) llccar(form);
    if (llsymcmp(magic, "lambda") ||
	llsymcmp(magic, "delay")) 
      compile_compile(form);
    else if (llsymcmp(magic, "define") ||
	     llsymcmp(magic, "set!"))
      compile_define(form);
    else if (llsymcmp(magic, "car"))
      compile_optimized_funcall(form, CAR, 1);
    else if (llsymcmp(magic, "cdr"))
      compile_optimized_funcall(form, CDR, 1);
    else if (llsymcmp(magic, "cons"))
      compile_optimized_funcall(form, CONS, 2);
    else if (llsymcmp(magic, "if"))
      compile_if(form);
    else if (llsymcmp(magic, "cond")) 
      compile_cond(form);
    else if (llsymcmp(magic, "quote"))
      compile_constant(llccar(llccdr(form)));
    else if (llsymcmp(magic, "let"))
      compile_let(form, LET);
    else if (llsymcmp(magic, "let*"))
      compile_let(form, LETSTAR);
    else if (llsymcmp(magic, "letrec"))
      compile_let(form, LETREC);
    else if (llsymcmp(magic, "while"))
      compile_while(form);
    else if (llsymcmp(magic, "cons-stream"))
      compile_constream(form);
    else if (llsymcmp(magic, "begin") ||
	     llsymcmp(magic, "sequence"))
      compile_forms(llccdr(form));
    else if (llsymcmp(magic, "the-environment"))
      compile_the_environment(form);
    else if (llsymcmp(magic, "call-with-current-continuation") ||
	     llsymcmp(magic, "call/cc"))
      compile_call_with_current_continuation(form);
    else
      compile_funcall(form, n);
  } else
    compile_funcall(form, n);
}

/* Byte compile a form into bytecode[pc].
   Leaves the result of the form on the stack. */
static void compile_form(form)	
     LLObj *form;
{
  if (llis_cons(form))
    compile_cons((LLCons *) form);
  else if (llis_sym(form))
    compile_varref(form);
  else
    compile_constant(form);
}

/* Byte compile a list of forms.  
   Leaves the result of the last one on the argument stack. */
static void compile_forms(forms)
     LLCons *forms;
{
  if (forms == (LLCons *) NIL)
    compile_constant(NIL);	/* No forms?  No value. */
  else
    while (forms != (LLCons *) NIL) {
      compile_form(llccar(forms));
      if ((forms = (LLCons *) llccdr(forms)) != (LLCons *) NIL)
	add_instruction(DISCARD); /* Throw away result of form 
				     if not the last */
    }
}

static resolve_labels()
{
  int n, *lr, pcval;
  
  for (n = numlabelrefs, lr = labelrefs;
       n;
       n--, lr++) {
    pcval = labels[bytecode[*lr] + (bytecode[*lr + 1] << 8)];
    bytecode[*lr] = (pcval & 0xff);
    bytecode[*lr + 1] = (pcval >> 8);
  }
}

static LLVector *make_object_vector()
{
  LLVector *vect;

  vect = llcmake_vector(numobjects);
  bcopy(objects, vect->vector, numobjects * sizeof(LLObj *));
				/* Yes, this is a gross abstraction */
				/* violation, but in the byte compiler */
				/* I think that's ok.  All abstractions */
				/* are pretty dead here anyway. */
  return vect;
}

static void begin_compile()
{
  pc = 0;
  numobjects = 0;
  numlabels = 0;
  numlabelrefs = 0;
}

/* Produce a closure/delay from a lambda/delay expression!
   The lambda expression is a list in the form:
   (lambda (arg1 arg2 ...) form1 form2 ...) */
LLObj *llcompile_lambda_or_delay (form, env)
     LLCons *form;		
     LLEnv *env;
{
  int nargs;			/* Number of arguments of lambda */
  LLCons *thelist, *argl;
  LLCons *forms;
  int lambda;			/* True if compiling a lambda */

  begin_compile();
  if (!(llis_cons(form)))
    llerror(LLBAD_LAMBDA);
  if (!((lambda = llsymcmp(llccar(form), "lambda")) ||
	llsymcmp(llccar(form), "delay")))
    llerror(LLBAD_LAMBDA);		/* First word must be LAMBDA ! */
  form = (LLCons *) llccdr(form);
  if (!(llis_cons(form)))
    llerror(LLBAD_LAMBDA);
  if (lambda) {
    thelist = argl = (LLCons *) llccar(form);
    forms = (LLCons *) llccdr(form);
    if (!llis_list(forms))
      llerror(LLBAD_LAMBDA);
    for (nargs = 0; 
	 argl != (LLCons *) NIL && llis_cons(argl); 
	 argl = (LLCons *) llccdr(argl), nargs++) 
      add_object(llccar(argl)); 
  } else {
    thelist = (LLCons *) NIL;
    /*nargs = 0;*/
    forms = form;
  }
  compile_forms(forms);
  add_instruction(RETURN);
  resolve_labels();
  if (lambda)
    return (LLObj *) 
      llmake_closure(llchars_to_string(bytecode, pc),
		     make_object_vector(),
		     thelist,
		     env);
  else
    return (LLObj *)
      llmake_delay(llchars_to_string(bytecode, pc),
		   make_object_vector(),
		   env);
}


/* This is essentially EVAL. */
void llcompile_and_run(expr, env, return_type)
     LLObj *expr;
     LLEnv *env;
     Instr return_type;
{
  LLString *string;
  LLVector *vector;

  begin_compile();
  compile_form(expr);
  add_instruction(return_type);
  resolve_labels();
  string = llchars_to_string(bytecode, pc);
  vector = make_object_vector();
  llccall_bytecode (string, vector, env);
}

/* This is a key entry point because it is the only thing that calls
   llrun_byte_code, the main loop of the byte code interpreter.  Also,
   it assumes that the current continuation is NULL. */
LLObj *lltop_level_form(expr)
     LLObj *expr;
{
  llcompile_and_run(expr, tlge, RETURN_C);
  return llrun_byte_code();
}

void llinit_comp()
{
}
