/*
 * KCLL -  Ken and Chee's Limey Lisp  
 * All Rights to the code and any products created with this code is 
 * hereby granted. 
 *
 * I.E. You can do whatever the hell you want with this code. 
 * The only restriction is that this copyright notice not be modified.
 */

/* Support for Objects 

Each scheme object has a C type of "Obj".  This structure (Obj) must
exist at the start of every scheme object.  The first two bytes in the
structure indicate the object's type.  This type number (called the
TAG) can be looked up in a public array of "Type Descriptors" called
object_tds.  Each type descriptor contains enough information to
completely describe the type:

   the size of the object;
   a function used for freeing the object, a function used to mark the
     object --- and all objects it refers to --- as "useful" (for garbage
     collection purposes);
   a function to print the object to a stream;
   and a function to determine if the next thing in a stream is of
      this type, and, if so, read it from the stream.
*/

#ifndef _OBJ_INC
#define _OBJ_INC

#include "error.h"

typedef short LLTag;		/* This C type is used to label 
				   objects so their scheme type can be 
				   determined run-time */

#define MAX_TDS 100		/* Maximum number of kinds of objects */

#define LLIS_OBJECT 		/* Macro used when defining new objects */ \
  LLTag _type			/* Tag identifying type of object 
				   (sign bit used by garbage collector) */

typedef struct llobj_s {		/* Generic object header.  You must 
				   include this at the beginning 
				   of every scheme object you create... 
				   use the LLIS_OBJECT macro. */
  LLIS_OBJECT;
} LLObj;

typedef struct obj_td_s {	/* Object Type Descriptor: 
				   Structure that defines an object type */
  LLIS_OBJECT;			/* Type descriptors are scheme objects too! */
  unsigned short size;		/* Size of static part of object, in bytes */
  char *name;			/* Name of the object type */
  void (*freefunc)();		/* Function to: free the object */
  void (*touchfunc)();		/*              touch the object for GC */
  void (*printfunc)();		/*              display the object */
  LLObj *(*readfunc)();		/*              read the object from a stream */
  int (*comparefunc)();		/*              EQ, EQV, EQUAL compare function */
} LLObj_td;

typedef enum llcompare_e {	/* Compare methods passed to the comparefunc */
  LLEQ,
  LLEQV,
  LLEQUAL
  } LLCompare;

extern LLObj_td *object_tds[MAX_TDS]; /* Array of pointers to object type descriptors */
extern LLTag llnull_t;		/* Tag of the NULL type -- which you should never use. */
LLTag lladd_obj_td ();
extern LLObj *NIL, *T;		/* Magic "NIL" and "T" objects */
				/* (NIL is actually much more magic than T) */
#define LLFALSE NIL		/* Eventually, these will be different */

LLObj *llcompare ();		/* Returns "T" or "LLFALSE" */
void llprint_obj ();
void llcall_touch_method();
void llcall_free_method();

/* Yes, this is slimy.  During GC, you have to mask off the high bit of 
   of the tag field because it's used to mark useful objects. */

#define llobj_tag_value(o) ((o)->_type & 0x7FFF)
#define llobj_tag_field(o) ((o)->_type)
#ifdef GC
#  define llobj_tag llobj_tag_value
#else
#  define llobj_tag llobj_tag_field
#endif
#define null(o) ((LLObj *) (o) == NIL)
#define lleq(o1, o2) (llcompare(LLEQ,(o1),(o2))!=LLFALSE)
#define lltouch_obj(obj) ((void) llcctouch_obj((LLObj *) (obj)))

#ifndef NULL
#define NULL 0
#endif

#endif
