#include "mysccs.h"
SCCSID("@(#)rtlink.c   %E%   SAP   %I%")

static char * this_File GNU_UNUSED = __FILE__;

/************************************************************************/
/* $Id: rtlink.c,v 1.2 1998/11/12 22:12:48 d019080 Exp $
 ************************************************************************/
/*
 *  (C) Copyright 1998  SAP AG Walldorf
 *
 * Author:  Martin Rex
 * 
 * SAP AG DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SAP AG BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 *
 * Providing a platform independent interface to load
 * "shared libraries" aka "shared objects" aka "DLLs"
 * dynamically at runtime.
 *
 * This code doesn't care whether you try to load a library
 * multiple times and it doesn't care about multi-threading ...
 *
 */


#include "common.h"
#include <errno.h>
#include "debug.h"
#include "rtlink.h"
#include "non_ansi.h"

#ifndef PATH_MAX
#  define LIBNAME_MAX    255
#else
#  define LIBNAME_MAX	 (PATH_MAX-1)
#endif

#define RTL_MAXLIBS   (9 + 1)



#ifdef RTLINK_APPLE_MAC

#  include <CodeFragments.h>
#  include <Gestalt.h>
#  include <OSUtils.h>
#  include <Traps.h>

   typedef struct UPPADMSTRUCT
   {
	RoutineDescriptor	*theUPP;
	struct UPPADMSTRUCT	*next;
   }	UPPADM;

#endif



struct rtl_shlib_s {
   char * libname;
   char * real_libname;
#if defined(RTLINK_APPLE_MAC)
   union {
      UPPADM	     * uppadm;
      void	     * ptr;
      Ulong	       val;
   } handle;
   CFragConnectionID   connid;  /* Connection ID of CodeFragment */
#else
   union {
      void    * ptr;
      Ulong	val;
   } handle;
#endif
};

#define ptr_handle     handle.ptr
#define val_handle     handle.val
#define ptr_uppadm     handle.uppadm

static struct rtl_shlib_s    rtl_shlib_adm[RTL_MAXLIBS];



#if defined(RTLINK_UNIX_DLOPEN)

   /******************************************************************/
   /*								     */
   /*	modern UNIX:    dlopen() / dlclose() / dlsym()		     */
   /*								     */
   /******************************************************************/

#  include <dlfcn.h>

#  define PLATFORM_SPECIFIC_DLOPEN(shlib)    unix_dlopen(shlib)
#  define PLATFORM_SPECIFIC_DLCLOSE(shlib)   unix_dlclose(shlib)
#  define PLATFORM_SPECIFIC_DLSYM(shlib, fname, attr, ptr)	  \
					     unix_dlsym(shlib, fname, ptr)
#ifndef DLOPEN_FLAGS
#  define DLOPEN_FLAGS (RTLD_NOW)
#endif

static int
unix_dlopen( struct rtl_shlib_s * shlib )
{
   void * ldrc;
   DEBUG_BEGIN(unix_dlopen)

   ldrc = dlopen( shlib->libname, DLOPEN_FLAGS );
   if ( ldrc==NULL ) {
      XDBG((D_ERR, "dlopen(\"%s\") FAILED:\n  \"%s\"\n",
		   shlib->libname, dlerror() ))
      return(RTLERR_LOAD_FAILED);
   }

   shlib->ptr_handle = ldrc;

   return(RTL_OK);

} /* unix_dlopen() */

/* ----- */

static int
unix_dlclose( struct rtl_shlib_s * shlib )
{
   void   * ldrc    = shlib->ptr_handle;
   DEBUG_BEGIN(unix_dlclose)

   if ( dlclose( ldrc )!= 0 ) {
      XDBG((D_ERR, "dlclose(\"%s\") FAILED:\n  \"%s\"\n",
		   shlib->libname, dlerror() ))
      return(RTLERR_UNLOAD_FAILED);
   }

   return(RTL_OK);

} /* unix_dlclose() */

/* ----- */

static int
unix_dlsym( struct rtl_shlib_s * shlib,
	    char * funcname, RTL_FUNC_ADR * pp_fptr )
{
   void  * ldrc = shlib->ptr_handle;
   DEBUG_BEGIN(unix_dlsym)

   *pp_fptr = (RTL_FUNC_ADR) dlsym(ldrc, funcname);
   if ( *pp_fptr == 0 ) {
      XDBG((D_INFO, "dlsym(\"%s\") failed:\n  \"%s\"\n",
		    funcname, dlerror() ))
      return(RTLERR_SYMBOL_FAILED);
   }

   return(RTL_OK);

} /* unix_dlsym() */


#elif defined(RTLINK_AIX_LOAD)

   /******************************************************************/
   /*								     */
   /*	AIX non-standard:    load(), unload(), nlist()		     */
   /*								     */
   /*   should work for AIX 3.2.5 and AIX 4.1.x			     */
   /*   in AIX 4.2 there is dlopen(), Wow!			     */
   /******************************************************************/

   /* This is so horribly stupid! */
   /* the calls are defined    int (*load)();          */
   /* and                      int unload( void * );   */

typedef int         (AIX_FUNC)();
typedef AIX_FUNC   * AIX_FUNC_PTR;

#include <nlist.h>

    
#  define PLATFORM_SPECIFIC_DLOPEN(shlib)    aix_load(shlib)
#  define PLATFORM_SPECIFIC_DLCLOSE(shlib)   aix_unload(shlib)
#  define PLATFORM_SPECIFIC_DLSYM(shlib, fname, attr, ptr)	  \
					     aix_nlist(shlib, fname, ptr)
static int
aix_load( struct rtl_shlib_s * shlib )
{
   AIX_FUNC_PTR ldrc;
   DEBUG_BEGIN(aix_load)

   ldrc = load( shlib->libname, 0, (char *) 0 );
   if ( ldrc==0 ) {
      XDBG((D_ERR, "load(\"%s\") FAILED:\n  \"%s\"\n",
		   shlib->libname, strerror(errno) ))
      return(RTLERR_LOAD_FAILED);
   }

   shlib->ptr_handle = (void *) ldrc;

   return(RTL_OK);

} /* aix_load() */

/* ----- */

static int
aix_unload( struct rtl_shlib_s * shlib )
{
   void  * ldrc;
   DEBUG_BEGIN(aix_unload)

   ldrc = (void *) shlib->ptr_handle;
   if ( unload( ldrc )!= 0 ) {
      XDBG((D_ERR, "unload(\"%s\") FAILED:\n  \"%s\"\n",
		   shlib->libname, strerror(errno) ))
      return(RTLERR_UNLOAD_FAILED);
   }

   return(RTL_OK);

} /* aix_unload() */

/* ----- */

/* AWWWW!  this looks so bogus and slow ... */
static int
aix_nlist( struct rtl_shlib_s * shlib,
	   char * funcname, RTL_FUNC_ADR * pp_fptr )
{
   struct nlist  nl[2];
   int           status;
   DEBUG_BEGIN(aix_nlist)


   nl[0]._n._n_name  = funcname;
   nl[0].n_value     = 0;
   nl[1]._n._n_name  = (char *)0;

   status = nlist( shlib->libname, nl );
   if ( status!=0  ||  nl[0].n_value == 0 ) {
      XDBG((D_INFO, "nlist(\"%s\") failed:\n  \"%s\"\n",
		    funcname, dlerror() ))
      return(RTLERR_SYMBOL_FAILED);
   }

   *pp_fptr = (RTL_FUNC_ADR) ( ((char *)shlib->ptr_handle) + nl[0].n_value );
   
   return(RTL_OK);

} /* aix_nlist() */


#elif defined(RTLINK_HPUX_SHLOAD)

   /******************************************************************/
   /*								     */
   /*	HP-UX non-standard:    shl_load(), shl_unload(),	     */
   /*			       shl_findsym()			     */
   /*								     */
   /*   should work for HP-UX 9.x an 10.x			     */
   /******************************************************************/

#include <dl.h>

#  define PLATFORM_SPECIFIC_DLOPEN(shlib)    hpux_shl_load(shlib)
#  define PLATFORM_SPECIFIC_DLCLOSE(shlib)   hpux_shl_unload(shlib)
#  define PLATFORM_SPECIFIC_DLSYM(shlib, fname, attr, ptr)	  \
				       hpux_shl_findsym(shlib, fname, ptr)

static int
hpux_shl_load( struct rtl_shlib_s * shlib )
{
   shl_t  ldrc;
   DEBUG_BEGIN(hpux_shl_load)

   ldrc = shl_load( shlib->libname, BIND_IMMEDIATE|BIND_VERBOSE, 0L );
   if ( ldrc==0 ) {
      XDBG((D_ERR, "shl_load(\"%s\") FAILED:\n  \"%s\"\n",
		   shlib->libname, strerror(errno) ))
      return(RTLERR_LOAD_FAILED);
   }

   shlib->ptr_handle = (void *) ldrc;

   return(RTL_OK);

} /* hpux_shl_load() */

/* ----- */

static int
hpux_shl_unload( struct rtl_shlib_s * shlib )
{
   shl_t   ldrc    = shlib->ptr_handle;
   DEBUG_BEGIN(hpux_shl_unload)

   if ( shl_unload( ldrc )!= 0 ) {
      XDBG((D_ERR, "shl_unload(\"%s\") FAILED:\n  \"%s\"\n",
		   shlib->libname, strerror(errno) ))
      return(RTLERR_UNLOAD_FAILED);
   }

   return(RTL_OK);

} /* hpux_shl_unload() */

/* ----- */

static int
hpux_shl_findsym( struct rtl_shlib_s * shlib,
		  char * funcname, RTL_FUNC_ADR * pp_fptr )
{
   shl_t   ldrc  = shlib->ptr_handle;
   void  * adr;
   DEBUG_BEGIN(hpux_shl_findsym)

   if ( shl_findsym( &ldrc, funcname, TYPE_PROCEDURE, &adr) != 0 ) {
      XDBG((D_INFO, "shl_findsym(\"%s\") failed:\n  \"%s\"\n",
		    funcname, strerror(errno) ))
      return(RTLERR_SYMBOL_FAILED);
   }

   *pp_fptr = (RTL_FUNC_ADR) adr;

   return(RTL_OK);

} /* hpux_shl_findsym() */


#elif defined(RTLINK_WINDOWS_LOADLIBRARY)

   /*******************************************************************/
   /*								      */
   /*	Microsoft Windows:    LoadLibrary(), FreeLibrary(),	      */
   /*			      GetProcAddress()			      */
   /*								      */
   /*  should work for 16-bit (Windows 3.1 and up) and 32-bit (Win32) */
   /*******************************************************************/

#  undef FAR
#  undef WINAPI
#  include <windows.h>

#  define PLATFORM_SPECIFIC_DLOPEN(shlib)    windows_loadlib(shlib)
#  define PLATFORM_SPECIFIC_DLCLOSE(shlib)   windows_freelib(shlib)
#  define PLATFORM_SPECIFIC_DLSYM(shlib, fname, attr, ptr)	  \
					windows_getproc(shlib, fname, ptr)

static void
windows_err( DWORD p_lasterr,  char * p_buf,  size_t p_buflen )
{
   DWORD    rval;
   char   * errmsg = NULL;
   char   * s;
   char     tmpmsg[128];
   Uint      i;
   DEBUG_BEGIN(windows_err)

   p_buf[0] = '\0';
   if ( p_buflen<5 )
      return;

   rval     = FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER
			     | FORMAT_MESSAGE_FROM_SYSTEM,
			     NULL, (DWORD) p_lasterr,
			     MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
			     (LPTSTR) &errmsg, 0, NULL );
   if ( rval==0 ) {

      sprintf(tmpmsg, "FormatMessage() failed for lasterr = 0x%08lx", p_lasterr);
      strncpy(p_buf, tmpmsg, p_buflen-1);
      p_buf[p_buflen-1] = '\0';

   } else {

      /* remove the stupid linebreaks from the messsage */
      for( i=0, s=errmsg ; *s!='\0' && i<(p_buflen-1) ; s++ ) {
	 if ( *s=='\r' ) { continue; }
	 if ( *s=='\n' ) {
	    p_buf[i++] = ' ';
	 } else {
	    p_buf[i++] = *s;
	 }
      }
      p_buf[i]= '\0';

      while( i>0 && p_buf[i-1]==' ' ) { i--; p_buf[i]='\0'; }

   }

   if ( errmsg!=NULL ) {
      LocalFree( (LPVOID) errmsg );
      errmsg = NULL;
   }

   return;

} /* windows_err() */

/* ---- */

static int
windows_loadlib( struct rtl_shlib_s * shlib )
{
   HINSTANCE   hInst;
   DWORD       lasterr;
   DWORD       len;
   char        tmpbuf[1024];
   char      * ptr = NULL;
   DEBUG_BEGIN(windows_loadlib)


   hInst = LoadLibrary( shlib->libname );
   if ( hInst==(HINSTANCE)0 ) {
      lasterr = GetLastError();
      windows_err( lasterr, tmpbuf, sizeof(tmpbuf)-1 );
      XDBG((D_ERR, "LoadLibrary(\"%s\") FAILED:\n"
		   "  (0x%08lx) = \"%s\"\n",
		   shlib->libname, (long)lasterr, tmpbuf ))
      return(RTLERR_LOAD_FAILED);
   }

   /******************************************************/
   /* Update the filename information for the loaded DLL */
   /* relative paths, PATH searching, and omitting the   */
   /* filename extension may still load successfully.    */
   /*                                                    */
   /* Try to find out which DLL was actually loaded and  */
   /* update the filename into our struct rtl_shlib_s    */
   /******************************************************/
   len = GetModuleFileName( (HMODULE)hInst, tmpbuf, sizeof(tmpbuf)-1 );
   if ( len>0 && len<LIBNAME_MAX ) {
      tmpbuf[len] = '\0';
      ptr = malloc( len + 1 );
      if ( ptr!=NULL ) {
	 memcpy( ptr, tmpbuf, len+1);
	 shlib->real_libname = ptr;
	 ptr = NULL;
      }
   }

   shlib->val_handle = (Ulong)hInst;

   return(RTL_OK);

} /* windows_loadlib() */

/* ----- */

static int
windows_freelib( struct rtl_shlib_s * shlib )
{
   HINSTANCE   hInst;
   DWORD       lasterr;
   char        tmpbuf[256];
   DEBUG_BEGIN(windows_freelib)

   hInst = (HINSTANCE) shlib->val_handle;
   if ( FreeLibrary( hInst )==0 ) {
      lasterr = GetLastError();
      windows_err( lasterr, tmpbuf, sizeof(tmpbuf)-1 );
      XDBG((D_ERR, "FreeLibrary(\"%s\") FAILED:\n"
		   "  (0x%08lx) = \"%s\"\n",
		   shlib->libname, (long)lasterr, tmpbuf ))
      return(RTLERR_LOAD_FAILED);
   }

   return(RTL_OK);

} /* windows_freelib() */

/* ----- */

static int
windows_getproc( struct rtl_shlib_s * shlib,
		 char * funcname, RTL_FUNC_ADR  * fptr )
{
   HINSTANCE   hInst;
   DWORD       lasterr;
   char        tmpbuf[256];
   DEBUG_BEGIN(windows_getproc)

   hInst = (HINSTANCE) shlib->val_handle;
   *fptr = (RTL_FUNC_ADR) GetProcAddress( hInst, funcname );
   if ( *fptr == 0 ) {
      lasterr = GetLastError();
      windows_err( lasterr, tmpbuf, sizeof(tmpbuf)-1 );
      XDBG((D_INFO, "GetProcAddress(\"%.128s\") failed:\n"
		    "  (0x%08lx) = \"%s\"\n",
		    shlib->libname, (long)lasterr, tmpbuf ))
      return(RTLERR_SYMBOL_FAILED);
   }

   return(RTL_OK);

} /* windows_getproc() */


#elif defined(RTLINK_APPLE_MAC)

   /*******************************************************************/
   /*								      */
   /*	Apple MacIntosh						      */
   /*	using Code Fragment Manager 				      */
   /*								      */
   /*								      */
   /*								      */
   /*******************************************************************/

#  define PLATFORM_SPECIFIC_DLOPEN(shlib)   mac_loadlib(shlib)
#  define PLATFORM_SPECIFIC_DLCLOSE(shlib)  mac_unloadlib(shlib)
#  define PLATFORM_SPECIFIC_DLSYM(shlib,fname, attr, ptr) \
					    mac_loadfunc(shlib,fname,attr,ptr)



#define sysEnvironsVersion  1           /* Version of Sysenvirons           */


/******************************************************************************
 * CtoPString()
 *
 *	converts a null-terimnated string into a pascal string.
 *	returns true if all chars of srcString fitted in destString.
 ******************************************************************************/
static Boolean
CtoPString( const char  * srcString, Str255 destString )
{
   register short	i;
   Str255		saveString;
	
   for (i = 0; (i < 256 && srcString[i] != 0); i++) {
      saveString[i + 1] = srcString[i];
   }

   saveString[0] = i;

   BlockMove(saveString, destString, saveString[0] + 1L );

   return (srcString[i] == 0);

} /* CtoPString() */


/******************************************************************************
 * TrapInstalled
 *
 *	Check whether a certain trap exists on this machine. For pre-Mac II
 *	machines, trap numbers only go up to 0x1FF.
 ******************************************************************************/
static Boolean
TrapInstalled( short theTrap )
{
   TrapType tType;
   short    numToolBoxTraps;
	
	 /* first determine the trap type		*/
				
   tType = (theTrap & 0x800) > 0 ? ToolTrap : OSTrap;
	
	 /* next find out how many traps there are	*/
				
   if (NGetTrapAddress( _InitGraf, ToolTrap) == NGetTrapAddress( 0xAA6E, ToolTrap)) {
      numToolBoxTraps = 0x200;
   } else {
      numToolBoxTraps = 0x400;
   }

	 /* check if the trap number is too big for the	*/
	 /* current trap table				*/
				
   if (tType == ToolTrap)
   {
      theTrap &= 0x7FF;
      if (theTrap >= numToolBoxTraps) {
	 theTrap = _Unimplemented;
      }
   }
	
	 /* the trap is implemented if its address is	*/
	 /* different from the unimplemented trap	*/
				
   return ( NGetTrapAddress( theTrap, tType)
	    != NGetTrapAddress(_Unimplemented, ToolTrap) );

} /* TrapInstalled() */


/******************************************************************************
 * GestaltIsPresent
 *
 *	Check whether Gestalt is present
 ******************************************************************************/

static Boolean
GestaltIsPresent(void)
{
   SysEnvRec	theWorld;		       /* System environment	*/
	
   SysEnvirons(sysEnvironsVersion, &theWorld); /* Check environment	*/

   if (theWorld.machineType<0) {
      return(false);
   } else {
      return(TrapInstalled(_Gestalt));
   }

} /* GestaltIsPresent() */


/******************************************************************************
 * CFMIsPresent
 *
 *	Check whether the Code Fragment Manager is present
 ******************************************************************************/
static Boolean
CFMIsPresent( void )
{
   if (GestaltIsPresent()) {
      long	response;
		
      return ((Gestalt(gestaltCFMAttr, &response) == noErr) &&
			(((response >> gestaltCFMPresent) & 1) != 0));
   }

   return false;

} /* CFMIsPresent() */


/* ----- */


static int
mac_loadlib( struct rtl_shlib_s  * shlib )
{
   OSErr		err;
   CFragConnectionID	connID;
   Ptr			libAddr;
   Str255		pLibName;
   Str255		errName;
   DEBUG_BEGIN(mac_loadlib)

   if ( !CFMIsPresent() ) {
      XDBG((D_ERR, "mac_loadlib(): Code Fragment Manager not present!\n"));
      return(RTLERR_LOAD_FAILED);
   }

   CtoPString(shlib->libname, pLibName);
   err = GetSharedLibrary( pLibName,

#  ifdef RTLINK_APPLE_MAC_68K
			   kMotorola68KCFragArch,
#  else
			   kPowerPCCFragArch,
#  endif
			   kPrivateCFragCopy,
			   &connID,
			   &libAddr,
			   errName );

   if ( err!=noErr ) {
      XDBG((D_ERR, "mac_loadlib(): GetSharedLibrary(%.150s) failed (%d)\n",
		   shlib->libname, (int) err ));
      return(RTLERR_LOAD_FAILED);
   }

   shlib->ptr_uppadm = NULL;
   shlib->connid     = connID;

   return(RTL_OK);

} /* mac_loadlib */

/* ----- */

static int
mac_unloadlib( struct rtl_shlib_s * shlib )
{
   OSErr      err;
   UPPADM   * uppadm;
   UPPADM   * next;
   DEBUG_BEGIN(mac_unloadlib)

   if ( !CFMIsPresent() ) {
      XDBG((D_ERR, "mac_unloadlib(): Code Fragment Manager not present!\n"));
      return(RTLERR_UNLOAD_FAILED);
   }

   err = CloseConnection( &(shlib->connid) );
   if ( err!=noErr ) {
      XDBG((D_ERR, "mac_unloadlib(): CloseConnection(%d) failed (%d)!\n",
		   (int) err));
      return(RTLERR_UNLOAD_FAILED);
   }

   for( uppadm = shlib->ptr_uppadm ; uppadm!=NULL ; uppadm = next ) {
      next = uppadm->next;
      DisposePtr( (Ptr) uppadm->theUPP );
      free( uppadm );
   }

   return(RTL_OK);

} /* mac_unloadlib() */

/* ----- */

static int
mac_loadfunc( struct rtl_shlib_s * shlib, char * funcname,
	      Uint32 attributes,  RTL_FUNC_ADR * fptr )
{
   OSErr	       err;
   Str255	       pFuncName;
   Ptr		       funcAddr  = 0;
   CFragSymbolClass    funcClass = 0;
   DEBUG_BEGIN(mac_loadfunc)

   if ( !CFMIsPresent() ) {
      XDBG((D_ERR, "mac_loadlib(): Code Fragment Manager not present!\n"));
      return(RTLERR_SYMBOL_FAILED);
   }

   CtoPString( funcname, pFuncName );
   err = FindSymbol( shlib->connid, pFuncName, &funcAddr, &funcClass );
   if ( err!=noErr ) {
      XDBG((V_INFO, "mac_loadlib(): FindSymbol(\"%.150s\") failed (%d)!\n",
		   funcname, (int) err ));
      return(RTLERR_SYMBOL_FAILED);
   }


#  ifdef  RTLINK_APPLE_MAC_68K

   if ( funcAddr==0 ) {

      XDBG((V_ERR, "mac_loadlib(): Strange FindSymbol(\"%.150s\") behaviour!\n",
		   funcname ));
      return(RTLERR_SYMBOL_FAILED);

   } else {

      RoutineDescriptor     funcDesc = BUILD_ROUTINE_DESCRIPTOR(0,0);
      RoutineDescriptor   * funcUPP  = NULL;
      UPPADM		  * uppadm   = NULL;

      funcDesc.routineRecords[0].procDescriptor = (ProcPtr) funcAddr;
      funcDesc.routineRecords[0].procInfo       = flags;
      funcDesc.routineRecords[0].ISA            = (kM68kISA | kCFM68kRTA);

      funcUPP = (RoutineDesciptor *) NewPtr( sizeof(RoutineDescriptor) );

      if (funcUPP!=0) {
	 *funcUPP = funcDesc;  /* copy the whole structure */

	 /* create new UPP control structure */
	 uppadm = malloc( sizeof(UPPADM) );

	 if ( uppadm==NULL ) {
	    DisposePtr( (Ptr) funcUPP );
	    XDBG((D_ERR, "mac_loadfunc(): malloc(%lu) failed!\n",
	          (unsigned long) sizeof(UPPADM) ));
	    return(RTLERR_OUT_OF_MEMORY);
	 }

	 /* chain it into the shlib control block */
	 uppadm->theUPP	   = funcUPP;
	 uppadm->next	   = shlib->ptr_uppadm;
	 shlib->ptr_uppadm = uppadm;

	 /* return the function pointer (function entry point) */
	 (*fptr)           = (RTL_FUNC_ADR) funcUPP;

      } else {

	 XDBG((D_ERR, "mac_loadfunc(): NewPtr(%lu) failed!\n",
		      (unsigned long) sizeof(RoutineDescriptor) ));
	 return(RTLERR_OUT_OF_MEMORY);

      }

   }

#  else /* MacIntosh on PowerPC */

   (*fptr) = (RTL_FUNC_ADR) funcAddr;

#  endif /* MacIntosh on PowerPC */

   return(RTL_OK);

} /* mac_loadfunc() */


#elif defined(RTLINK_OS400)

   /*******************************************************************/
   /*								      */
   /*	 IBM AS/400 running OS/400				      */
   /*								      */
   /*								      */
   /*								      */
   /*******************************************************************/

#  error  Implementation for RUNTIME LOADING of shared libaries missing!!

#elif defined(RTLINK_OS2)

   /*******************************************************************/
   /*								      */
   /*	 x86-PC running OS/2	  				      */
   /*								      */
   /*	 INCOMPLETE and UNTESTED				      */
   /*								      */
   /*******************************************************************/

#  define INCL_DOS
#  define INCL_DOSERRORS
#  include <os2.h>

#  define PLATFORM_SPECIFIC_DLOPEN(shlib)	os2_loadlib(shlib)
#  define PLATFORM_SPECIFIC_DLCLOSE(shlib)	os2_unloadlib(shlib)
#  define PLATFORM_SPECIFIC_DLSYM(shlib,fname,attr,ptr)	 \
						os2_funcname(shlib, fname, ptr)

static int
os2_loadlib( struct rtl_shlib_s * shlib )
{
   APIRET   rc;
   HMODULE  hmod;

   rc = DosQueryModuleHandle( shlib->libname, &hmod );
   if ( rc==ERROR_MOD_NOT_FOUND ) {
      /* DLL not found in Memory, so try to load it */

      rc = DosLoadModule( NULL, 0L, shlib->libname, &hmod );
      if ( rc!=0 ) {
	 /* FAILURE: the DLL wasn't found or couldn't be loaded! */
	 /* MISSING: how do I read out the error code on OS/2 ?? */
	 return(RTLERROR_LOAD_FAILED);
      }
      /* success, the DLL was loaded */
      shlib->ptr_handle = (void *) hmod;

   }

   return(RTL_OK);

}/* os2_loadlib() */


static int
os2_unloadlib( struct rtl_shlib_s * shlib )
{
   APIRET   rc;
   HMODULE  hmod;

   hmod = (HMODULE) (shlib->ptr_handle);
   rc = DosFreeModule( hmod );
   if ( rc!=0 ) {
      /* FAILURE during unloading of DLL */
      /* MISSING: how do I read out the error code on OS/2 ?? */
      return(RTLERR_UNLOAD);
   }

   return(RTL_OK);

} /* os2_unloadlib() */


static int
os2_funcname( struct rtl_shlib_s * shlib, char * funcname, RTL_FUNC_ADR * fptr )
{
   APIRET    rc;
   HMODULE   hmod;

   hmod = (HMODULE) (shlib->ptr_handle);

   rc = DosQueryProcAddr( hmod, 0L, (PSZ)funcname, (PFN *)fptr );
   if ( rc!=0 ) {
      /* FAILURE: resolving of address for funcname "funcname" failed! */
      *fptr = (RTL_FUNC_ADR *) 0;
      return(RTLERR_SYMBOL_FAILED);
   }

   return(RTL_OK);

} /* os2_funcname() */


#else

#  error  Implementation for RUNTIME LOADING of shared libaries missing!!

#endif




/*
 * rtl_load_library()
 */
int
rtl_load_library( char * p_libname,  RTL_HANDLE  * pp_handle )
{
   char     * ptr    = NULL;
   size_t     namelen;
   int        slot;
   int	      rc     = 0;
   DEBUG_BEGIN(rtl_load_library)


   XDBG((D_ARG, "\t\t&libname        = ptr:%p,\n"
		"\t\t&handle         = ptr:%p )\n",
		p_libname, pp_handle ))
   XDBG((D_STRING, "libname", p_libname ))

   if ( pp_handle==NULL )
      ERROR_RETURN_RC(RTLERR_INVALID_VALUE);

   (*pp_handle) = RTL_INVALID_HANDLE;

   if ( p_libname==NULL || p_libname[0]=='\0' )
      ERROR_RETURN_RC(RTLERR_INVALID_VALUE);

   /* determine length of filename for shared library */
   namelen = Strnlen(p_libname, LIBNAME_MAX);
   if ( namelen>=LIBNAME_MAX )
      ERROR_RETURN_RC(RTLERR_NAME_TOO_LONG);

   /* find empty slot in our table */
   for( slot=RTL_INVALID_HANDLE+1 ; slot<RTL_MAXLIBS ; slot++ ) {
      if ( rtl_shlib_adm[slot].libname==NULL ) { break; }
   }

   if ( slot>=RTL_MAXLIBS ) 
      ERROR_RETURN_RC(RTLERR_TABLE_FULL);

   /* clean slot entry ... safety */
   memset( &(rtl_shlib_adm[slot]), 0, sizeof(rtl_shlib_adm[0]) );

   /* allocate buffer for filename of shared library */
   ptr = (char *) malloc( namelen + 1 );
   if ( ptr==NULL )
      ERROR_RETURN_RC(RTLERR_OUT_OF_MEMORY);

   memcpy(ptr, p_libname, namelen);
   ptr[namelen]		       = '\0';
   rtl_shlib_adm[slot].libname = ptr;

   /* try to actually open&load the shared library now */
   rc = PLATFORM_SPECIFIC_DLOPEN( &(rtl_shlib_adm[slot]) );

   if ( rc==0 ) {

      /* success! */
      (*pp_handle) = (RTL_HANDLE)slot;

      XDBG((D_RET, "loading of slot #%u, (ptr_handle= %p)\n"
		   "\tshared library \"%.150s\" succeded\n",
		   slot, rtl_shlib_adm[slot].ptr_handle, ptr))
   } else {

      /* failure of PLATFORM_SPECIFIC_DLOPEN() */
      (*pp_handle) = RTL_INVALID_HANDLE;

      rtl_shlib_adm[slot].libname    = NULL;
      rtl_shlib_adm[slot].ptr_handle = NULL;
error:
      if ( ptr!=NULL ) { free( ptr );   ptr = NULL; }
      if ( rtl_shlib_adm[slot].real_libname!=NULL ) {
	 free(rtl_shlib_adm[slot].real_libname);
	 rtl_shlib_adm[slot].real_libname = NULL;
      }

      XDBG((D_RET, "loading of shared\n"
		   "\tlibrary \"%.150s\" failed (%s)\n",
		   p_libname, rtl_error_name(rc) ))
   }

   return(rc);

} /* rtl_load_library() */




/*
 * rtl_library_name()
 *
 * On most platforms, a flat filename without path may work
 * without the file being present in the current directory.
 * In such cases it would really be interesting to know which
 * library was actually used.  Not all platforms allow to
 * retrieve this information -- M$ Windows does.
 */
int
rtl_library_name( RTL_HANDLE      p_handle,
		  char         ** pp_libname )
{
   int     rc = 0;
   DEBUG_BEGIN(rtl_library_name)


   XDBG((D_ARG, "\t\t handle         = ptr:%p,\n"
		"\t\t&libname        = ptr:%p )\n",
		p_handle, pp_libname ))

   if ( pp_libname==NULL )
      ERROR_RETURN_RC(RTLERR_INVALID_HANDLE);

   (*pp_libname) = 0;

   if ( p_handle<=RTL_INVALID_HANDLE  ||  p_handle>=RTL_MAXLIBS )
      ERROR_RETURN_RC(RTLERR_INVALID_HANDLE);

   if ( rtl_shlib_adm[p_handle].libname==NULL )
      ERROR_RETURN_RC(RTLERR_EMPTY_SLOT);

   if ( rtl_shlib_adm[p_handle].real_libname!=NULL ) {
      (*pp_libname) = rtl_shlib_adm[p_handle].real_libname;
   } else {
      (*pp_libname) = rtl_shlib_adm[p_handle].libname;
   }

error:
   return(rc);

} /* rtl_library_name() */



/*
 * rtl_unload_library()
 */
int
rtl_unload_library( RTL_HANDLE   * pp_handle )
{
   int     rc    = 0;
   int     slot;
   DEBUG_BEGIN(rtl_unload_library)


   XDBG((D_ARG, "\t\t&handle         = ptr:%p )\n", pp_handle ))

   if ( pp_handle==NULL
	||  (*pp_handle)<=RTL_INVALID_HANDLE  ||  (*pp_handle)>=RTL_MAXLIBS )
      ERROR_RETURN_RC(RTLERR_INVALID_HANDLE);

   slot = (*pp_handle);
   if ( rtl_shlib_adm[slot].libname==NULL )
      ERROR_RETURN_RC(RTLERR_EMPTY_SLOT);

   rc = PLATFORM_SPECIFIC_DLCLOSE( &(rtl_shlib_adm[slot]) );

   if ( rc==0 ) {

      XDBG((D_RET, "unloading of slot #%u, (ptr_handle= %p)\n"
		   "\tshared library \"%.150s\" succeded\n",
		   slot, rtl_shlib_adm[slot].ptr_handle,
		   rtl_shlib_adm[slot].libname ))

      if ( rtl_shlib_adm[slot].libname!=NULL ) {
	 free( rtl_shlib_adm[slot].libname );
      }
      rtl_shlib_adm[slot].libname      = NULL;

      if ( rtl_shlib_adm[slot].real_libname!=NULL ) {
	 free( rtl_shlib_adm[slot].real_libname );
      }
      rtl_shlib_adm[slot].real_libname = NULL;

      rtl_shlib_adm[slot].ptr_handle   = NULL;

      (*pp_handle)                     = RTL_INVALID_HANDLE;

   } else {

      XDBG((D_RET, "unloading of shared\n"
		   "\t library \"%.150s\" failed (%s)\n",
		   rtl_shlib_adm[slot].libname, rtl_error_name(rc) ))
error:
      ; /* a label needs a statement -- at least an empty one */ 
   }

   return(rc);

} /* rtl_unload_library() */



/*
 * rtl_load_function()
 */
int
rtl_load_function( RTL_HANDLE     p_handle,       char	        * p_prefix,
		   char         * p_funcname,
	           Uint32         p_attributes,   RTL_FUNC_ADR  * pp_fptr )
{
   int     rc = 0;
   size_t  len;
   char    funcname[128];
   DEBUG_BEGIN(rtl_load_function)


   XDBG((D_ARG, "\t\t handle         = %d,\n"
	        "\t\t&prefix         = ptr:%p,\n"
		"\t\t&funcname       = ptr:%p,\n"
		"\t\t attributes     = %d,\n"
		"\t\t&funcptr        = ptr:%p )\n",
		p_handle, p_prefix, p_funcname,
		p_attributes, pp_fptr ))

   XDBG((D_STRING, "prefix",   p_prefix))
   XDBG((D_STRING, "funcname", p_funcname))

   if ( pp_fptr==NULL )
      ERROR_RETURN_RC(RTLERR_INVALID_HANDLE);

   (*pp_fptr) = (RTL_FUNC_ADR)0;

   if ( p_handle<=RTL_INVALID_HANDLE  ||  p_handle>=RTL_MAXLIBS )
      ERROR_RETURN_RC(RTLERR_INVALID_HANDLE);

   if ( rtl_shlib_adm[p_handle].libname==NULL )
      ERROR_RETURN_RC(RTLERR_EMPTY_SLOT);

   if ( p_funcname==NULL )
      ERROR_RETURN_RC(RTLERR_INVALID_VALUE);

   if ( Strmaxcpy(funcname, p_prefix, (sizeof(funcname)-1))==NULL )
      ERROR_RETURN_RC(RTLERR_INVALID_HANDLE);

   len = strlen(funcname);
   if ( Strmaxcpy(&funcname[len], p_funcname, (sizeof(funcname)-len))==NULL )
      ERROR_RETURN_RC(RTLERR_INVALID_VALUE);

   rc = PLATFORM_SPECIFIC_DLSYM( &(rtl_shlib_adm[p_handle]),
				 funcname, p_attributes, pp_fptr );

   if ( rc==0 ) {
      DEBUG_EXEC(int  len = 28 - strlen(funcname);)
      DEBUG_EXEC(if (len<=0) { len=1; } )

      XDBG((D_RET, "of \"%s\"%.*s:= ptr:%p\n",
		   funcname, len, "", (void *) *pp_fptr ))

   } else {

      XDBG((D_RET, "of \"%s\" FAILED\n"
		      "\tfor shared library \"%s\"\n",
		      funcname, rtl_shlib_adm[p_handle].libname ))
error:
      (*pp_fptr) = (RTL_FUNC_ADR)0;
   }

   return(rc);

} /* rtl_load_func() */



/*
 * rtl_error_name()
 */
char *
rtl_error_name( int p_rc )
{
   switch( p_rc ) {
      case RTL_OK:			return("RTL_OK");
      case RTLERR_LOAD_FAILED:		return("RTLERR_LOAD_FAILED");
      case RTLERR_UNLOAD_FAILED:	return("RTLERR_UNLOAD_FAILED");
      case RTLERR_SYMBOL_FAILED:	return("RTLERR_SYMBOL_FAILED");
      case RTLERR_TABLE_FULL:		return("RTLERR_TABLE_FULL");
      case RTLERR_INVALID_HANDLE:	return("RTLERR_INVALID_HANDLE");
      case RTLERR_INVALID_VALUE:	return("RTLERR_INVALID_VALUE");
      case RTLERR_NAME_TOO_LONG:	return("RTLERR_NAME_TOO_LONG");
      case RTLERR_OUT_OF_MEMORY:	return("RTLERR_OUT_OF_MEMORY");
      default:				break;
   }

   return("not an RTLERR value");

} /* rtl_error_name() */
