/* ==== signal.c ============================================================
 * Copyright (c) 1993, 1994 by Chris Provenzano, proven@mit.edu
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *  This product includes software developed by Chris Provenzano.
 * 4. The name of Chris Provenzano may not be used to endorse or promote 
 *	  products derived from this software without specific prior written
 *	  permission.
 *
 * THIS SOFTWARE IS PROVIDED BY CHRIS PROVENZANO ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL CHRIS PROVENZANO BE LIABLE FOR ANY 
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * Description : Queue functions.
 *
 *  1.00 93/07/21 proven
 *      -Started coding this file.
 */

#ifndef lint
static const char rcsid[] = "$Id: signal.c,v 1.59 1994/12/13 07:28:14 proven Exp $";
#endif

#include <pthread.h>
#include <signal.h>

/* This will force init.o to get dragged in; if you've got support for
   C++ initialization, that'll cause pthread_init to be called at
   program startup automatically, so the application won't need to
   call it explicitly.  */

extern char __pthread_init_hack;
char *__pthread_init_hack_2 = &__pthread_init_hack;

/*
 * Time which select in fd_kern_wait() will sleep.
 * If there are no threads to run we sleep for an hour or until
 * we get an interrupt or an fd thats awakens. To make sure we
 * don't miss an interrupt this variable gets reset too zero in
 * sig_handler_real().
 */
struct timeval __fd_kern_wait_timeout = { 0, 0 };

/*
 * Global for user-kernel lock, and blocked signals
 */
static volatile	sigset_t sig_to_process;
static volatile	int	sig_count = 0;

static void sig_handler(int signal);
static void set_thread_timer();
void sig_prevent(void);
void sig_resume(void);

/* Make this global so BROKEN signal semantics can restart signals */
#if defined(SA_RESTART) || defined(SA_RESETHAND)
	struct sigaction act;
#endif

/* ==========================================================================
 * context_switch()
 *
 * This routine saves the current state of the running thread gets
 * the next thread to run and restores it's state. To allow different
 * processors to work with this routine, I allow the machdep_restore_state()
 * to either return or have it return from machdep_save_state with a value
 * other than 0, this is for implementations which use setjmp/longjmp. 
 */
static void context_switch()
{
	struct pthread **current, *next, *last, **dead;

	if (pthread_run->state == PS_RUNNING) {
		/* Put current thread back on the queue */
		pthread_prio_queue_enq(pthread_current_prio_queue, pthread_run);
	}

	/* save floating point registers if necessary */
	if (!(pthread_run->attr.flags & PTHREAD_NOFLOAT)) {
		machdep_save_float_state(pthread_run);
	}
	/* save state of current thread */
	if (machdep_save_state()) {
		return;
	}

	/* Poll all fds */
	fd_kern_poll();

	/* Remove dead threads */
	for (dead = &pthread_dead; *dead; ) {
		if ((*dead) == pthread_run) {
			dead = &((*dead)->next);
			continue;
		}
		if ((*dead)->attr.flags & PTHREAD_DETACHED) {
			if (!((*dead)->attr.stackaddr_attr)) {
				free (machdep_pthread_cleanup(&((*dead)->machdep_data)));
			}
			for (current = &pthread_link_list; (*current) != (*dead);
			  current = &((*current)->pll));
			*dead = (*current)->next;
			next = (*current)->pll;
			free (*current);
			*current = next;
		} else {
			dead = &((*dead)->next);
		}
	}

	last = pthread_run;

context_switch_reschedule:;
	/* Are there any threads to run */
	if (pthread_run = pthread_prio_queue_deq(pthread_current_prio_queue)) {
		/* restore floating point registers if necessary */
		if (!(pthread_run->attr.flags & PTHREAD_NOFLOAT)) {
			machdep_restore_float_state();
		}
        /* restore state of new current thread */
		machdep_restore_state();
        return;
    }

	/* Are there any threads at all */
	for (next = pthread_link_list; next; next = next->pll) {
		if (next->state != PS_DEAD) {
			/*
			 * Do a wait, timeout is set to a hour unless we get an interrupt
			 * before the select in wich case it polls and returns. 
			 */
			fd_kern_wait();

			/* Check for interrupts, but ignore SIGVTALR */
			sigdelset(&sig_to_process, SIGVTALRM); 

			if (SIG_ANY(sig_to_process)) {
				/* Process interrupts */
				sig_handler(0); 
			}

			goto context_switch_reschedule;
		}
	}
	pthread_run = last;
	exit(0);
}

/* ==========================================================================
 * sig_handler_pause()
 * 
 * Wait until a signal is sent to the process.
 */
void sig_handler_pause()
{
	sigset_t sig_to_block, sig_to_pause, oset;

	sigfillset(&sig_to_block);
	sigemptyset(&sig_to_pause);
	machdep_sys_sigprocmask(SIG_BLOCK, &sig_to_block, &oset);
	if (!(SIG_ANY(sig_to_process))) {
		sigsuspend(&sig_to_pause);
	}
	machdep_sys_sigprocmask(SIG_UNBLOCK, &sig_to_block, &oset);
}

/* ==========================================================================
 * context_switch_done()
 *
 * This routine does all the things that are necessary after a context_switch()
 * calls the machdep_restore_state(). DO NOT put this in the context_switch()
 * routine because sometimes the machdep_restore_state() doesn't return
 * to context_switch() but instead ends up in machdep_thread_start() or
 * some such routine, which will need to call this routine and
 * sig_check_and_resume().
 */
void context_switch_done()
{
	sigdelset(&sig_to_process, SIGVTALRM);
	set_thread_timer();
}

/* ==========================================================================
 * set_thread_timer()
 *
 * Assums kernel is locked.
 */
static void set_thread_timer()
{
	static int last_sched_attr = SCHED_RR;

	switch (pthread_run->attr.schedparam_policy) {
	case SCHED_RR:
		machdep_set_thread_timer(&(pthread_run->machdep_data));
		break;
	case SCHED_FIFO:
		if (last_sched_attr != SCHED_FIFO) {
			machdep_unset_thread_timer(NULL);
		}
		break;
	case SCHED_IO:
		if ((last_sched_attr != SCHED_IO) && (!sig_count)) {
			machdep_set_thread_timer(&(pthread_run->machdep_data));
		}
		break;
	default:
		machdep_set_thread_timer(&(pthread_run->machdep_data));
		break;
	} 
    last_sched_attr = pthread_run->attr.schedparam_policy;
}

/* ==========================================================================
 * sigvtalrm()
 */
static inline void sigvtalrm() 
{
	if (sig_count) {
		sigset_t sigall, oset;

		sig_count = 0;

		/* Unblock all signals */
		sigemptyset(&sigall);
		machdep_sys_sigprocmask(SIG_SETMASK, &sigall, &oset); 
	}
	context_switch();
	context_switch_done();
}

/* ==========================================================================
 * sigdefault()
 */
static inline void sigdefault()
{
	int ret;

	ret = pthread_sig_register();
	if (pthread_run && (ret > pthread_run->pthread_priority)) {
		sigvtalrm();
	}
}

/* ==========================================================================
 * sig_handler()
 *
 * Process signal that just came in, plus any pending on the signal mask.
 * All of these must be resolved.
 *
 * Assumes the kernel is locked. 
 */
static void sig_handler(int sig)
{
	int ret;

	if (pthread_kernel_lock != 1) {
		PANIC();
	}

sig_handler_top:;

	switch(sig) {
	case 0:
		break;
	case SIGVTALRM:
		sigvtalrm();
		break;
	case SIGALRM:
		sigdelset(&sig_to_process, SIGALRM);
		switch (ret = sleep_wakeup()) {
		default:
			if (pthread_run && (ret > pthread_run->pthread_priority)) {
				sigvtalrm();
			}
		case 0:
			break;
		case NOTOK:
			/* Do the registered action, no threads were sleeping */
			sigdefault();
			break;
		} 
		break;
	case SIGCHLD:
		sigdelset(&sig_to_process, SIGCHLD);
		switch (ret = wait_wakeup()) {
		default:
			if (pthread_run && (ret > pthread_run->pthread_priority)) {
				sigvtalrm();
			}
		case 0:
			break;
		case NOTOK:
			/* Do the registered action, no threads were waiting */
			sigdefault();
			break;
		} 
		break;

#ifdef SIGINFO
	case SIGINFO:
		pthread_dump_info ();
		/* Then fall through, invoking the application's
		   signal handler after printing our info out.

		   I'm not convinced that this is right, but I'm not
		   100% convinced that it is wrong, and this is how
		   Chris wants it done...  */
#endif

	default:
		/* Do the registered action */
		sigdefault();
		break;
	}

	/* Determine if there are any other signals */
	if (SIG_ANY(sig_to_process)) {
		for (sig = 1; sig <= SIGMAX; sig++) {
			if (sigismember(&sig_to_process, sig)) {
				goto sig_handler_top;
			}
		}
	}
}

/* ==========================================================================
 * sig_handler_real()
 * 
 * On a multi-processor this would need to use the test and set instruction
 * otherwise the following will work.
 */
void sig_handler_real(int sig)
{
	/* Get around systems with BROKEN signal handlers */
	#if defined(SA_RESETHAND)
	sigaction(sig, &act, NULL);
	#endif

	if (pthread_kernel_lock) {
		__fd_kern_wait_timeout.tv_sec = 0;
		sigaddset(&sig_to_process, sig);
		return;
	}
	pthread_kernel_lock++;

	sig_count++;
	sig_handler(sig);
	pthread_kernel_lock--;
}

/* ==========================================================================
 * sig_handler_fake()
 */
void sig_handler_fake(int sig)
{
	if (pthread_kernel_lock) {
		sigaddset(&sig_to_process, sig);
		return;
	}
	pthread_kernel_lock++;
	sig_handler(sig);
	while (!(--pthread_kernel_lock)) {
		if (SIG_ANY(sig_to_process)) {
			pthread_kernel_lock++;
			sig_handler(0);
		} else {
			break;
		}
	}
}

/* ==========================================================================
 * pthread_sched_other_resume()
 *
 * Check if thread to be resumed is of higher priority and if so
 * stop current thread and start new thread.
 */
pthread_sched_other_resume(struct pthread * pthread)
{
	pthread->state = PS_RUNNING;
	pthread_prio_queue_enq(pthread_current_prio_queue, pthread);

	if (pthread->pthread_priority > pthread_run->pthread_priority) {
		if (pthread_kernel_lock == 1) {
			sig_handler(SIGVTALRM);
		}
	}

	/* Only bother if we are truely unlocking the kernel */
	while (!(--pthread_kernel_lock)) {
		if (SIG_ANY(sig_to_process)) {
			pthread_kernel_lock++;
			sig_handler(0);
		} else {
			break;
		}
		if (pthread_run->sigcount) {
			pthread_sig_process();
		}
	}
}

/* ==========================================================================
 * pthread_resched_resume()
 *
 * This routine assumes that the caller is the current pthread, pthread_run
 * and that it has a lock the kernel thread and it wants to reschedule itself.
 */
void pthread_resched_resume(enum pthread_state state)
{
	pthread_run->state = state;
	sig_handler(SIGVTALRM);

	/* Only bother if we are truely unlocking the kernel */
	while (!(--pthread_kernel_lock)) {
		if (SIG_ANY(sig_to_process)) {
			pthread_kernel_lock++;
			sig_handler(0);
		} else {
			break;
		}
		if (pthread_run->sigcount) {
			pthread_sig_process();
		}
	}
}

/* ==========================================================================
 * pthread_sched_resume()
 */
void pthread_sched_resume()
{
	/* Only bother if we are truely unlocking the kernel */
	while (!(--pthread_kernel_lock)) {
		if (SIG_ANY(sig_to_process)) {
			pthread_kernel_lock++;
			sig_handler(0);
		} else {
			break;
		}
		if (pthread_run->sigcount) {
			pthread_sig_process();
		}
	}
}

/* ==========================================================================
 * pthread_sched_prevent()
 */
void pthread_sched_prevent(void)
{
	pthread_kernel_lock++;
}

/* ==========================================================================
 * sig_init()
 *
 * SIGVTALRM	(NOT POSIX) needed for thread timeslice timeouts.
 *				Since it's not POSIX I will replace it with a 
 *				virtual timer for threads.
 * SIGALRM		(IS POSIX) so some special handling will be
 * 				necessary to fake SIGALRM signals
 */
void sig_init(void)
{
	int sig_init[] = { SIGCHLD, SIGALRM, SIGVTALRM, 0 };
	int i, j;

#if defined(SA_RESTART) || defined(SA_RESETHAND)
	act.sa_handler = sig_handler_real;
	sigemptyset(&(act.sa_mask));
#if defined(SA_RESTART)
	act.sa_flags = SA_RESTART;
#else
	act.sa_flags = SA_RESETHAND;
#endif
#endif

	/* Initialize the important signals */
	for (i = 0; sig_init[i]; i++) {

#if defined(SA_RESTART) || defined(SA_RESETHAND)
		if (sigaction(sig_init[i], &act, NULL)) {
#else
		if (signal(sig_init[i], sig_handler_real)) { 
#endif
			PANIC();
		}
	}

	/* Initialize the rest of the signals */
	for (j = 0; j < SIGMAX; j++) {
		for (i = 0; sig_init[i]; i++) {
			if (sig_init[i] == j) {
				goto sig_next;
			}
		}
		
#if defined(SA_RESTART) || defined(SA_RESETHAND)
		sigaction(sig_init[i], &act, NULL);
#else
		signal(sig_init[i], sig_handler_real);
#endif

		sig_next:;
	}
}

