/* twait_test.c - fairly elaborate interactive test for timed waits with I/O */

/* SNL, 7 April 1995 */

/* Yes, Martha, we do leak core.  I've already spent too much time on this test
   program, and it isn't really significant wrt the purpose of the test. */

#include <pthread.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include "sock_lib.h"

#ifndef IMPLEMENTATION_mit
# include <timers.h>
# define ts_sec	tv_sec
# define ts_nsec tv_nsec
# define TIMEVAL_TO_TIMESPEC(tv, ts) {                                   \
        (ts)->ts_sec = (tv)->tv_sec;                                    \
        (ts)->ts_nsec = (tv)->tv_usec * 1000;                           \
}

# define THREAD_ATTR			thread_attr

#else /* ! IMPLEMENTATION_mit */

# define pthread_mutexattr_default	NULL
# define pthread_condattr_default	NULL
# define THREAD_ATTR			&thread_attr

typedef void *pthread_addr_t;

#endif /* IMPLEMENTATION_mit */

struct io_thread_info {
  char *msg;
  char *host_name;
  int test_count;
  int inner_loops;
};

struct probe_thread_info {
  int tout_ms;
  int n_loops;
};

struct thread_rec {
  struct thread_rec *next;
  pthread_t thread;
  char *type;
};

pthread_cond_t probe_wakeup;
pthread_cond_t finished;
pthread_mutex_t finished_lock;
int finished_flag = 0;
unsigned finished_tout = 10;	/* milliseconds */

pthread_mutex_t statistics_lock;
int n_io_loops = 0;
int n_io_bytes_written = 0;
int n_io_bytes_read = 0;

pthread_mutex_t thread_list_lock;
struct thread_rec *thread_list = 0;

pthread_attr_t thread_attr;

extern void *malloc();

void
remember_thread(pthread_t thread, char *type)
{
  struct thread_rec *new_entry;

  new_entry = (struct thread_rec *)malloc(sizeof(struct thread_rec));
  new_entry->next = 0;
  new_entry->thread = thread;
  new_entry->type = (char *)malloc(strlen(type) + 1);
  strcpy(new_entry->type, type);
  pthread_mutex_lock(&thread_list_lock);
  if (!thread_list)
    thread_list = new_entry;
  else {
    struct thread_rec *last, *ptr;

    ptr = thread_list;
    do {
      last = ptr;
      ptr = ptr->next;
    } while (ptr);
    last->next = new_entry;
  }
  pthread_mutex_unlock(&thread_list_lock);
}

void
forget_thread(pthread_t thread)
{
  struct thread_rec *ptr, *last;

  pthread_mutex_lock(&thread_list_lock);
  for (ptr = thread_list, last = 0; ptr; last = ptr, ptr = ptr->next)
    if (pthread_equal(ptr->thread, thread)) {
      if (!last)
	thread_list = thread_list->next;
      else
	last->next = ptr->next;
      free(ptr->type);
      free(ptr);
      break;
    }
  pthread_mutex_unlock(&thread_list_lock);
}

int
check_finished()
{
  struct timespec abstime = { 0, 0 };
  struct timeval tv;
  int x, result;

  gettimeofday(&tv, NULL);
  tv.tv_sec += (finished_tout / 1000);
  tv.tv_usec += ((finished_tout % 1000) * 1000);
  TIMEVAL_TO_TIMESPEC(&tv, &abstime);
  x = pthread_cond_timedwait(&finished, &finished_lock, &abstime);
  if (!x || finished_flag) {
    printf("thread 0x%08lx got finished signal -- cleaning up\n");
    result = 1;
  } else
    result = 0;
  pthread_mutex_unlock(&finished_lock);
  return result;
}

void *
io_thread(void *arg)
{
  int count = 0, test_count, msg_len, in_loops, n_out;
  char *host_name, *msg;
  struct io_thread_info *info;

  info = (struct io_thread_info *)arg;
  host_name = info->host_name;
  msg = info->msg;
  msg_len = strlen(msg);
  n_out = test_count = info->test_count;
  in_loops = info->inner_loops;
  free(info);
  remember_thread(pthread_self(), "io");
  count = 0;
  printf("* IO thread 0x%08lx starting %d outer x %d inner loops...\n",
	 pthread_self(), test_count, in_loops);
  while (test_count-- >= 0) {
    int c, s, inner_loop, bytes_written, bytes_read;

    if (check_finished())
      break;
    s = socket_to_service(host_name, "echo", "tcp");
    if (s < 0) {
      printf("socket_to_service => %d\n", s);
      sleep(5);			/* wait a bit */
      continue;
    }
    bytes_written = bytes_read = 0;
    for (inner_loop = 0; inner_loop < in_loops; inner_loop++) {
      char buffer[128];
      int nread, done, nwrite;

      nwrite = write(s, msg, msg_len);
      if (nwrite != msg_len) {
	printf("write(%d) => socket failed (%d: %d)\n", msg_len, nwrite, errno);
	abort();
      }
      bytes_written += msg_len;
      while ((nread = read(s, buffer, sizeof(buffer))) > 0) {
	bytes_read += nread;
	done = 0;
	while (nread-- > 0)
	  if (buffer[nread] == '\n') {
	    done = 1;
	    break;
	  }
	if (done)
	  break;
      }
      if (!done) {
	printf("EOF\n");
	break;
      }
    }
    close(s);
    pthread_mutex_lock(&statistics_lock);
    n_io_loops++;
    n_io_bytes_written += bytes_written;
    n_io_bytes_read += bytes_read;
    pthread_mutex_unlock(&statistics_lock);
    bytes_written = bytes_read = 0;
    count++;
  }
  printf("IO thread %lx quitting after %d/%d outer loops\n", count, n_out);
  forget_thread(pthread_self());
  pthread_exit(0);
  return 0;			/* never get here */
}

void *
probe_thread(void *arg)
{
  struct timeval tv, start, done;
  struct timespec next_time;
  int n_loops, n, done_ms, start_ms, expected_ms, tout_ms;
  struct probe_thread_info *info = (struct probe_thread_info *)arg;

  tout_ms = info->tout_ms;
  n = n_loops = info->n_loops;
  free(info);
  remember_thread(pthread_self(), "probe");
  gettimeofday(&start, NULL);
  start_ms = (start.tv_sec * 1000) + (start.tv_usec / 1000);
  while (n_loops-- > 0) {
    int x;

    if (check_finished())
      break;
    gettimeofday(&tv, NULL);
    tv.tv_sec += (tout_ms / 1000);
    tv.tv_usec += ((tout_ms % 1000) * 1000);
    TIMEVAL_TO_TIMESPEC(&tv, &next_time);
    x = pthread_cond_timedwait(&probe_wakeup, &statistics_lock, &next_time);
    if (!x) {
      gettimeofday(&done, NULL);
      done_ms = (done.tv_sec * 1000) + (done.tv_usec / 1000);
      expected_ms = tout_ms * (n - n_loops);
      printf("\nprobe 0x%08lx got wakeup signal at loop %d after %d ms (%d)\n\tn=%d read=%d write=%d\n",
	     pthread_self(), n - n_loops, done_ms - start_ms, expected_ms,
	     n_io_loops, n_io_bytes_read,
	     n_io_bytes_written);
    }
    pthread_mutex_unlock(&statistics_lock);
  }
  gettimeofday(&done, NULL);
  expected_ms = tout_ms * n;
  done_ms = (done.tv_sec * 1000) + (done.tv_usec / 1000);
  printf("probe 0x%08lx finished %d loops of %d ms in %d ms real time (expected %d)\n",
	 pthread_self(), n, tout_ms, done_ms - start_ms, expected_ms);
  forget_thread(pthread_self());
  pthread_exit(0);
  return 0;			/* never get here */
}

void
list_threads()
{
  struct thread_rec *ptr;
  int count;

  pthread_mutex_lock(&thread_list_lock);
  for (ptr = thread_list, count = 0; ptr; ptr = ptr->next, count++)
    printf("%d: 0x%08lx (%s)\n", count, ptr->thread, ptr->type);
  pthread_mutex_unlock(&thread_list_lock);
}

void
fork_io_thread(char *cmd_buf)
{
  struct io_thread_info *info;
  pthread_t t;
  char host[256], msg[256];
  int count, iloops, add_newline;

  if (sscanf(cmd_buf, "%s %d %d %s", host, &count, &iloops, msg) != 4) {
    printf("io_thread: arguments `%s' don't scan\n", cmd_buf);
    return;
  }
  info = (struct io_thread_info *)malloc(sizeof(struct io_thread_info));
  add_newline = (msg[strlen(msg)-1] != '\n');
  info->msg = (char *)malloc(strlen(msg) + 1 + add_newline);
  strcpy(info->msg, msg);
  if (add_newline)
    strcat(info->msg, "\n");
  info->host_name = (char *)malloc(strlen(host) + 1);
  strcpy(info->host_name, host);
  info->test_count = count;
  info->inner_loops = iloops;
  if (pthread_create(&t, THREAD_ATTR, io_thread, (void *)info) < 0)
    printf("could not fork io thread\n");
  else
    printf("forked io thread 0x%08lx\n", t);
}

void
fork_probe_thread(char *cmd_buf)
{
  struct probe_thread_info *info;
  pthread_t t;
  int count, tout;

  if (sscanf(cmd_buf, "%d %d", &count, &tout) != 2) {
    printf("probe_thread: arguments `%s' don't scan\n", cmd_buf);
    return;
  }
  info = (struct probe_thread_info *)malloc(sizeof(struct probe_thread_info));
  info->tout_ms = tout;
  info->n_loops = count;
  if (pthread_create(&t, THREAD_ATTR, probe_thread, (void *)info) < 0)
    printf("could not fork probe thread\n");
  else
    printf("forked probe thread 0x%08lx\n", t);
}

void *
command_line_thread(void *arg)
{
  int nread;
  char buffer[512];

  (void) write(1, ">", 1);
  while ((nread = read(0, buffer, sizeof(buffer)-1)) > 0) {
    buffer[nread] = '\0';
    if (buffer[nread-1] == '\n')
      buffer[nread-1] = '\0';
    if (!strcmp(buffer, "quit"))
      break;
    if (!strcmp(buffer, "exit")) {
      printf("exiting forcibly...\n");
      exit(1);
    }
    if (!strcmp(buffer, "stats"))
      pthread_cond_broadcast(&probe_wakeup);
    else if (buffer[0] == 'i' && buffer[1] == 'o')
      fork_io_thread(&buffer[2]);
    else if (buffer[0] == 'p' && buffer[1] == 'b')
      fork_probe_thread(&buffer[2]);
    else if (!strcmp(buffer, "threads"))
      list_threads();
    else if (!strcmp(buffer, "kill")) {
      printf("broadcasting condition...\n");
      pthread_cond_broadcast(&finished);
#ifdef IMPLEMENTATION_dec
    } else if (!strcmp(buffer, "debug")) {
      cma_debug();
#endif /* IMPLEMENTATION_dec */
    } else if (strlen(buffer)) {
      if (buffer[0] != '?' && buffer[0] != 'h')
	printf("? illegal command: %s\n", buffer);
      printf("one of: quit, stats, io <host> <count> <iloops> <msg>, pb <count> <tout>, threads, ?, h\n");
    }
    (void) write(1, ">", 1);
  }
  printf("cleaning up...\n");
  pthread_mutex_lock(&finished_lock);
  finished_flag = 1;
  pthread_mutex_unlock(&finished_lock);
  pthread_cond_broadcast(&finished);
  printf("bye.\n");
  pthread_exit(0);
  return 0;			/* never get here */
}

main(int argc, char **argv)
{
  pthread_t command_loop;
  int status;

  setbuf(stdout, NULL);
#ifdef IMPLEMENTATION_mit
  printf("%s running [mit version]\n", argv[0]);
#else
  printf("%s running [dec version]\n", argv[0]);
#endif /* IMPLEMENTATION_mit */
  pthread_cond_init(&probe_wakeup, pthread_condattr_default);
  pthread_cond_init(&finished, pthread_condattr_default);
  pthread_mutex_init(&finished_lock, pthread_mutexattr_default);
  pthread_mutex_init(&statistics_lock, pthread_mutexattr_default);
  pthread_mutex_init(&thread_list_lock, pthread_mutexattr_default);
  printf("probe_wakeup=0x%lx\nfinished=0x%lx\nfinished_lock=0x%lx\nstatistics_lock=0x%lx\nthread_list_lock=0x%lx\n",
	 &probe_wakeup, &finished, &finished_lock, &statistics_lock,
	 &thread_list_lock);
#ifdef IMPLEMENTATION_mit
  pthread_attr_init(&thread_attr);
  thread_attr.schedparam_policy = SCHED_FIFO;
  pthread_attr_setstacksize(&thread_attr, (size_t)(1024 * 200));
#else /* ! IMPLEMENTATION_MIT */
  pthread_attr_create(&thread_attr);
  pthread_attr_setsched(&thread_attr, SCHED_FIFO);
  pthread_attr_setstacksize(&thread_attr, 1024 * 200);
#endif /* IMPLEMENTATION_mit */

  /* Start the command loop */
  if (pthread_create(&command_loop, THREAD_ATTR, command_line_thread,0) < 0) {
    printf("could not create command-line thread\n");
    exit(1);
  }

  /* We're done -- let all the other threads die now. */
  pthread_join(command_loop, (pthread_addr_t *)&status);
  pthread_exit(0);
}
