/*
 * KCLL -  Ken and Chee's Limey Lisp  
 * All Rights to the code and any products created with this code is 
 * hereby granted. 
 *
 * I.E. You can do whatever the hell you want with this code. 
 * The only restriction is that this copyright notice not be modified.
 */

#include <ctype.h>
#include "num.h"
#include "stream.h"
#include "cont.h"

LLTag llnum_t;

static void print_num (num, stream)
     LLNum *num;
     LLStream *stream;
{
  llstrprintf(stream, "%lf", num->value);
}

LLNum *llread_num(stream)
     LLStream *stream;
{
  char c;
  double n, d;
  int decimal = 0;
  
  d = 1.0;
  n = 0.0;
  while (isdigit(c = llread_stream(stream)) || c=='.') 
    if (c=='.')
      decimal = 1;
    else
      if (decimal)
	n = n + (d = d / 10.0) * (double) (c-48);
      else
	n = (n * 10.0) + (double) (c-48);
  llunread_stream(stream, (unsigned char)c);
  return (lldouble_to_num(n));
}

LLNum *lldouble_to_num(n)
     double n;
{
  LLNum *num;

  num = (LLNum *) llmake_obj(llnum_t);
  num->value = n;
  return num;
}



/*  Math stuff */

/*  adds all top arguments from the stack. 
    The number of args should be in llargc 
    returns value onto the stack.
*/ 


static void do_math(type, retval) 
     LLMathType type; 
     double *retval;
{
  switch (type) {
  case NUM_ADD:
    *retval += llnum_to_double((LLNum *)llnextcheckedarg(llnum_t));
    break;
  case NUM_SUB:
    *retval -= llnum_to_double((LLNum *)llnextcheckedarg(llnum_t));
    break;
  case NUM_MUL:
    *retval *= llnum_to_double((LLNum *)llnextcheckedarg(llnum_t));
    break;
  case NUM_DIV:
    *retval /= llnum_to_double((LLNum *)llnextcheckedarg(llnum_t));
    break;
  }
}

static void llarithmetic(type, retval)
     LLMathType type;
     double retval;
{
  while(llmoreargs()) {
    do_math(type, &retval);
  }
  llpusharg(lldouble_to_num(retval));
}

/* comparators */

static LLObj *compare(type, a,b)
     LLCompareType type;
     double a , b;
{
  switch (type) {
  case NUM_LT: 
    return( (a < b) ? T: NIL);
  case NUM_GT: 
    return( (a > b) ? T: NIL);
  case NUM_EQ:
    return( (a == b) ? T: NIL);
  case NUM_LE:
    return( (a <= b) ? T: NIL);
  case NUM_GE:
    return( (a >= b) ? T: NIL);
  case NUM_NE:
    return( (a != b) ? T: NIL);
  }
}

static void num_compare(type)
     LLCompareType type; 
{
  LLNum *num;
  double val1, val2;

  num = (LLNum *)(llnextcheckedarg(llnum_t));
  val1 = llnum_to_double(num);

  num = (LLNum *)(llnextcheckedarg(llnum_t));
  val2 = llnum_to_double(num);

  lllastarg();
  llpusharg(compare(type, val1, val2)); 
}

void lladdone() {
  double num; 
  num = llnum_to_double((llnextcheckedarg(llnum_t)));
  llpusharg(lldouble_to_num(num + 1.0));
}

void llsubone() {
  double num; 
  num = llnum_to_double((llnextcheckedarg(llnum_t)));
  llpusharg(lldouble_to_num(num - 1.0));
}

void llnum_add() {  llarithmetic(NUM_ADD, 0.0); }
void llnum_mul() {  llarithmetic(NUM_MUL, 1.0); }

void llnum_sub() {
  LLNum *num; 

  if(llargc <= 1) llarithmetic(NUM_SUB, 0.0);
  else {
    num = (LLNum *) llnextcheckedarg(llnum_t);
    llarithmetic(NUM_SUB, llnum_to_double(num)); 
  }
}

void llnum_div() 
{
  LLNum *num; 

  if(llargc <= 1) 
    llarithmetic(NUM_DIV, 1.0);
  else {
    num = (LLNum *) llnextcheckedarg(llnum_t);
    llarithmetic(NUM_DIV, llnum_to_double(num)); 
  }
}

void llngt() { num_compare(NUM_GT); }
void llnlt() { num_compare(NUM_LT); }
void llnge() { num_compare(NUM_GE); }
void llnle() { num_compare(NUM_LE); }
void llneq() { num_compare(NUM_EQ); }
void llnne() { num_compare(NUM_NE); }

int llnum_compare(cmptype, n1, n2) 
     LLCompare cmptype; 
     LLNum *n1, *n2; 
{
  return(compare(cmptype, 
		 llnum_to_double(n1), 
		 llnum_to_double(n2)) == T);
}

void llinit_num()
{
  llnum_t = lladd_obj_td (sizeof(LLNum), "Number",
		      0, 0, print_num, 0);
  				/* Note: the read method for numbers
				   is called from read_obj (read.c) */
  
  llregister_cfunc(llngt,">");
  llregister_cfunc(llnlt,"<");
  llregister_cfunc(llnge,">=");
  llregister_cfunc(llnle,"<=");
  llregister_cfunc(llneq,"=");
  llregister_cfunc(llnne,"/=");
  llregister_cfunc(llnne,"!=");
  llregister_cfunc(llnum_add, "+");
  llregister_cfunc(llnum_sub, "-");
  llregister_cfunc(llnum_mul, "*");
  llregister_cfunc(llnum_div, "/");
  llregister_cfunc(lladdone, "1+");
  llregister_cfunc(llsubone, "-1+");
}

