/*
 * cstack.h - A stack where all cards can be seen.
 */

/* ---------- Macros ---------- */
#define CSTACK_DRAW_SPACE 23

/* ---------- Structure definition ---------- */
struct cstack {
     struct card **cardpp;	/* Dynamic array of ptrs to cards*/
     struct area *ap;		/* Is it displayed? */
     int num_cards;		/* Number of cards in stack */
     int max_cards;		/* Size of array */
     int invert:1;		/* Should the bottom card be inverted? */
};

/* ---------- Functions ---------- */
/*
 * cstack_create()
 * 
 * Create a new, empty card stack.
 */
struct cstack *cstack_create(void);

/*
 * cstack_destroy(cstack, everything)
 * 
 * Destroy the cstack.  If everything is TRUE, then destroy all of the cards
 * as well.
 */
void cstack_destroy(struct cstack *, int);

/*
 * cstack_makeempty(cstack, freecards) 
 * 
 * Make the cstack empty.  If freecards is TRUE, then destroy all of the
 * cards currently in the cstack.
 */
void cstack_makeempty(struct cstack *, int);

/*
 * cstack_getnumcards(cstack) --> int
 * 
 * Return the number of cards in the stack.
 */
int cstack_getnumcards(struct cstack *);

/*
 * cstack_getcard(cstack, n) --> struct card *
 * 
 * Return the nth card from the stack.  0 is the bottom card and n-1 is the
 * top card.  Returns NULL if n doesn't exist.
 */
struct card *cstack_getcard(struct cstack *, int);

/*
 * cstack_bottomcard(cstack) --> struct card *
 * 
 * Return the bottom card from the stack.  Returns NULL if there are no
 * cards in the stack.
 */
struct card *cstack_bottomcard(struct cstack *);

/*
 * cstack_removecard(cstack) --> struct card *
 * 
 * Remove the bottom card from cstack and return that card.  Returns NULL
 * if cstack is empty.
 */
struct card *cstack_removecard(struct cstack *);

/*
 * cstack_split(cstack, n) --> struct cstack *
 * 
 * Modify cstack to contain cards 0--(n-1) and create and return a new stack
 * which contains cards n--(cstack_getnumcards(cstack)-1).  NOTE that either
 * stack can be empty.
 */
struct cstack *cstack_split(struct cstack *, int);

/*
 * cstack_join(cstack, other_cstack) --> result
 * 
 * Make cstack and other_cstack into one stack and destroy other_cstack.  
 */
int cstack_join(struct cstack *, struct cstack *);

/*
 * cstack_addcard(cstack, card) --> result
 * 
 * Add card to the bottom of cstack.  
 */
int cstack_addcard(struct cstack *, struct card *);

/*
 * cstack_isempty(cstack) --> int
 * 
 * Return TRUE if cstack is empty.  FALSE otherwise.  
 */
int cstack_isempty(struct cstack *);

/*
 * cstack_setinvert(cstack, bool) --> int
 * 
 * Set the invert status of cstack to bool.  Return the previous value. 
 */
int cstack_setinvert(struct cstack *, int);

/*
 * cstack_getinvert(cstack) --> int
 * 
 * Return the invert status of cstack.
 */
int cstack_getinvert(struct cstack *);

/*
 * cstack_draw(cstack, windp, x, y)
 * 
 * Draw the cstack at location (x,y) in windp.
 */
void cstack_draw(struct cstack *, struct wind *, int, int);

/*
 * cstack_draw_card(cstack, windp, cardnum, x, y)
 * 
 * Draw the cardnum card in the stack at location (x,y) in windp.
 */
void cstack_draw_card(struct cstack *, struct wind *, int, int, int);

/*
 * cstack_setarea(cstackp, areap)
 * 
 * Tell the cstack that it is being displayed in areap.
 */
#define cstack_setarea(c,a)  ((c)->ap = (a))

/*
 * cstack_getarea(cstackp) -> struct area *
 * 
 * Get the area we're displayed in.
 */
#define cstack_getarea(c)  ((c)->ap)

