/** @file if_usb.c
  * @brief This file contains functions used in USB interface module.
  *  
  * (c) Copyright  2003-2006, Marvell International Ltd. 
  * All Rights Reserved
  *
  * This software file (the "File") is distributed by Marvell International 
  * Ltd. under the terms of the GNU General Public License Version 2, June 1991 
  * (the "License").  You may use, redistribute and/or modify this File in 
  * accordance with the terms and conditions of the License, a copy of which 
  * is available along with the File in the license.txt file or by writing to 
  * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
  * 02111-1307 or on the worldwide web at http://www.gnu.org/licenses/gpl.txt.
  *
  * THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE 
  * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE 
  * ARE EXPRESSLY DISCLAIMED.  The License provides additional details about 
  * this warranty disclaimer.
  *
  */
/********************************************************
Change log:
	09/29/05: Add Doxygen format comments
	02/01/06: Correct the usage of compile flag HELPER_IMAGE
	02/13/06: Add a FW ready flag for interoperability issue
	02/22/06: Add support for new VID/PID
	03/02/06: Add receive exception handling, fix some memory leak issues
	03/03/06: Add FW download failure handling
	03/10/06: Fix the rmmod/insmod issue
	04/07/06: Add Linux kernel 2.6 support
	04/21/06: Add hotplug firmware loading
	
********************************************************/

#include	"if_usb.h"

#include	<linux/firmware.h>

/********************************************************
		Local Variables
********************************************************/
#define MARVELL_USB_DEVICE(vid, pid, name) \
           USB_DEVICE(vid, pid),  \
           .driver_info = (u32)name

/* Context definition for Interrupt simulation */ 
#define TX_SUCCESS	1
#define RX_SUCCESS	2
#define TX_FAILURE	-1 

#define MESSAGE_HEADER_LEN	4

static wlan_private *pwlanpriv;
static struct usb_card_rec usb_cardp;
const char usbdriver_name[] = "usb8xxx"; 
static u8 usb_int_cause = 0;
u8 usb_fw_ready = TRUE;

/** struct This structure contains the device signature */
struct usb_device_id if_usb_table[] = {
	
	/* Enter the device signature inside */
	{MARVELL_USB_DEVICE(USB8388_VID_1, USB8388_PID_1, "Marvell WLAN USB Adapter")},
	{MARVELL_USB_DEVICE(USB8388_VID_2, USB8388_PID_2, "Marvell WLAN USB Adapter")},
	
	/* Terminating entry */
	{} ,
};

int if_prog_firmware(wlan_private *priv);
static int if_usb_submit_rx_urb(wlan_private *priv);

static void if_usb_receive(struct urb *urb, struct pt_regs *regs);
static int if_usb_probe(struct usb_interface *intf,  
			const struct usb_device_id *id);
static void if_usb_disconnect(struct usb_interface *intf);

/** if_usb_driver */
static struct usb_driver if_usb_driver = {
		/* driver name */
	.name =	usbdriver_name,		

		/* probe function name */
	.probe = if_usb_probe,

		/* disconnect function  name */
	.disconnect = if_usb_disconnect,

		/* device signature table */
	.id_table = if_usb_table,

};

MODULE_DEVICE_TABLE(usb, if_usb_table);

/********************************************************
		Global Variables
********************************************************/


/********************************************************
		Local Functions
********************************************************/
/** 
 *  @brief  Interrupt handler
 *  @param priv         pointer to wlan_private
 *  @param flag   	flag
 *  @param context  	context parameter
 *  @return 	   	N/A
 */
static void if_usb_interrupt(wlan_private *priv, u8 flags, int context)
{
	wlan_interrupt(priv->wlan_dev.netdev);
}

/** 
 *  @brief  call back function to handle the status of the URB
 *  @param urb 		pointer to urb structure
 *  @return 	   	N/A
 */
static void if_usb_write_bulk_callback(struct urb *urb, struct pt_regs *regs)
{
	wlan_private		*priv = (wlan_private *)(urb->context);
	struct	usb_card_rec	*cardp = priv->wlan_dev.card;

	ENTER();

	cardp->tx_urb_pending = FALSE;

	/* handle the transmission complete validations */

	if (urb->status != 0) {
		/* print the failure status number for debug */
		PRINTM(FATAL, "URB in failure status\n");
		if (!cardp->IsFWDnld) {
			if_usb_interrupt(priv, 0, TX_FAILURE);
		} else {
			goto tx_exit;
		}
	} else {
		PRINTM(INFO, "URB status is successfull\n");
		PRINTM(INFO, "Actual length transmitted %d\n", urb->actual_length);
		priv->wlan_dev.dnld_sent = DNLD_RES_RECEIVED;
	}
	if (!cardp->IsFWDnld) {
		usb_int_cause |= HIS_TxDnLdRdy;
		if_usb_interrupt(priv, HIS_TxDnLdRdy, TX_SUCCESS);
	}

tx_exit:		
	LEAVE();
	return;
}

/** 
 *  @brief  free tx/rx urb, skb and rx buffer
 *  @param cardp	pointer usb_card_rec
 *  @return 	   	N/A
 */
static void if_usb_free(struct usb_card_rec *cardp)
{
	ENTER();

	/* Unlink tx & rx urb */
	if (cardp->tx_urb)
		usb_kill_urb(cardp->tx_urb);
	if (cardp->rx_urb)		
		usb_kill_urb(cardp->rx_urb);

	msleep_interruptible(10);

	/* Free tx & rx urb */
	if (cardp->tx_urb) {
		if (!cardp->tx_urb_pending) {
			usb_free_urb(cardp->tx_urb);
			cardp->tx_urb = NULL;
		}
		else {
			PRINTM(FATAL, "Tx urb pending, can not free tx urb\n");
		}
	}

	if (cardp->rx_urb) {
		if (!cardp->rx_urb_pending) {
			usb_free_urb(cardp->rx_urb);
			cardp->rx_urb = NULL;
		}
		else {
			PRINTM(FATAL, "Rx urb pending, can not free rx urb\n");
		}
	}

	if (cardp->skb) {
		kfree_skb(cardp->skb);
		cardp->skb = NULL;
	}

	if (cardp->bulk_out_buffer) {
		kfree(cardp->bulk_out_buffer);
		cardp->bulk_out_buffer = NULL;
	}

	LEAVE();
	return ;
}

/** 
 *  @brief sets the configuration values
 *  @param udev		pointer to usb_device
 *  @param ifnum	interface number
 *  @param id		pointer to usb_device_id
 *  @return 	   	address of gloabl variable usb_cardp 
 */
static int if_usb_probe(struct usb_interface *intf,
	                  const struct usb_device_id *id)
{
	struct usb_device *udev;
	struct usb_host_interface *iface_desc;
	struct usb_endpoint_descriptor	*endpoint;
	int				i;
		
	ENTER();

	udev = interface_to_usbdev(intf);

	/* Check probe is for our device */
	for (i=0; if_usb_table[i].idVendor; i++)
	{
		if (udev->descriptor.idVendor == if_usb_table[i].idVendor &&
			udev->descriptor.idProduct == if_usb_table[i].idProduct)
		{
			PRINTM(INFO, "VID/PID = %X/%X, matched\n", udev->descriptor.idVendor, udev->descriptor.idProduct);
			break;
		}
	}

	if ( if_usb_table[i].idVendor ) {

		usb_cardp.udev = udev;
		iface_desc = intf->cur_altsetting;		

		PRINTM(INFO, "bcdUSB = 0x%X bDeviceClass = 0x%X"
			" bDeviceSubClass = 0x%X, bDeviceProtocol = 0x%X\n",
			udev->descriptor.bcdUSB,
			udev->descriptor.bDeviceClass,
			udev->descriptor.bDeviceSubClass,
			udev->descriptor.bDeviceProtocol);

		for (i = 0; i < iface_desc->desc.bNumEndpoints; ++i) {
			endpoint = &iface_desc->endpoint[i].desc;			
			if ((endpoint->bEndpointAddress & BULK_ENDPOINT_PRESENT) &&
				((endpoint->bmAttributes & BULK_ENDPOINT_MASK) == BULK_ENDPOINT_FOUND)) {
				/* we found a bulk in endpoint */
				PRINTM(INFO, "Bulk in size is %d\n", 
						endpoint->wMaxPacketSize);
				if (!(usb_cardp.rx_urb = usb_alloc_urb(0, GFP_KERNEL))) {
					PRINTM(INFO, "Rx URB allocation failed\n");
					goto error;
				}
				usb_cardp.rx_urb_pending = FALSE;
				usb_cardp.rx_urb_recall = FALSE;
				
				usb_cardp.bulk_in_size = 
						endpoint->wMaxPacketSize;
				usb_cardp.bulk_in_endpointAddr = 
						(endpoint->bEndpointAddress & ENDPOINT_NUMBER_MASK);
				PRINTM(INFO, "in_endpoint = %d\n",
						endpoint->bEndpointAddress);
			}

			if (((endpoint->bEndpointAddress & BULK_ENDPOINT_PRESENT) == BULK_OUT_PRESENT) &&
				((endpoint->bmAttributes & BULK_ENDPOINT_MASK) == BULK_ENDPOINT_FOUND)) {
				/* We found bulk out endpoint */
				if (!(usb_cardp.tx_urb = usb_alloc_urb(0, GFP_KERNEL))) {
					PRINTM(INFO, "Tx URB allocation failed\n");
					goto error;
				}
				usb_cardp.tx_urb_pending = FALSE;

				usb_cardp.bulk_out_size = 
						endpoint->wMaxPacketSize;
				PRINTM(INFO, "Bulk out size is %d\n", 
						endpoint->wMaxPacketSize);
				usb_cardp.bulk_out_endpointAddr = 
						endpoint->bEndpointAddress;
				PRINTM(INFO, "out_endpoint = %d\n",
						endpoint->bEndpointAddress);
				usb_cardp.bulk_out_buffer = 
				kmalloc(MRVDRV_ETH_TX_PACKET_BUFFER_SIZE,
								GFP_DMA);

				if (!usb_cardp.bulk_out_buffer) {
					PRINTM(INFO, "Could not allocate buffer\n");
					goto error;
				}
			}
		}

		/* At this point wlan_add_card() will be called */
		if (!(pwlanpriv = usb_cardp.add(&usb_cardp))) {
			goto error;
		}

		usb_get_dev(udev);
		usb_set_intfdata(intf, &usb_cardp);
		LEAVE();
	
		/* 
		 * return card structure, which can be got back in the 
		 * diconnect function as the ptr
		 * argument.
		 */
		return 0;
	} else {
		PRINTM(INFO, "Discard the Probe request\n");
		PRINTM(INFO, "VID = 0x%X PID = 0x%X\n", udev->descriptor.idVendor,
				udev->descriptor.idProduct);
	}
error:
	if_usb_free(&usb_cardp);	

	LEAVE();
	return (-ENOMEM);
}

/** 
 *  @brief free resource and cleanup
 *  @param udev		pointer to usb_device
 *  @param ptr		pointer to usb_cardp
 *  @return 	   	N/A
 */
static void if_usb_disconnect(struct usb_interface *intf)
{
	struct usb_card_rec	*cardp = usb_get_intfdata(intf);
	wlan_private		*priv = (wlan_private*)cardp->priv;
	wlan_adapter		*Adapter = priv->adapter;	
	
	ENTER();

	/* 
	 * Update Surprie removed to TRUE  
	 *  Free all the URB's allocated 
	 */
	if (!Adapter->SurpriseRemoved) {
		Adapter->SurpriseRemoved = TRUE;
	}

	/* Unlink and free urb */
	if_usb_free(cardp);

	/* card is removed and we can call wlan_remove_card */
	PRINTM(INFO, "call remove card\n");
	cardp->remove(cardp);

	usb_set_intfdata(intf,NULL);
	usb_put_dev(interface_to_usbdev(intf));
        
	LEAVE();
	return ;
}	

/**  
 *  @brief This function transfer the data to the device.
 *  @param priv 	pointer to wlan_private
 *  @param payload	pointer to payload data
 *  @param nb		data length
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int usb_tx_block(wlan_private *priv, u8 *payload, u16 nb)
{
	/* pointer to card structure */
	struct usb_card_rec	*cardp = priv->wlan_dev.card; 
	int			ret = WLAN_STATUS_FAILURE;

	ENTER();

	/* check if device is removed */
	if (priv->adapter->SurpriseRemoved) {
		PRINTM(INFO, "Device removed\n");
		goto tx_ret;
	}

	/* use USB API macro FILL_BULK_URB or API function 
	 * usb_fill_bulk_urb() used to set the 
	 * configuration information to send the bulk URB
	 */
	usb_fill_bulk_urb( cardp->tx_urb, cardp->udev, 
			usb_sndbulkpipe(cardp->udev, 
					cardp->bulk_out_endpointAddr), 
			payload, nb, if_usb_write_bulk_callback, priv);

	cardp->tx_urb->transfer_flags |= URB_ZERO_PACKET;

	if ((ret=usb_submit_urb(cardp->tx_urb, GFP_ATOMIC))) {
			/*  transfer failed */
			PRINTM(INFO, "usb_submit_urb Failed\n");
			
			ret = WLAN_STATUS_FAILURE;
	} else {
			cardp->tx_urb_pending = TRUE;
			PRINTM(INFO, "usb_submit_urb Success\n");
			ret = WLAN_STATUS_SUCCESS;
	}

tx_ret:
	LEAVE();
	return ret;
}

/**  
 *  @brief This function submit the rx data to OS
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int if_usb_submit_rx_urb(wlan_private *priv)
{
	struct usb_card_rec	*cardp = priv->wlan_dev.card;
	int			ret = WLAN_STATUS_FAILURE;
	
	ENTER();

	if(!(cardp->skb = dev_alloc_skb(MRVDRV_ETH_RX_PACKET_BUFFER_SIZE))) {
		PRINTM(FATAL, "No free skb\n");
		goto rx_ret;
	}

	/* Fill the receive configuration URB and initialise the Rx call back */
	usb_fill_bulk_urb(cardp->rx_urb, cardp->udev,
			usb_rcvbulkpipe(cardp->udev, cardp->bulk_in_endpointAddr),
			cardp->skb->tail + IPFIELD_ALIGN_OFFSET,
			MRVDRV_ETH_RX_PACKET_BUFFER_SIZE,
			if_usb_receive, priv);

	cardp->rx_urb->transfer_flags |= URB_ZERO_PACKET;

	PRINTM(INFO, "Pointer for rx_urb %p\n", cardp->rx_urb); 
	if ((ret = usb_submit_urb(cardp->rx_urb, GFP_ATOMIC))) {
		/* handle failure conditions */
		PRINTM(INFO, "Submit Rx URB failed\n");
		ret = WLAN_STATUS_FAILURE;
	} 
	else {
		cardp->rx_urb_pending = TRUE;
		PRINTM(INFO, "Submit Rx URB Success\n");
		ret = WLAN_STATUS_SUCCESS;
	}

rx_ret:
	LEAVE();
	return ret;
}

/**  
 *  @brief This function reads of the packet into the upload buff, 
 *  wake up the main thread and initialise the Rx callack.
 *  
 *  @param urb		pointer to struct urb
 *  @return 	   	N/A
 */
static void if_usb_receive(struct urb *urb, struct pt_regs *regs)
{
	wlan_private	*priv = (wlan_private *)urb->context;
	struct		usb_card_rec *cardp = 
			(struct usb_card_rec *)priv->wlan_dev.card;

	int		RecvLength = urb->actual_length;
	u8		*RecvBuff = NULL;
	u32		RecvType;
	u8		*cmdBuf;

	FWSyncHeader	SyncFWHeader;

#define MRVDRV_MIN_PKT_LEN	30

	ENTER();

	cardp->rx_urb_pending = FALSE;

	if (RecvLength) {
		if (urb->status) {
			PRINTM(INFO, "URB Status is failed\n");
			if (cardp->skb) {
				kfree_skb(cardp->skb);
				cardp->skb = NULL;
			}
			goto setup_for_next;
		}

		RecvBuff = cardp->skb->data + IPFIELD_ALIGN_OFFSET;
		memcpy(&RecvType, RecvBuff, sizeof(u32));
		PRINTM(INFO, "Recv length = 0x%x\n", RecvLength);
		PRINTM(INFO, "Receive type = 0x%X\n", RecvType);
		RecvType = wlan_le32_to_cpu(RecvType);
		PRINTM(INFO, "Receive type after = 0x%X\n", RecvType);
	} else if (urb->status) {
		if (!cardp->rx_urb_recall) {
			PRINTM(INFO, "Card Is Removed\n");
			priv->adapter->SurpriseRemoved = TRUE;
			// Wake up main thread to handle card removal.
			if_usb_interrupt(priv, 0, 0);
		}
		goto rx_exit;
	}

	if (cardp->IsFWDnld) {
		
		memcpy(&SyncFWHeader, RecvBuff, sizeof(FWSyncHeader));


		if (!SyncFWHeader.Cmd) {
			PRINTM(INFO, "FW received Blk with correct CRC\n");
			PRINTM(INFO, "FW received Blk SeqNum = %d\n",SyncFWHeader.SeqNum);
			cardp->CRC_OK = TRUE;
		} else {
			PRINTM(INFO, "FW received Blk with CRC error\n");
			cardp->CRC_OK = FALSE;
		}
		
		if (cardp->skb) {
			kfree_skb(cardp->skb);
			cardp->skb = NULL;
		}

		if (cardp->FWFinalBlk) {
			cardp->FwDnldOver = TRUE;
			cardp->IsFWDnld = FALSE;
			goto rx_exit;
		}

		if_prog_firmware(priv);

		if_usb_submit_rx_urb(priv);

		goto rx_exit;
	} 

	switch (RecvType) {
		case CMD_TYPE_DATA:
			if (RecvLength > MRVDRV_ETH_RX_PACKET_BUFFER_SIZE + 
				MESSAGE_HEADER_LEN || RecvLength < MRVDRV_MIN_PKT_LEN) {
				PRINTM(INFO, "Packet length is Invalid\n");

				if (cardp->skb) {
					kfree_skb(cardp->skb);
					cardp->skb = NULL;
				}

				break;
			}

			usb_int_cause |= HIS_RxUpLdRdy;
			skb_reserve(cardp->skb, IPFIELD_ALIGN_OFFSET);
		        skb_put(cardp->skb, RecvLength);
			skb_pull(cardp->skb, MESSAGE_HEADER_LEN);
			list_add_tail((struct list_head *)cardp->skb,
				(struct list_head *)&priv->adapter->RxSkbQ);	

			priv->wlan_dev.upld_len = (RecvLength - MESSAGE_HEADER_LEN);

			if_usb_interrupt(priv, HIS_RxUpLdRdy, RX_SUCCESS);
			break;

		case CMD_TYPE_REQUEST: 
			if (RecvLength > MRVDRV_SIZE_OF_CMD_BUFFER) {
				PRINTM(INFO, "The receive buffer is too large\n");
				if (cardp->skb) {
					kfree_skb(cardp->skb);
					cardp->skb = NULL;
				}

				break;	
			}

			/* take care of CurCmd = NULL case by reading the 
			 * data to clear the interrupt */
			if (!priv->adapter->CurCmd) {
				cmdBuf = priv->wlan_dev.upld_buf;
				priv->adapter->HisRegCpy &= ~HIS_CmdUpLdRdy;
			} else {
				cmdBuf = priv->adapter->CurCmd->BufVirtualAddr;
			}

			usb_int_cause |= HIS_CmdUpLdRdy;
			priv->wlan_dev.upld_len = (RecvLength - MESSAGE_HEADER_LEN);
			memcpy(cmdBuf, RecvBuff+MESSAGE_HEADER_LEN,
				       		priv->wlan_dev.upld_len);
			
			if (cardp->skb) {
				kfree_skb(cardp->skb);
				cardp->skb = NULL;
			}
			PRINTM(INFO, "Wake up main thread to handle cmd response\n");
			if_usb_interrupt(priv, HIS_CmdUpLdRdy, RX_SUCCESS);
			break;
	
		case CMD_TYPE_INDICATION:
			/* Event cause handling */
			priv->adapter->EventCause = *(u32 *)(RecvBuff + MESSAGE_HEADER_LEN);
		
			PRINTM(INFO, "**EVENT** 0x%X\n", priv->adapter->EventCause);
			if (priv->adapter->EventCause & 0xffff0000) {
				SendTxFeedback(priv); 	
				break;
			}
			priv->adapter->EventCause = wlan_le32_to_cpu(priv->adapter->EventCause) << 3; 

			usb_int_cause |= HIS_CardEvent;
			if (cardp->skb) {
				kfree_skb(cardp->skb);
				cardp->skb = NULL;
			}
			if_usb_interrupt(priv, HIS_CardEvent, RX_SUCCESS);
			
			break;

		default:
			if (cardp->skb) {
				kfree_skb(cardp->skb);
				cardp->skb = NULL;
			}
			break;
	}
	
setup_for_next:
	if_usb_submit_rx_urb(priv);
rx_exit:
	LEAVE();
	return;
}

/********************************************************
		Global Functions
********************************************************/

/** 
 *  @brief This function downloads data to FW  
 *  @param priv		pointer to wlan_private structure
 *  @param type		type of data
 *  @param buf		pointer to data buffer
 *  @param len		number of bytes
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int sbi_host_to_card(wlan_private *priv, u8 type, u8 *payload, u16 nb)
{
	int			ret = WLAN_STATUS_FAILURE;
	u32			tmp;
	struct usb_card_rec	*cardp = 
				(struct usb_card_rec *)priv->wlan_dev.card;

	ENTER();

	PRINTM(INFO, "*** type = %u\n",type);
	PRINTM(INFO, "Size after = %d\n", nb);
	
	if (type == MVMS_CMD) {
		tmp = wlan_cpu_to_le32(CMD_TYPE_REQUEST);
		priv->wlan_dev.dnld_sent = DNLD_CMD_SENT;
		memcpy(cardp->bulk_out_buffer, (u8 *)&tmp, MESSAGE_HEADER_LEN);
		
	} else {
		tmp = wlan_cpu_to_le32(CMD_TYPE_DATA);
		priv->wlan_dev.dnld_sent = DNLD_DATA_SENT;
		memcpy(cardp->bulk_out_buffer, (u8*)&tmp, MESSAGE_HEADER_LEN);
	}
	
	memcpy((cardp->bulk_out_buffer + MESSAGE_HEADER_LEN), payload, nb);

	ret = usb_tx_block(priv, cardp->bulk_out_buffer, nb + MESSAGE_HEADER_LEN);
	
	return ret;
}

/**  
 *  @brief This function reads data from FW.
 *  @param priv		pointer to wlan_private structure
 *  @param type		type of data to read
 *  @param nb		number of data bytes
 *  @param payload	buffer for payload data
 *  @param npayload	number of bytes to read
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_card_to_host(wlan_private *priv, u32 type, u32 *nb, u8 *payload,
							u16 npayload)
{
	/* handle dummy conditions as upload buf is ready in the 
	 * Rx callback
	 */
       return WLAN_STATUS_SUCCESS;
} 

/** 
 *  @brief This is a dummy function
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_reenable_host_interrupt(wlan_private *priv, u8 bits)
{
	return WLAN_STATUS_SUCCESS;
}

int sbi_get_int_status(wlan_private *priv , u8 *ireg)
{
	unsigned long flags;
	ENTER();

	spin_lock_irqsave(&driver_lock, flags);
	*ireg = usb_int_cause;
	usb_int_cause = 0;
	spin_unlock_irqrestore(&driver_lock, flags);

	PRINTM(INFO, "Int cause is 0x%X\n", *ireg);
	
	LEAVE();

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This is a dummy function
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_read_event_cause(wlan_private *priv)
{
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This is a dummy function
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_get_cis_info(wlan_private *priv)
{
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief This is a dummy function
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_unregister_dev(wlan_private *priv)
{
	return WLAN_STATUS_SUCCESS;
}

/**  
 *  @brief This function registers driver.
 *  @param add		pointer to add_card callback function
 *  @param remove	pointer to remove card callback function
 *  @param arg		pointer to call back function parameter
 *  @return 	   	pointer to usb_cardp
 */
int *sbi_register(wlan_notifier_fn_add add, wlan_notifier_fn_remove remove, 
								void *arg)
{
	ENTER();

	/* 
	 * Make use of the global variable cardp,
	 * to assign the call backs
	 */

	PRINTM(INFO, "udev pointer is at %p\n", usb_cardp.udev);

	usb_cardp.add = (usb_notifier_fn_add)add;
	usb_cardp.remove = (usb_notifier_fn_remove)remove;
	
	/* 
	 * API registers the Marvell USB driver
	 * to the USB system
	 */
	usb_register(&if_usb_driver); 

	LEAVE();

	/* Return the cardp structure to wlan layer */
	return (int *)&usb_cardp;
}

/**  
 *  @brief This function removes usb driver.
 *  @return 	   	N/A
 */
void sbi_unregister(void)
{
	ENTER();

	// Need to send a Reset command to device before USB resources freed and wlan_remove_card() called,
	// then device can handle FW download again.
	if (pwlanpriv) {
		PrepareAndSendCommand(pwlanpriv, HostCmd_CMD_802_11_RESET,
			HostCmd_ACT_HALT, 0, 0, NULL);
		msleep_interruptible(10);
	}

	/* API unregisters thedriver from USB subsystem */
	usb_deregister(&if_usb_driver);

	LEAVE();
	return ;
}

/** 
 *  @brief This is a dummy function
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_probe_card(void * card_p)
{
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief  This function register usb device and initialize parameter
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int sbi_register_dev(wlan_private *priv)
{

	struct usb_card_rec *cardp = 
			(struct usb_card_rec *)priv->wlan_dev.card;
	ENTER(); 

	cardp->priv = priv;
	cardp->eth_dev = priv->wlan_dev.netdev;
	priv->hotplug_device = &(cardp->udev->dev);
	/* 
	 * Receive hander is initialised for command
	 * or data receive
	 */ 
	PRINTM(INFO, "cardp = %p  priv = %p\n", cardp, priv);
	if (if_usb_submit_rx_urb(priv) < 0) {
		PRINTM(INFO, "URB submission is failed\n");
		sbi_unregister();
		LEAVE();
		return WLAN_STATUS_FAILURE;
	}
	
	PRINTM(INFO, "udev pointer is at %p\n", cardp->udev);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief  stub function
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_verify_fw_download(wlan_private *priv)
{
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief  This function download FW
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int if_prog_firmware(wlan_private *priv)
{
	struct usb_card_rec	*cardp = priv->wlan_dev.card;
	FWData			fwdata;
	
	u8 *firmware = priv->firmware->data;

	ENTER();

	if (!cardp->CRC_OK) {
		cardp->TotalBytes = cardp->FwLastBlkSent;
		cardp->FWSeqNum = cardp->LastSeqNum - 1;
	}

	PRINTM(INFO, "TotalBytes = %d\n", cardp->TotalBytes);


	memcpy(&fwdata.fwheader, &firmware[cardp->TotalBytes], sizeof(FWHeader));


	cardp->FwLastBlkSent = cardp->TotalBytes;
	cardp->TotalBytes += sizeof(FWHeader);

	PRINTM(INFO, "Copy Data\n");	
	memcpy(fwdata.data, &firmware[cardp->TotalBytes], fwdata.fwheader.DataLength);
	
	PRINTM(INFO, "Data Length = %d\n", fwdata.fwheader.DataLength);

	cardp->FWSeqNum = cardp->FWSeqNum + 1;

	fwdata.SeqNum = cardp->FWSeqNum;
	cardp->LastSeqNum = fwdata.SeqNum;
	cardp->TotalBytes += fwdata.fwheader.DataLength;

	if (fwdata.fwheader.DnldCmd == FW_HAS_DATA_TO_RECV) {
		PRINTM(INFO, "There is data to follow\n");
		PRINTM(INFO, "SeqNum = %d TotalBytes = %d\n", cardp->FWSeqNum, cardp->TotalBytes);
		memcpy(cardp->bulk_out_buffer, &fwdata.fwheader, FW_DATA_XMIT_SIZE);
		usb_tx_block(priv, cardp->bulk_out_buffer, FW_DATA_XMIT_SIZE);
		
	} else if (fwdata.fwheader.DnldCmd == FW_HAS_LAST_BLOCK) {
		PRINTM(INFO, "Host has finished FW downloading\n");
		PRINTM(INFO, "Donwloading FW JUMP BLOCK\n");
		memcpy(cardp->bulk_out_buffer, &fwdata.fwheader, FW_DATA_XMIT_SIZE);
		usb_tx_block(priv, cardp->bulk_out_buffer, FW_DATA_XMIT_SIZE);
		
		cardp->FWFinalBlk = TRUE;
	}

	PRINTM(INFO, "The firmware download is done size is %d\n", cardp->TotalBytes); 
	
	LEAVE();

	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief This function download fw
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_prog_firmware(wlan_private *priv)
{
	struct usb_card_rec	*cardp = priv->wlan_dev.card;
	int	i = 0;
	
	ENTER();

	usb_fw_ready = FALSE;
	
	cardp->TotalBytes = 0;
	cardp->FwLastBlkSent = 0;
	cardp->CRC_OK = TRUE;
	cardp->FwDnldOver = FALSE; 
	cardp->IsFWDnld = TRUE;
	cardp->FWSeqNum = -1;
	cardp->TotalBytes = 0;
	cardp->FWFinalBlk = FALSE;
	
	if_prog_firmware(priv);
	
	do {
		PRINTM(INFO, "Wlan sched timeout\n");
		i ++;
		msleep_interruptible(100);
		if (priv->adapter->SurpriseRemoved || i>=20)
			break;
	} while (!cardp->FwDnldOver);

	if (!cardp->FwDnldOver) {
		PRINTM(FATAL, "FW download failure, time = %d ms\n", i*100);
		LEAVE();
		return WLAN_STATUS_FAILURE;
	}

	if_usb_submit_rx_urb(priv);

	/* Delay 200 ms to waiting for the FW ready */
	msleep_interruptible(200);
	
	usb_fw_ready = TRUE;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This is a dummy function
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_enable_host_int (wlan_private *priv)	
{
	return WLAN_STATUS_SUCCESS;	
}

/** 
 *  @brief This is a dummy function
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_disable_host_int(wlan_private *priv)		
{
	return WLAN_STATUS_SUCCESS;
}

#ifdef ENABLE_PM
/** 
 *  @brief This is a dummy function
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
inline int sbi_suspend(wlan_private * priv)
{
   	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This is a dummy function
 *  @param priv		pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
inline int sbi_resume(wlan_private * priv)
{
   	return WLAN_STATUS_SUCCESS;
}
#endif


