/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil -*- */
/* vi: set expandtab shiftwidth=4 tabstop=4 textwidth=78: */

/** @file wlan_scan.h
  *
  * @brief Interface for the wlan network scan routines
  *
  * Driver interface functions and type declarations for the scan module
  *   implemented in wlan_scan.c.
  *
  * @sa wlan_scan.c
  *
  * (c) Copyright  2003-2006, Marvell International Ltd. 
  * All Rights Reserved
  *
  * This software file (the "File") is distributed by Marvell International 
  * Ltd. under the terms of the GNU General Public License Version 2, June 1991 
  * (the "License").  You may use, redistribute and/or modify this File in 
  * accordance with the terms and conditions of the License, a copy of which 
  * is available along with the File in the license.txt file or by writing to 
  * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
  * 02111-1307 or on the worldwide web at http://www.gnu.org/licenses/gpl.txt.
  *
  * THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE 
  * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE 
  * ARE EXPRESSLY DISCLAIMED.  The License provides additional details about 
  * this warranty disclaimer.
  *
  */
/*************************************************************
Change Log:
    01/11/06: Initial revision. New scan code, relocate related functions

************************************************************/

#ifndef _WLAN_SCAN_H
#define _WLAN_SCAN_H

/**
 *  @brief Maximum number of channels that can be sent in a setuserscan ioctl
 *
 *  @sa wlan_ioctl_user_scan_cfg
 */
#define WLAN_IOCTL_USER_SCAN_CHAN_MAX  50

//! Infrastructure BSS scan type in wlan_scan_cmd_config
#define WLAN_SCAN_BSS_TYPE_BSS         1

//! Adhoc BSS scan type in wlan_scan_cmd_config
#define WLAN_SCAN_BSS_TYPE_IBSS        2

//! Adhoc or Infrastructure BSS scan type in wlan_scan_cmd_config, no filter
#define WLAN_SCAN_BSS_TYPE_ANY         3


/**
 * @brief Structure used internally in the wlan driver to configure a scan.
 *
 * Sent to the command processing module to configure the firmware
 *   scan command prepared by wlan_cmd_802_11_scan.
 *
 * @sa wlan_scan_networks
 *
 */
typedef struct {
    /**
     *  @brief BSS Type to be sent in the firmware command
     *
     *  Field can be used to restrict the types of networks returned in the
     *    scan.  Valid settings are:
     *
     *   - WLAN_SCAN_BSS_TYPE_BSS  (infrastructure)
     *   - WLAN_SCAN_BSS_TYPE_IBSS (adhoc)
     *   - WLAN_SCAN_BSS_TYPE_ANY  (unrestricted, adhoc and infrastructure)
     */
    u8  bssType;

    /**
     *  @brief Specific BSSID used to filter scan results in the firmware
     */
    u8  specificBSSID[MRVDRV_ETH_ADDR_LEN];

    /**
     *  @brief Length of TLVs sent in command starting at tlvBuffer
     */
    int tlvBufferLen;

    /**
     *  @brief SSID TLV(s) and ChanList TLVs to be sent in the firmware command
     *
     *  @sa TLV_TYPE_CHANLIST, MrvlIEtypes_ChanListParamSet_t
     *  @sa TLV_TYPE_SSID, MrvlIEtypes_SsIdParamSet_t
     */
    u8  tlvBuffer[1]; //!< SSID TLV(s) and ChanList TLVs are stored here
} wlan_scan_cmd_config;

/**
 *  @brief IOCTL channel sub-structure sent in wlan_ioctl_user_scan_cfg
 *
 *  Multiple instances of this structure are included in the IOCTL command
 *   to configure a instance of a scan on the specific channel.
 */
typedef struct {
    u8  chanNumber; //!< Channel Number to scan
    u8  radioType;  //!< Radio type: 'B/G' Band = 0, 'A' Band = 1
    u8  scanType;   //!< Scan type: Active = 0, Passive = 1
    u16 scanTime;   //!< Scan duration in milliseconds; if 0 default used
} wlan_ioctl_user_scan_chan;

/**
 *  @brief IOCTL input structure to configure an immediate scan cmd to firmware
 *
 *  Used in the setuserscan (WLAN_SET_USER_SCAN) private ioctl.  Specifies
 *   a number of parameters to be used in general for the scan as well
 *   as a channel list (wlan_ioctl_user_scan_chan) for each scan period
 *   desired.
 *
 *  @sa wlan_set_user_scan_ioctl
 */
typedef struct {

    /**
     *  @brief Flag set to keep the previous scan table intact
     *
     *  If set, the scan results will accumulate, replacing any previous
     *   matched entries for a BSS with the new scan data
     */
    u8   keepPreviousScan; //!< Do not erase the existing scan results

    /**
     *  @brief BSS Type to be sent in the firmware command
     *
     *  Field can be used to restrict the types of networks returned in the
     *    scan.  Valid settings are:
     *
     *   - WLAN_SCAN_BSS_TYPE_BSS  (infrastructure)
     *   - WLAN_SCAN_BSS_TYPE_IBSS (adhoc)
     *   - WLAN_SCAN_BSS_TYPE_ANY  (unrestricted, adhoc and infrastructure)
     */
    u8   bssType;

    /**
     *  @brief Configure the number of probe requests for active chan scans
     */
    u8   numProbes;

    /**
     *  @brief BSSID filter sent in the firmware command to limit the results
     */
    u8   specificBSSID[MRVDRV_ETH_ADDR_LEN];

    /**
     *  @brief SSID filter sent in the firmware command to limit the results
     */
    char specificSSID[MRVDRV_MAX_SSID_LENGTH + 1];

    /**
     *  @brief Variable number (fixed maximum) of channels to scan up
     */
    wlan_ioctl_user_scan_chan chanList[WLAN_IOCTL_USER_SCAN_CHAN_MAX];
} __attribute__ ((packed)) wlan_ioctl_user_scan_cfg;

/**
 *  @brief Sub-structure passed in wlan_ioctl_get_scan_table_entry for each BSS
 *
 *  Fixed field information returned for the scan response in the IOCTL
 *    response.
 */
typedef struct {
    u8  bssid[6];   //!< BSSID of this network
    u8  channel;    //!< Channel this beacon/probe response was detected
    u8  rssi;       //!< RSSI for the received packet
    u64 networkTSF; //!< TSF value from the firmware at packet reception
}  __attribute__ ((packed)) wlan_ioctl_get_scan_table_fixed;

/**
 *  @brief Structure passed in the wlan_ioctl_get_scan_table_info for each
 *         BSS returned in the WLAN_GET_SCAN_RESP IOCTL
 *
 *  @sa wlan_get_scan_table_ioctl
 */
typedef struct {

    /**
     *  @brief Fixed field length included in the response.
     *
     *  Length value is included so future fixed fields can be added to the
     *   response without breaking backwards compatibility.  Use the length
     *   to find the offset for the bssInfoLength field, not a sizeof() calc.
     */
    u32 fixedFieldLength;

    /**
     *  @brief Always present, fixed length data fields for the BSS
     */
    wlan_ioctl_get_scan_table_fixed fixedFields;

    /**
     *  @brief Length of the BSS Information (probe resp or beacon) that
     *         follows starting at bssInfoBuffer
     */
    u32 bssInfoLength;

    /**
     *  @brief Probe response or beacon scanned for the BSS.
     *
     *  Field layout:
     *   - TSF              8 octets
     *   - Beacon Interval  2 octets
     *   - Capability Info  2 octets
     *
     *   - IEEE Infomation Elements; variable number & length per 802.11 spec
     */
    u8  bssInfoBuffer[1];
}  __attribute__ ((packed)) wlan_ioctl_get_scan_table_entry;

/**
 *  @brief WLAN_GET_SCAN_RESP private IOCTL struct to retrieve the scan table
 *
 *  @sa wlan_get_scan_table_ioctl
 */
typedef struct {

    /**
     *  - Zero based scan entry to start retrieval in command request
     *  - Number of scans entires returned in command response
     */
     u32 scanNumber;

     /**
      * Buffer marker for multiple wlan_ioctl_get_scan_table_entry structures.
      *   Each struct is padded to the nearest 32 bit boundary.
      */
     u8  scan_table_entry_buffer[1];

 } __attribute__ ((packed)) wlan_ioctl_get_scan_table_info;

/**
 *  @brief Structure used to store information for each beacon/probe response
 */
typedef struct {
    u8 MacAddress[ETH_ALEN];

    struct WLAN_802_11_SSID Ssid;

    /* WEP encryption requirement */
    u32 Privacy;

    /* receive signal strength in dBm */
    long Rssi;

    u32 Channel;

    u16 BeaconPeriod;

    u32 ATIMWindow;

    WLAN_802_11_NETWORK_TYPE NetworkTypeInUse;
    WLAN_802_11_NETWORK_INFRASTRUCTURE InfrastructureMode;
    WLAN_802_11_RATES SupportedRates;

    int extra_ie;

    u8 TimeStamp[8];  //!< TSF value included in the beacon/probe response
    IEEEtypes_PhyParamSet_t PhyParamSet;
    IEEEtypes_SsParamSet_t  SsParamSet;
    IEEEtypes_CapInfo_t Cap;
    u8 DataRates[WLAN_SUPPORTED_RATES];

    u64 networkTSF; //!< TSF timestamp from the current firmware TSF


    IEEEtypes_CountryInfoFullSet_t CountryInfo;

    struct WPA_SUPPLICANT   wpa_supplicant;
    struct WPA_SUPPLICANT   wpa2_supplicant;


} BSSDescriptor_t;

extern int SSIDcmp(struct WLAN_802_11_SSID* ssid1, struct WLAN_802_11_SSID* ssid2);
extern int FindSSIDInList(wlan_adapter* Adapter,
                          struct WLAN_802_11_SSID* ssid, u8* bssid, int mode );
extern int FindBestSSIDInList(wlan_adapter* Adapter);
extern int FindBSSIDInList(wlan_adapter* Adapter, u8* bssid, int mode);

extern int FindBestNetworkSsid(wlan_private* priv, struct WLAN_802_11_SSID* pSSID);

extern int SendSpecificSSIDScan(wlan_private* priv, 
                                struct WLAN_802_11_SSID* pRequestedSSID,
                                u8 keepPreviousScan);
extern int SendSpecificBSSIDScan(wlan_private* priv, 
                                 u8* bssid,
                                 u8 keepPreviousScan);

extern u8 wlan_scan_radio_to_band(u8 scanBand);

extern int wlan_get_scan_table_ioctl(wlan_private* priv, struct iwreq* wrq);
extern int wlan_set_user_scan_ioctl(wlan_private* priv, struct iwreq* wrq);

extern int wlan_associate(wlan_private* priv, BSSDescriptor_t* pBSSDesc);

extern int wlan_cmd_802_11_scan(wlan_private* priv,
                                struct HostCmd_DS_COMMAND* cmd,
                                void* pdata_buf);

extern int wlan_ret_802_11_scan(wlan_private* priv, struct HostCmd_DS_COMMAND* resp);

extern int wlan_extscan_ioctl(wlan_private* priv, struct ifreq* req);


#ifdef __KERNEL__
extern int wlan_get_scan(struct net_device* dev, struct iw_request_info* info,
                         struct iw_point* dwrq, char* extra);
extern int wlan_set_scan(struct net_device* dev, struct iw_request_info* info,
                         struct iw_param* vwrq, char* extra);
#endif

#endif /* _WLAN_SCAN_H */
