/* bootldr-debug.S
 * Copyright (c) 2004 Andrew Zabolotny
 *
 * Debugging code, split away to avoid cluttering main file.
 */

@ Shortcut to make life easier
#if defined CONFIG_LAB_DEBUG_FFUART || defined CONFIG_LAB_DEBUG_BTUART \
 || defined CONFIG_LAB_DEBUG_STUART
#define CONFIG_LAB_DEBUG_SERIAL
#endif

@-----------------------------------------------------------------------------
@	Printing Functions
@-----------------------------------------------------------------------------

@-----------------------------------------------------------------------------
@ out_str -- prints the string pointed to by register R0
@ IN:		r0 - address of the string to display
@ MOD:		r0-r5, advances r9
@-----------------------------------------------------------------------------
out_str:	mov	r4, lr
		mov	r5, r0
1:		ldrb	r0, [r5], #1
		tst	r0, r0
		moveq	pc, r4
		adr	lr, 1b
		b	out_char

@-----------------------------------------------------------------------------
@ out_hex_word -- prints the 4 bytes in R0 as 8 hex ascii characters
@ IN:		r0 - the number to print
@ MODIFIES:	r0-r7, advances r9
@-----------------------------------------------------------------------------
out_hex_word:	mov     r6, lr
		mov	r7, r0
		mov	r0, r0, lsr #24
		bl      out_hex_byte
		mov	r0, r7, lsr #16
		bl      out_hex_byte
		mov	r0, r7, lsr #8
		bl      out_hex_byte
		mov	r0, r7
		mov	lr, r6
		@ fall through

@-----------------------------------------------------------------------------
@ out_hex_byte -- prints the lower byte of R0 as a hexadecimal number
@ IN:		r0 - the number to print
@ MODIFIES:	r0-r5, advances r9
@-----------------------------------------------------------------------------
out_hex_byte:	mov	r4, lr
		mov	r5, r0
		mov	r0, r0, lsr #4
		bl	out_hex_nibble
		mov	lr, r4
		mov	r0, r5
		@ fall through

@-----------------------------------------------------------------------------
@ out_hex_nibble -- prints the least-significant nibble in R0 as a hex digit
@ IN:		r0 - nibble to write as Hex
@ MODIFIES:	r0-r3, advances r9
@-----------------------------------------------------------------------------
out_hex_nibble:	and	r0, r0, #0xf
		add	r0, r0, #'0'
		cmp	r0, #'9'
		addgt	r0, r0, #7
		b	out_char

#ifdef CONFIG_LAB_DEBUG_SERIAL

@-----------------------------------------------------------------------------
@ out_init - Initialize Serial Communications
@
@ Following reset, the UART is disabled. So, we do the following:
@	1) set the uart gpio lines in their alternate reg state
@	2) set the right baud rate (115200) and set to 8 bit/char mode
@	3) enable the fifos
@	4) turn on the uart
@-----------------------------------------------------------------------------
out_init:
		ldr	r1, UART_BASE
		@ disable the ffuart IRQs just in case
		mov	r0, #0
		str	r0, [r1, #__OFFSET(FFIER)]

		@ now we start setting up the uart
		@ lets set up the baud rate
		@ baud rate = 14.7456 * 10^6 / (16 * x) = 921600/x
		@ 115200 => x = 8
		@ get access to baud control registers
		mov	r0, #LCR_DLAB
		str	r0, [r1, #__OFFSET(FFLCR)]
		@ set the baud rate
		mov	r0, #0x8
		str	r0, [r1, #__OFFSET(FFDLL)]
		mov	r0, #0x0
		str	r0, [r1, #__OFFSET(FFDLH)]
		@ clear the latch bit and set to 8 bit per character mode
		mov	r0, #(LCR_WLS1|LCR_WLS0)
		str	r0, [r1, #__OFFSET(FFLCR)]
		@ turn off modem stuff
		@mov	r0, #0
		@str	r0, [r1, #__OFFSET(FFMCR)]
		@ enable the fifos and reset them for good measure
		mov	r0, #(FCR_TRFIFOE|FCR_RESETTF|FCR_RESETRF) @FCR_ITL_32
		str	r0, [r1, #__OFFSET(FFFCR)]

		@ enable the uart
		mov	r0,#IER_UUE
		str	r0,[r1, #__OFFSET(FFIER)]

		mov	pc, lr		@ All done, return

@-----------------------------------------------------------------------------
@ out_char -- prints the character in R0
@ IN:		r0 - the character
@ MODIFIES:	r1,r2
@-----------------------------------------------------------------------------
out_char:	ldr	r1, UART_BASE
1:		ldr     r2, [r1, #__OFFSET(FFLSR)]@ Get UART status
		tst     r2, #LSR_TEMT		@ Check TX empty
		beq     1b
		str	r0, [r1, #__OFFSET(FFTHR)]
		ldr     r0, [r1, #__OFFSET(FFLSR)]

		mov	pc, lr

UART_BASE:
#if defined CONFIG_LAB_DEBUG_FFUART
		.long	__BASE(FFUART)
#elif defined CONFIG_LAB_DEBUG_BTUART
		.long	__BASE(BTUART)
#elif defined CONFIG_LAB_DEBUG_STUART
		.long	__BASE(STUART)
#endif

#endif /* CONFIG_LAB_DEBUG_SERIAL */

#ifdef CONFIG_LAB_DEBUG_JTAG

@-----------------------------------------------------------------------------
@ out_char -- prints the character in R0 through JTAG
@ IN:		r0 - the character
@ MODIFIES:	r1,r2
@-----------------------------------------------------------------------------
out_char:
1:		mrc	p14, 0, r15, c14, c0, 0
		bvs	1b
		mcr	p14, 0, r0, c8, c0, 0

out_init:	@ fake init routine
		mov	pc, lr

#endif /* CONFIG_LAB_DEBUG_JTAG */

@-----------------------------------------------------------------------------
@	On-screen message display
@-----------------------------------------------------------------------------

#ifdef CONFIG_LAB_DEBUG_DISPLAY

@-----------------------------------------------------------------------------
@ out_init -- initialize display output
@-----------------------------------------------------------------------------
out_init:
		ldr	r9, DISPLAY_ADDR
		mov	pc, lr

@-----------------------------------------------------------------------------
@ out_char -- prints the character in R0 to the display
@ IN:		r0 - character
@ MOD:		r0-r3, advances r9
@-----------------------------------------------------------------------------
out_char:
		tst	r12, #BF_NEED_RESTORE_R9
		ldrne	r9, saved_r9
		
		cmp	r0, #'\n'
		bne	2f

		@ Execute a delay on new line since otherwise we can
		@ miss important events since the text goes too quick
		mov	r2, #0x4000000
0:		subs	r2, r2, #1
		bne	0b

		@ Now skip to the beginning of the next line
		ldr	r2, DISPLAY_ADDR
		sub	r2, r9, r2
1:		cmp	r2, #(CONFIG_LAB_DISPLAY_W*CONFIG_LAB_DISPLAY_BPP)
		subge	r2, r2, #(CONFIG_LAB_DISPLAY_W*CONFIG_LAB_DISPLAY_BPP*8)
		bge	1b

		sub	r9, r9, r2
		add	r9, r9, #(CONFIG_LAB_DISPLAY_W*CONFIG_LAB_DISPLAY_BPP*8)

		@ Clear the next line
		b	6f

2:		adr	r1, disp_font
		@ Get the address of character bitmap in r1
		and	r0, r0, #0xff
		add	r1, r1, r0, lsl #3
3:		@ Get the next bitmap of one character scanline
		ldrb	r2, [r1], #1
		@ Use the 9th bit as the loop counter
		orr	r2, r2, #0x100
4:		tst	r2, #0x80
		moveq	r3, #0
		mvnne	r3, #0
#if CONFIG_LAB_DISPLAY_BPP == 1
		strb	r3, [r9], #1
#endif
#if CONFIG_LAB_DISPLAY_BPP == 2
		strh	r3, [r9], #2
#endif
#if CONFIG_LAB_DISPLAY_BPP == 4
		str	r3, [r9], #4
#endif
		@ Shift the character line bitmap one bit
		mov	r2, r2, lsl #1
		tst	r2, #0x10000	@ The 8th bit reached position 8+8?
		beq	4b
		@ Advance to next scanline
		add	r9, r9, #((CONFIG_LAB_DISPLAY_W-8)*CONFIG_LAB_DISPLAY_BPP)

		@ Check for end of character
		tst	r1, #7
		bne	3b

		@ Advance r9 to next cell
		sub	r9, r9, #((CONFIG_LAB_DISPLAY_W-1)*CONFIG_LAB_DISPLAY_BPP*8)

		@ Check if r9 has reached the right margin
		ldr	r0, DISPLAY_ADDR
		sub	r0, r9, r0
5:		cmp	r0, #(CONFIG_LAB_DISPLAY_W*CONFIG_LAB_DISPLAY_BPP)
		blt	8f
		subs	r0, r0, #(CONFIG_LAB_DISPLAY_W*CONFIG_LAB_DISPLAY_BPP*8)
		bge	5b

		@ Advance to next character line
		add	r9, r9, #(CONFIG_LAB_DISPLAY_W*CONFIG_LAB_DISPLAY_BPP*7)

		@ Check if we're past the end of screen
		ldr	r0, DISPLAY_END
		cmp	r9, r0
		@ If so, start printing from the top line
		ldrge	r9, DISPLAY_ADDR

		@ Clear the next line
6:		mov	r0, r9
		mov	r1, #0

		add	r2, r0, #(CONFIG_LAB_DISPLAY_W*CONFIG_LAB_DISPLAY_BPP*8)
7:		str	r1, [r0], #4
		cmp	r0, r2
		blt	7b
		
8:		tst	r12, #BF_NEED_RESTORE_R9
		str	r9, saved_r9

		mov	pc, lr

DISPLAY_ADDR:	.long	CONFIG_LAB_DISPLAY_ADDR
DISPLAY_END:	.long	CONFIG_LAB_DISPLAY_ADDR + \
		CONFIG_LAB_DISPLAY_W * CONFIG_LAB_DISPLAY_H * CONFIG_LAB_DISPLAY_BPP

@ algorithm relies on lower 3 bits of char address to be 0
		.align	3
@ The 8x8 font shamelessly borrowed from drivers/video/console/font_8x8.c
disp_font:	.byte	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
		.byte	0x7e, 0x81, 0xa5, 0x81, 0xbd, 0x99, 0x81, 0x7e
		.byte	0x7e, 0xff, 0xdb, 0xff, 0xc3, 0xe7, 0xff, 0x7e
		.byte	0x6c, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00
		.byte	0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x10, 0x00
		.byte	0x38, 0x7c, 0x38, 0xfe, 0xfe, 0xd6, 0x10, 0x38
		.byte	0x10, 0x38, 0x7c, 0xfe, 0xfe, 0x7c, 0x10, 0x38
		.byte	0x00, 0x00, 0x18, 0x3c, 0x3c, 0x18, 0x00, 0x00
		.byte	0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff
		.byte	0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00
		.byte	0xff, 0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff
		.byte	0x0f, 0x07, 0x0f, 0x7d, 0xcc, 0xcc, 0xcc, 0x78
		.byte	0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18
		.byte	0x3f, 0x33, 0x3f, 0x30, 0x30, 0x70, 0xf0, 0xe0
		.byte	0x7f, 0x63, 0x7f, 0x63, 0x63, 0x67, 0xe6, 0xc0
		.byte	0x18, 0xdb, 0x3c, 0xe7, 0xe7, 0x3c, 0xdb, 0x18
		.byte	0x80, 0xe0, 0xf8, 0xfe, 0xf8, 0xe0, 0x80, 0x00
		.byte	0x02, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x02, 0x00
		.byte	0x18, 0x3c, 0x7e, 0x18, 0x18, 0x7e, 0x3c, 0x18
		.byte	0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x00
		.byte	0x7f, 0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x00
		.byte	0x3e, 0x61, 0x3c, 0x66, 0x66, 0x3c, 0x86, 0x7c
		.byte	0x00, 0x00, 0x00, 0x00, 0x7e, 0x7e, 0x7e, 0x00
		.byte	0x18, 0x3c, 0x7e, 0x18, 0x7e, 0x3c, 0x18, 0xff
		.byte	0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x00
		.byte	0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00
		.byte	0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00
		.byte	0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00
		.byte	0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xfe, 0x00, 0x00
		.byte	0x00, 0x24, 0x66, 0xff, 0x66, 0x24, 0x00, 0x00
		.byte	0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x00, 0x00
		.byte	0x00, 0xff, 0xff, 0x7e, 0x3c, 0x18, 0x00, 0x00
		.byte	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
		.byte	0x18, 0x3c, 0x3c, 0x18, 0x18, 0x00, 0x18, 0x00
		.byte	0x66, 0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00
		.byte	0x6c, 0x6c, 0xfe, 0x6c, 0xfe, 0x6c, 0x6c, 0x00
		.byte	0x18, 0x3e, 0x60, 0x3c, 0x06, 0x7c, 0x18, 0x00
		.byte	0x00, 0xc6, 0xcc, 0x18, 0x30, 0x66, 0xc6, 0x00
		.byte	0x38, 0x6c, 0x38, 0x76, 0xdc, 0xcc, 0x76, 0x00
		.byte	0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00
		.byte	0x0c, 0x18, 0x30, 0x30, 0x30, 0x18, 0x0c, 0x00
		.byte	0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x18, 0x30, 0x00
		.byte	0x00, 0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00
		.byte	0x00, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x00
		.byte	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x30
		.byte	0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00
		.byte	0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x80, 0x00
		.byte	0x38, 0x6c, 0xc6, 0xd6, 0xc6, 0x6c, 0x38, 0x00
		.byte	0x18, 0x38, 0x18, 0x18, 0x18, 0x18, 0x7e, 0x00
		.byte	0x7c, 0xc6, 0x06, 0x1c, 0x30, 0x66, 0xfe, 0x00
		.byte	0x7c, 0xc6, 0x06, 0x3c, 0x06, 0xc6, 0x7c, 0x00
		.byte	0x1c, 0x3c, 0x6c, 0xcc, 0xfe, 0x0c, 0x1e, 0x00
		.byte	0xfe, 0xc0, 0xc0, 0xfc, 0x06, 0xc6, 0x7c, 0x00
		.byte	0x38, 0x60, 0xc0, 0xfc, 0xc6, 0xc6, 0x7c, 0x00
		.byte	0xfe, 0xc6, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x00
		.byte	0x7c, 0xc6, 0xc6, 0x7c, 0xc6, 0xc6, 0x7c, 0x00
		.byte	0x7c, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00
		.byte	0x00, 0x18, 0x18, 0x00, 0x00, 0x18, 0x18, 0x00
		.byte	0x00, 0x18, 0x18, 0x00, 0x00, 0x18, 0x18, 0x30
		.byte	0x06, 0x0c, 0x18, 0x30, 0x18, 0x0c, 0x06, 0x00
		.byte	0x00, 0x00, 0x7e, 0x00, 0x00, 0x7e, 0x00, 0x00
		.byte	0x60, 0x30, 0x18, 0x0c, 0x18, 0x30, 0x60, 0x00
		.byte	0x7c, 0xc6, 0x0c, 0x18, 0x18, 0x00, 0x18, 0x00
		.byte	0x7c, 0xc6, 0xde, 0xde, 0xde, 0xc0, 0x78, 0x00
		.byte	0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0x00
		.byte	0xfc, 0x66, 0x66, 0x7c, 0x66, 0x66, 0xfc, 0x00
		.byte	0x3c, 0x66, 0xc0, 0xc0, 0xc0, 0x66, 0x3c, 0x00
		.byte	0xf8, 0x6c, 0x66, 0x66, 0x66, 0x6c, 0xf8, 0x00
		.byte	0xfe, 0x62, 0x68, 0x78, 0x68, 0x62, 0xfe, 0x00
		.byte	0xfe, 0x62, 0x68, 0x78, 0x68, 0x60, 0xf0, 0x00
		.byte	0x3c, 0x66, 0xc0, 0xc0, 0xce, 0x66, 0x3a, 0x00
		.byte	0xc6, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0x00
		.byte	0x3c, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00
		.byte	0x1e, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78, 0x00
		.byte	0xe6, 0x66, 0x6c, 0x78, 0x6c, 0x66, 0xe6, 0x00
		.byte	0xf0, 0x60, 0x60, 0x60, 0x62, 0x66, 0xfe, 0x00
		.byte	0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6, 0xc6, 0x00
		.byte	0xc6, 0xe6, 0xf6, 0xde, 0xce, 0xc6, 0xc6, 0x00
		.byte	0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00
		.byte	0xfc, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00
		.byte	0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xce, 0x7c, 0x0e
		.byte	0xfc, 0x66, 0x66, 0x7c, 0x6c, 0x66, 0xe6, 0x00
		.byte	0x3c, 0x66, 0x30, 0x18, 0x0c, 0x66, 0x3c, 0x00
		.byte	0x7e, 0x7e, 0x5a, 0x18, 0x18, 0x18, 0x3c, 0x00
		.byte	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00
		.byte	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00
		.byte	0xc6, 0xc6, 0xc6, 0xd6, 0xd6, 0xfe, 0x6c, 0x00
		.byte	0xc6, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0xc6, 0x00
		.byte	0x66, 0x66, 0x66, 0x3c, 0x18, 0x18, 0x3c, 0x00
		.byte	0xfe, 0xc6, 0x8c, 0x18, 0x32, 0x66, 0xfe, 0x00
		.byte	0x3c, 0x30, 0x30, 0x30, 0x30, 0x30, 0x3c, 0x00
		.byte	0xc0, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x02, 0x00
		.byte	0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x3c, 0x00
		.byte	0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff
		.byte	0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x76, 0x00
		.byte	0xe0, 0x60, 0x7c, 0x66, 0x66, 0x66, 0xdc, 0x00
		.byte	0x00, 0x00, 0x7c, 0xc6, 0xc0, 0xc6, 0x7c, 0x00
		.byte	0x1c, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 0x00
		.byte	0x00, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0x7c, 0x00
		.byte	0x3c, 0x66, 0x60, 0xf8, 0x60, 0x60, 0xf0, 0x00
		.byte	0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8
		.byte	0xe0, 0x60, 0x6c, 0x76, 0x66, 0x66, 0xe6, 0x00
		.byte	0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x3c, 0x00
		.byte	0x06, 0x00, 0x06, 0x06, 0x06, 0x66, 0x66, 0x3c
		.byte	0xe0, 0x60, 0x66, 0x6c, 0x78, 0x6c, 0xe6, 0x00
		.byte	0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00
		.byte	0x00, 0x00, 0xec, 0xfe, 0xd6, 0xd6, 0xd6, 0x00
		.byte	0x00, 0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x00
		.byte	0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0x00
		.byte	0x00, 0x00, 0xdc, 0x66, 0x66, 0x7c, 0x60, 0xf0
		.byte	0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0x1e
		.byte	0x00, 0x00, 0xdc, 0x76, 0x60, 0x60, 0xf0, 0x00
		.byte	0x00, 0x00, 0x7e, 0xc0, 0x7c, 0x06, 0xfc, 0x00
		.byte	0x30, 0x30, 0xfc, 0x30, 0x30, 0x36, 0x1c, 0x00
		.byte	0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00
		.byte	0x00, 0x00, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00
		.byte	0x00, 0x00, 0xc6, 0xd6, 0xd6, 0xfe, 0x6c, 0x00
		.byte	0x00, 0x00, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0x00
		.byte	0x00, 0x00, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0xfc
		.byte	0x00, 0x00, 0x7e, 0x4c, 0x18, 0x32, 0x7e, 0x00
		.byte	0x0e, 0x18, 0x18, 0x70, 0x18, 0x18, 0x0e, 0x00
		.byte	0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00
		.byte	0x70, 0x18, 0x18, 0x0e, 0x18, 0x18, 0x70, 0x00
		.byte	0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
		.byte	0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0x00
		.byte	0x7c, 0xc6, 0xc0, 0xc0, 0xc6, 0x7c, 0x0c, 0x78
		.byte	0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00
		.byte	0x0c, 0x18, 0x7c, 0xc6, 0xfe, 0xc0, 0x7c, 0x00
		.byte	0x7c, 0x82, 0x78, 0x0c, 0x7c, 0xcc, 0x76, 0x00
		.byte	0xc6, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x76, 0x00
		.byte	0x30, 0x18, 0x78, 0x0c, 0x7c, 0xcc, 0x76, 0x00
		.byte	0x30, 0x30, 0x78, 0x0c, 0x7c, 0xcc, 0x76, 0x00
		.byte	0x00, 0x00, 0x7e, 0xc0, 0xc0, 0x7e, 0x0c, 0x38
		.byte	0x7c, 0x82, 0x7c, 0xc6, 0xfe, 0xc0, 0x7c, 0x00
		.byte	0xc6, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0x7c, 0x00
		.byte	0x30, 0x18, 0x7c, 0xc6, 0xfe, 0xc0, 0x7c, 0x00
		.byte	0x66, 0x00, 0x38, 0x18, 0x18, 0x18, 0x3c, 0x00
		.byte	0x7c, 0x82, 0x38, 0x18, 0x18, 0x18, 0x3c, 0x00
		.byte	0x30, 0x18, 0x00, 0x38, 0x18, 0x18, 0x3c, 0x00
		.byte	0xc6, 0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0xc6, 0x00
		.byte	0x38, 0x6c, 0x7c, 0xc6, 0xfe, 0xc6, 0xc6, 0x00
		.byte	0x18, 0x30, 0xfe, 0xc0, 0xf8, 0xc0, 0xfe, 0x00
		.byte	0x00, 0x00, 0x7e, 0x18, 0x7e, 0xd8, 0x7e, 0x00
		.byte	0x3e, 0x6c, 0xcc, 0xfe, 0xcc, 0xcc, 0xce, 0x00
		.byte	0x7c, 0x82, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0x00
		.byte	0xc6, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0x00
		.byte	0x30, 0x18, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0x00
		.byte	0x78, 0x84, 0x00, 0xcc, 0xcc, 0xcc, 0x76, 0x00
		.byte	0x60, 0x30, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00
		.byte	0xc6, 0x00, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0xfc
		.byte	0xc6, 0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x38, 0x00
		.byte	0xc6, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00
		.byte	0x18, 0x18, 0x7e, 0xc0, 0xc0, 0x7e, 0x18, 0x18
		.byte	0x38, 0x6c, 0x64, 0xf0, 0x60, 0x66, 0xfc, 0x00
		.byte	0x66, 0x66, 0x3c, 0x7e, 0x18, 0x7e, 0x18, 0x18
		.byte	0xf8, 0xcc, 0xcc, 0xfa, 0xc6, 0xcf, 0xc6, 0xc7
		.byte	0x0e, 0x1b, 0x18, 0x3c, 0x18, 0xd8, 0x70, 0x00
		.byte	0x18, 0x30, 0x78, 0x0c, 0x7c, 0xcc, 0x76, 0x00
		.byte	0x0c, 0x18, 0x00, 0x38, 0x18, 0x18, 0x3c, 0x00
		.byte	0x0c, 0x18, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0x00
		.byte	0x18, 0x30, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00
		.byte	0x76, 0xdc, 0x00, 0xdc, 0x66, 0x66, 0x66, 0x00
		.byte	0x76, 0xdc, 0x00, 0xe6, 0xf6, 0xde, 0xce, 0x00
		.byte	0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00, 0x00
		.byte	0x38, 0x6c, 0x6c, 0x38, 0x00, 0x7c, 0x00, 0x00
		.byte	0x18, 0x00, 0x18, 0x18, 0x30, 0x63, 0x3e, 0x00
		.byte	0x00, 0x00, 0x00, 0xfe, 0xc0, 0xc0, 0x00, 0x00
		.byte	0x00, 0x00, 0x00, 0xfe, 0x06, 0x06, 0x00, 0x00
		.byte	0x63, 0xe6, 0x6c, 0x7e, 0x33, 0x66, 0xcc, 0x0f
		.byte	0x63, 0xe6, 0x6c, 0x7a, 0x36, 0x6a, 0xdf, 0x06
		.byte	0x18, 0x00, 0x18, 0x18, 0x3c, 0x3c, 0x18, 0x00
		.byte	0x00, 0x33, 0x66, 0xcc, 0x66, 0x33, 0x00, 0x00
		.byte	0x00, 0xcc, 0x66, 0x33, 0x66, 0xcc, 0x00, 0x00
		.byte	0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88
		.byte	0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa
		.byte	0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd
		.byte	0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18
		.byte	0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18
		.byte	0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18
		.byte	0x36, 0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36
		.byte	0x00, 0x00, 0x00, 0x00, 0xfe, 0x36, 0x36, 0x36
		.byte	0x00, 0x00, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18
		.byte	0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36, 0x36
		.byte	0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36
		.byte	0x00, 0x00, 0xfe, 0x06, 0xf6, 0x36, 0x36, 0x36
		.byte	0x36, 0x36, 0xf6, 0x06, 0xfe, 0x00, 0x00, 0x00
		.byte	0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00, 0x00
		.byte	0x18, 0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18
		.byte	0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00
		.byte	0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18
		.byte	0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18
		.byte	0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00
		.byte	0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18
		.byte	0x18, 0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18
		.byte	0x36, 0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36
		.byte	0x36, 0x36, 0x37, 0x30, 0x3f, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36, 0x36
		.byte	0x36, 0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0xff, 0x00, 0xf7, 0x36, 0x36, 0x36
		.byte	0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36
		.byte	0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00
		.byte	0x36, 0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36
		.byte	0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00
		.byte	0x36, 0x36, 0x36, 0x36, 0xff, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18
		.byte	0x00, 0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36
		.byte	0x36, 0x36, 0x36, 0x36, 0x3f, 0x00, 0x00, 0x00
		.byte	0x18, 0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18
		.byte	0x00, 0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36
		.byte	0x36, 0x36, 0x36, 0x36, 0xff, 0x36, 0x36, 0x36
		.byte	0x18, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18
		.byte	0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18
		.byte	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
		.byte	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff
		.byte	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0
		.byte	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f
		.byte	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x76, 0xdc, 0xc8, 0xdc, 0x76, 0x00
		.byte	0x78, 0xcc, 0xcc, 0xd8, 0xcc, 0xc6, 0xcc, 0x00
		.byte	0xfe, 0xc6, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0x00
		.byte	0x00, 0x00, 0xfe, 0x6c, 0x6c, 0x6c, 0x6c, 0x00
		.byte	0xfe, 0xc6, 0x60, 0x30, 0x60, 0xc6, 0xfe, 0x00
		.byte	0x00, 0x00, 0x7e, 0xd8, 0xd8, 0xd8, 0x70, 0x00
		.byte	0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x7c, 0xc0
		.byte	0x00, 0x76, 0xdc, 0x18, 0x18, 0x18, 0x18, 0x00
		.byte	0x7e, 0x18, 0x3c, 0x66, 0x66, 0x3c, 0x18, 0x7e
		.byte	0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0x6c, 0x38, 0x00
		.byte	0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x6c, 0xee, 0x00
		.byte	0x0e, 0x18, 0x0c, 0x3e, 0x66, 0x66, 0x3c, 0x00
		.byte	0x00, 0x00, 0x7e, 0xdb, 0xdb, 0x7e, 0x00, 0x00
		.byte	0x06, 0x0c, 0x7e, 0xdb, 0xdb, 0x7e, 0x60, 0xc0
		.byte	0x1e, 0x30, 0x60, 0x7e, 0x60, 0x30, 0x1e, 0x00
		.byte	0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00
		.byte	0x00, 0xfe, 0x00, 0xfe, 0x00, 0xfe, 0x00, 0x00
		.byte	0x18, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x7e, 0x00
		.byte	0x30, 0x18, 0x0c, 0x18, 0x30, 0x00, 0x7e, 0x00
		.byte	0x0c, 0x18, 0x30, 0x18, 0x0c, 0x00, 0x7e, 0x00
		.byte	0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18, 0x18, 0x18
		.byte	0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8, 0x70
		.byte	0x00, 0x18, 0x00, 0x7e, 0x00, 0x18, 0x00, 0x00
		.byte	0x00, 0x76, 0xdc, 0x00, 0x76, 0xdc, 0x00, 0x00
		.byte	0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00
		.byte	0x0f, 0x0c, 0x0c, 0x0c, 0xec, 0x6c, 0x3c, 0x1c
		.byte	0x6c, 0x36, 0x36, 0x36, 0x36, 0x00, 0x00, 0x00
		.byte	0x78, 0x0c, 0x18, 0x30, 0x7c, 0x00, 0x00, 0x00
		.byte	0x00, 0x00, 0x3c, 0x3c, 0x3c, 0x3c, 0x00, 0x00
		.byte	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00

#endif /* CONFIG_LAB_DISPLAY_ADDR */

saved_r9:	.long	0

@ Thunks callable from C (stack is already set up)
.globl c_out_str
c_out_str:	stmdb	sp!, {r0,r1,r2,r3,r4,r5,r9,r12,lr}
		mov	r12, #BF_NEED_RESTORE_R9
		bl	out_str
		ldmia	sp!, {r0,r1,r2,r3,r4,r5,r9,r12,pc}

.globl c_out_hex2
c_out_hex2:	stmdb	sp!, {r0,r1,r2,r3,r4,r5,r9,r12,lr}
		mov	r12, #BF_NEED_RESTORE_R9
		bl	out_hex_byte
		ldmia	sp!, {r0,r1,r2,r3,r4,r5,r9,r12,pc}

.globl c_out_hex8
c_out_hex8:	stmdb	sp!, {r0-r7,r9,r12,lr}
		mov	r12, #BF_NEED_RESTORE_R9
		ldr	r9, saved_r9
		bl	out_hex_word
		ldmia	sp!, {r0-r7,r9,r12,pc}
