/*
 * Support for power detection code on the HTC Apache phone.
 *
 * (c) Copyright 2006 Kevin O'Connor <kevin@koconnor.net>
 *
 * This file may be distributed under the terms of the GNU GPL license.
 */

#include <linux/kernel.h>
#include <linux/interrupt.h> // request_irq
#include <linux/battery.h>
#include <linux/platform_device.h>
#include <linux/workqueue.h>
#include <linux/adc.h>
#include <linux/ad7877.h>

#include <asm/arch/htcapache-gpio.h>
#include <asm/apm.h>

// Last detected values.
static u32 BattAC;
static u32 BattVoltage, BattCurrent, BattTemp;


/****************************************************************
 * Periodic battery sensing timer
 ****************************************************************/

struct senseInfo {
	struct adc_request req;
	struct adc_sense pins[5];

	struct delayed_work work;
	struct workqueue_struct *wq;
};

// Work callback - rearms itself once every 10 seconds.
static void
handle_adc(struct work_struct *work)
{
	struct delayed_work *dw = container_of(work, struct delayed_work,
	                                       work);
	struct senseInfo *si = container_of(dw, struct senseInfo, work);

	adc_request_sample(&si->req);

	// These multipliers are from observations of wince behavior.
	BattVoltage = si->pins[0].value * 1106 / 1024;
	BattCurrent = si->pins[1].value * 1157 / 1024;
	BattTemp = si->pins[2].value;

	queue_delayed_work(si->wq, &si->work, msecs_to_jiffies(1000));

#if 0
	printk(KERN_INFO "1=%d 2=%d 3=%d v=%d c=%d t=%d\n"
	       , si->pins[0].value, si->pins[1].value, si->pins[2].value
	       , BattVoltage, BattCurrent, BattTemp);
#endif
}


/****************************************************************
 * Plug in/out events
 ****************************************************************/

// Interrupt handler called when plugging/unplugging mini-usb port
static irqreturn_t
power_isr(int irq, void *dev_id)
{
	int haspower = !htcapache_egpio_isset(EGPIO_NR_HTCAPACHE_PWR_IN_PWR);
	int hashigh = !htcapache_egpio_isset(EGPIO_NR_HTCAPACHE_PWR_IN_HIGHPWR);
	if (haspower && !hashigh) {
		// USB plug - activate usb transceiver.
		htcapache_egpio_set(EGPIO_NR_HTCAPACHE_USB_PWR);
	} else {
		htcapache_egpio_clear(EGPIO_NR_HTCAPACHE_USB_PWR);
	}
	BattAC = (hashigh << 1) | haspower;
	printk("USB mini plug event (pwr=%d ac=%d)\n"
	       , haspower, hashigh);
	return IRQ_HANDLED;
}


/****************************************************************
 * APM support
 ****************************************************************/

#ifdef CONFIG_PM
static void
get_apm_status(struct apm_power_info *info)
{
	s32 batt_pct;

	// XXX - charging not yet supported.
	if (BattAC)
		info->ac_line_status = APM_AC_ONLINE;
	else
		info->ac_line_status = APM_AC_OFFLINE;

	// XXX - Based on WinCE battery percentage - just a guess.
	batt_pct = (BattVoltage - 3600) * 310 / 1024;
	if (batt_pct < 0)
		batt_pct = 0;
	else if (batt_pct > 100)
		batt_pct = 100;
	info->battery_life = batt_pct;

	if (batt_pct < 5)
		info->battery_status = APM_BATTERY_STATUS_CRITICAL;
	else if (batt_pct < 20)
		info->battery_status = APM_BATTERY_STATUS_LOW;
	else
		info->battery_status = APM_BATTERY_STATUS_HIGH;

#if 0
	printk(KERN_INFO "ac=%d v=%d  s=%d l=%d bs=%d\n"
	       , BattAC, BattVoltage
	       , info->ac_line_status, info->battery_life
	       , info->battery_status);
#endif
}
#endif


/****************************************************************
 * Battery class
 ****************************************************************/

int get_min_voltage(struct battery *b)
{
    return 0;
}
int get_min_current(struct battery *b)
{
    return 0;
}
int get_max_voltage(struct battery *b)
{
    return 4750; /* mV */
}
int get_max_current(struct battery *b)
{
    return 1900; /* positive 1900 mA */
}
int get_voltage(struct battery *b)
{
	return BattVoltage;
}
int get_Current(struct battery *b)
{
	return BattCurrent;
}
int get_status(struct battery *b)
{
	return !!BattAC;
}

static struct battery apache_power = {
    .name               = "htcapache_primary",
    .get_min_voltage    = get_min_voltage,
    .get_min_current    = get_min_current,
    .get_max_voltage    = get_max_voltage,
    .get_max_current    = get_max_current,
//    .get_temp           = get_temp,
    .get_voltage        = get_voltage,
    .get_current        = get_Current,
    .get_status         = get_status,
};


/****************************************************************
 * Setup
 ****************************************************************/

static struct senseInfo Sensors = {
	.pins = {
		[0] = { .name = "ad7877:aux1", },
		[1] = { .name = "ad7877:aux2", },
		[2] = { .name = "ad7877:bat1", },
		[3] = { .name = "ad7877:aux3", },
		[4] = { .name = "ad7877:bat2", },
	},
};

static int
power_probe(struct platform_device *pdev)
{
	struct senseInfo *si = &Sensors;
	int result;

	// Setup plug interrupt
	result = request_irq(IRQ_EGPIO(EGPIO_NR_HTCAPACHE_PWR_IN_PWR)
			     , power_isr, SA_SAMPLE_RANDOM,
			     "usb_plug", NULL);
	if (result)
		printk("unable to claim irq %d for usb plug event. Got %d\n"
		       , IRQ_EGPIO(EGPIO_NR_HTCAPACHE_PWR_IN_PWR), result);

	// Run the interrupt handler to initialize.
	power_isr(0, NULL);

	// Setup battery sensing.
	si->req.senses = si->pins;
	si->req.num_senses = ARRAY_SIZE(si->pins);
	adc_request_register(&si->req);

	battery_register(&pdev->dev, &apache_power);

	INIT_DELAYED_WORK(&si->work, handle_adc);
	si->wq = create_workqueue(pdev->dev.bus_id);
	if (!si->wq)
		return -ESRCH;

	queue_delayed_work(si->wq, &si->work, msecs_to_jiffies(1000));

#ifdef CONFIG_PM
	apm_get_power_status = get_apm_status;
#endif
	return 0;
}

static int
power_remove(struct platform_device *pdev)
{
	struct senseInfo *si = &Sensors;
	free_irq(IRQ_EGPIO(EGPIO_NR_HTCAPACHE_PWR_IN_PWR), NULL);
	cancel_rearming_delayed_work(&si->work);
	destroy_workqueue(si->wq);
	adc_request_unregister(&si->req);
	return 0;
}

static struct platform_driver power_driver = {
	.driver = {
		.name           = "htcapache-power",
	},
	.probe          = power_probe,
	.remove         = power_remove,
};

static int __init power_init(void)
{
	return platform_driver_register(&power_driver);
}

static void __exit power_exit(void)
{
	platform_driver_unregister(&power_driver);
}

module_init(power_init)
module_exit(power_exit)

MODULE_LICENSE("GPL");
