/** @file wlan_tx.c
  * @brief This file contains the handling of TX in wlan driver.
  * 
  * (c) Copyright  2003-2006, Marvell International Ltd. 
  * All Rights Reserved
  *
  * This software file (the "File") is distributed by Marvell International 
  * Ltd. under the terms of the GNU General Public License Version 2, June 1991 
  * (the "License").  You may use, redistribute and/or modify this File in 
  * accordance with the terms and conditions of the License, a copy of which 
  * is available along with the File in the license.txt file or by writing to 
  * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
  * 02111-1307 or on the worldwide web at http://www.gnu.org/licenses/gpl.txt.
  *
  * THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE 
  * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE 
  * ARE EXPRESSLY DISCLAIMED.  The License provides additional details about 
  * this warranty disclaimer.
  *
  */
/********************************************************
Change log:
	09/28/05: Add Doxygen format comments
	12/13/05: Add Proprietary periodic sleep support
	01/05/06: Add kernel 2.6.x support	
	04/06/06: Add TSPEC, queue metrics, and MSDU expiry support
********************************************************/

#include	"include.h"

/********************************************************
		Local Variables
********************************************************/

typedef struct
{
    struct ieee80211_radiotap_header hdr;
    u8           rate;
    u8           txpower;
    u8           rts_retries;
    u8           data_retries;
    //u8           pad[IEEE80211_RADIOTAP_HDRLEN - 12];
} __attribute__ ((packed)) TxRadiotapHdr_t;

#define TX_RADIOTAP_PRESENT (   \
		(1 << IEEE80211_RADIOTAP_RATE)    | \
		(1 << IEEE80211_RADIOTAP_DBM_TX_POWER)  | \
		(1 << IEEE80211_RADIOTAP_RTS_RETRIES) | \
		(1 << IEEE80211_RADIOTAP_DATA_RETRIES)  | \
		0)

typedef struct
{
    u8  dest_addr[6];
    u8  src_addr[6];
    u16 ethertype;

} __attribute__ ((packed)) EthII_Hdr_t;

#define IEEE80211_FC_VERSION_MASK    0x0003
#define IEEE80211_FC_TYPE_MASK       0x000c
#define IEEE80211_FC_TYPE_MGT        0x0000
#define IEEE80211_FC_TYPE_CTL        0x0004
#define IEEE80211_FC_TYPE_DATA       0x0008
#define IEEE80211_FC_SUBTYPE_MASK    0x00f0
#define IEEE80211_FC_TOFROMDS_MASK   0x0300
#define IEEE80211_FC_TODS_MASK       0x0100
#define IEEE80211_FC_FROMDS_MASK     0x0200
#define IEEE80211_FC_NODS            0x0000
#define IEEE80211_FC_TODS            0x0100
#define IEEE80211_FC_FROMDS          0x0200
#define IEEE80211_FC_DSTODS          0x0300
typedef struct
{
    u16  FrameControl;
    u16  DurationID;
    u8  address1[6];
    u8  address2[6];
    u8  address3[6];
    u16  SeqControl;
    u8  address4[6];

} __attribute__ ((packed)) IEEE80211_Hdr_t;


/********************************************************
		Global Variables
********************************************************/

/********************************************************
		Local Functions
********************************************************/

/**
 *  @brief This function converts Tx/Rx rates from IEEE80211_RADIOTAP_RATE 
 *  units (500 Kb/s) into Marvell WLAN format (see Table 8 in Section 3.2.1)
 *
 *  @param rate    Input rate
 *  @return      Output Rate (0 if invalid)
 */
u32 convert_radiotap_rate_to_mv(u8 rate)
{
	switch (rate) {
		case  2: /*   1 Mbps */ return 0 | (1<<4);
		case  4: /*   2 Mbps */ return 1 | (1<<4);
		case 11: /* 5.5 Mbps */ return 2 | (1<<4);
		case 22: /*  11 Mbps */ return 3 | (1<<4);
		case 12: /*   6 Mbps */ return 4 | (1<<4);
		case 18: /*   9 Mbps */ return 5 | (1<<4);
		case 24: /*  12 Mbps */ return 6 | (1<<4);
		case 36: /*  18 Mbps */ return 7 | (1<<4);
		case 48: /*  24 Mbps */ return 8 | (1<<4);
		case 72: /*  36 Mbps */ return 9 | (1<<4);
		case 96: /*  48 Mbps */ return 10 | (1<<4);
		case 108: /*  54 Mbps */ return 11 | (1<<4);
	}
	return 0;
}

/** 
 *  @brief This function processes a single packet and sends
 *  to IF layer
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param skb     A pointer to skb which includes TX packet
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int SendSinglePacket(wlan_private *priv, struct sk_buff *skb)
{
	wlan_adapter   *Adapter = priv->adapter;
	int             ret = WLAN_STATUS_SUCCESS;
	struct TxPD	LocalTxPD;
	struct TxPD	*pLocalTxPD = &LocalTxPD;
	u8		*p802x_hdr;
	TxRadiotapHdr_t *pradiotap_hdr;
	u32 new_rate;
	u8		*ptr = priv->adapter->TmpTxBuf;

	ENTER();

	if ( (priv->adapter->debugmode & MRVDRV_DEBUG_TX_PATH) != 0 )
		HEXDUMP("TX packet: ", skb->data, MIN(skb->len, 100));

		if (!skb->len || (skb->len > 
					MRVDRV_ETH_TX_PACKET_BUFFER_SIZE)) {
			PRINTM(INFO, "Tx Error: Bad skb length %d : %d\n",
				skb->len, MRVDRV_ETH_TX_PACKET_BUFFER_SIZE);
			ret = WLAN_STATUS_FAILURE;
			goto done;
		}

		memset(pLocalTxPD, 0, sizeof(struct TxPD));

		pLocalTxPD->TxPacketLength = skb->len;

		if(Adapter->PSState != PS_STATE_FULL_POWER){
			if(TRUE == CheckLastPacketIndication(priv)){
				Adapter->TxLockFlag = 1;
				pLocalTxPD->PowerMgmt = 
					MRVDRV_TxPD_POWER_MGMT_LAST_PACKET;
			}
		}
		/* offset of actual data */
		pLocalTxPD->TxPacketLocation = sizeof(struct TxPD);

		/* TxCtrl set by user or default */
		pLocalTxPD->TxControl = Adapter->PktTxCtrl;

		p802x_hdr = skb->data;
		if ( priv->adapter->radiomode == WLAN_RADIOMODE_RADIOTAP ) {

			/* locate radiotap header */
			pradiotap_hdr = (TxRadiotapHdr_t*)skb->data;
		
			/* set TxPD fields from the radiotap header */
			new_rate = convert_radiotap_rate_to_mv(
							pradiotap_hdr->rate);
			if ( new_rate != 0 ) {
				/* erase TxControl[4:0] */
				pLocalTxPD->TxControl &= ~0x1f;
				/* write new TxControl[4:0] */
				pLocalTxPD->TxControl |= new_rate;
			}

			/* skip the radiotap header */
			p802x_hdr += sizeof(TxRadiotapHdr_t);
			pLocalTxPD->TxPacketLength -= sizeof(TxRadiotapHdr_t);	

		}
		/* copy destination address from 802.3 or 802.11 header */
		if ( priv->adapter->linkmode == WLAN_LINKMODE_802_11 )  
			memcpy(pLocalTxPD->TxDestAddrHigh, p802x_hdr + 4, 
			       MRVDRV_ETH_ADDR_LEN);
		else
			memcpy(pLocalTxPD->TxDestAddrHigh, p802x_hdr, 
			       MRVDRV_ETH_ADDR_LEN);



		HEXDUMP("TxPD", (u8 *) pLocalTxPD, sizeof(struct TxPD));

		memcpy(ptr, pLocalTxPD, sizeof(struct TxPD));
	
		ptr += sizeof(struct TxPD);
		
		HEXDUMP("Tx Data", (u8 *) p802x_hdr, 
				pLocalTxPD->TxPacketLength);
		memcpy(ptr, p802x_hdr, pLocalTxPD->TxPacketLength);
		ret = sbi_host_to_card(priv, MVMS_DAT,
		       		priv->adapter->TmpTxBuf,  
				pLocalTxPD->TxPacketLength + sizeof(struct TxPD));


	if (ret) {
		PRINTM(INFO, "Tx Error: sbi_host_to_card failed: 0x%X\n",ret);
		goto done;
	}

	PRINTM(INFO, "SendSinglePacket succeeds\n");

done:
	if (!ret) {
		priv->stats.tx_packets++;
			priv->stats.tx_bytes += skb->len;
	}
	else {
		priv->stats.tx_dropped++;
		priv->stats.tx_errors++;
	}

	if (!ret && priv->adapter->radiomode == WLAN_RADIOMODE_RADIOTAP)  {
		/* Keep the skb to echo it back once Tx feedback is 
		   received from FW */
		skb_orphan(priv->adapter->CurrentTxSkb);
		/* stop processing outgoing pkts */
		os_stop_queue(priv);
		/* freeze any packets already in our queues */
		priv->adapter->TxLockFlag = 1;
	} else 
	{
		os_free_tx_packet(priv);
		os_start_queue(priv);
	}

	LEAVE();
	return ret;
}

/********************************************************
		Global functions
********************************************************/

/** 
 *  @brief This function checks the conditions and sends packet to IF
 *  layer if everything is ok.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
void wlan_process_tx(wlan_private *priv)
{
	unsigned long flags;
	wlan_adapter   *Adapter = priv->adapter;

	ENTER();
    
	if(priv->wlan_dev.dnld_sent) {
		PRINTM(MSG, "TX Error: dnld_sent = %d, not sending\n",
						priv->wlan_dev.dnld_sent);
		goto done;
	}


	SendSinglePacket(priv, Adapter->CurrentTxSkb);

	spin_lock_irqsave(&driver_lock, flags);
	priv->adapter->HisRegCpy &= ~HIS_TxDnLdRdy;
	spin_unlock_irqrestore(&driver_lock, flags);

done:
	LEAVE();
}

/** 
 *  @brief This function queues the packet received from
 *  kernel/upper layer and wake up the main thread to handle it.
 *  
 *  @param priv    A pointer to wlan_private structure
  * @param skb     A pointer to skb which includes TX packet
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_tx_packet(wlan_private *priv, struct sk_buff *skb)
{
	ulong		flags;
	wlan_adapter 	*Adapter = priv->adapter;
	int		ret = WLAN_STATUS_SUCCESS;

	ENTER();

 	HEXDUMP("TX Data", skb->data, MIN(skb->len, 100));

	spin_lock_irqsave(&Adapter->CurrentTxLock, flags);

	{
		Adapter->TxSkbNum ++;
		list_add_tail((struct list_head *) skb,(struct list_head *) &priv->adapter->TxSkbQ);
		wake_up_interruptible(&priv->MainThread.waitQ);
	}
	spin_unlock_irqrestore(&Adapter->CurrentTxLock, flags);

	LEAVE();

	return ret;
}


/** 
 *  @brief This function tells firmware to send a NULL data packet.
 *  
 *  @param priv     A pointer to wlan_private structure
 *  @param pwr_mgmt indicate if power management bit should be 0 or 1
 *  @return 	    n/a
 */
int SendNullPacket(wlan_private *priv, u8 pwr_mgmt)
{
	wlan_adapter 	*Adapter = priv->adapter;
  	struct TxPD txpd;
  	int ret = WLAN_STATUS_SUCCESS;
	u8		*ptr = priv->adapter->TmpTxBuf;

	ENTER();

	if (priv->adapter->SurpriseRemoved == TRUE) {
    		ret = WLAN_STATUS_FAILURE;
		goto done;
   	}

	if (priv->adapter->MediaConnectStatus == WlanMediaStateDisconnected) {
    		ret = WLAN_STATUS_FAILURE;
		goto done;
   	}

  	memset(&txpd, 0, sizeof(struct TxPD));

  	txpd.TxControl   = Adapter->PktTxCtrl;
 	txpd.PowerMgmt   = pwr_mgmt;
 	txpd.TxPacketLocation = sizeof(struct TxPD);


   	memcpy(ptr, &txpd, sizeof(struct TxPD));

   	ret = sbi_host_to_card(priv, MVMS_DAT,
		       		 priv->adapter->TmpTxBuf, sizeof(struct TxPD));

   	if (ret != 0)
  	{
   		PRINTM(INFO, "TX Error: SendNullPacket failed!\n");
		goto done;
   	}

done:
  	LEAVE();
	return ret;
}

/** 
 *  @brief This function check if we need send last packet indication.
 *  
 *  @param priv     A pointer to wlan_private structure
 *
 *  @return 	   TRUE or FALSE
 */
u8 CheckLastPacketIndication(wlan_private *priv)
{
	wlan_adapter 	*Adapter = priv->adapter;
	u8 ret = FALSE; 
	if(Adapter->sleep_period.period == 0)
		goto done;
done:	
	return ret;	
}

/** 
 *  @brief This function sends to the host the last transmitted packet,
 *  filling the radiotap headers with transmission information.
 *  
 *  @param priv     A pointer to wlan_private structure
 *  @param status   A 32 bit value containing transmission status.
 *
 *  @returns void
 */
void SendTxFeedback(wlan_private *priv)
{
	wlan_adapter 	*Adapter = priv->adapter;
	TxRadiotapHdr_t *radiotap_hdr; 	
	u32 		status = Adapter->EventCause;
	int txfail;
	int try_count;


	if (Adapter->radiomode != WLAN_RADIOMODE_RADIOTAP || 
	    Adapter->CurrentTxSkb == NULL)
		return;

	radiotap_hdr = (TxRadiotapHdr_t*) Adapter->CurrentTxSkb->data;

	if ((Adapter->debugmode & MRVDRV_DEBUG_TX_PATH) != 0 )
		HEXDUMP("TX feedback: ", (u8 *) radiotap_hdr, 
			 MIN(Adapter->CurrentTxSkb->len, 100));

	txfail = (status >> 24);
	
#if 0
	/* The version of roofnet that we've tested does not use this yet 
	 * But it may be used in the future.
	 */
	if (txfail)
		radiotap_hdr->flags &= IEEE80211_RADIOTAP_F_TX_FAIL;
#endif
	try_count = (status >> 16) & 0xff;
	radiotap_hdr->data_retries = (try_count) ? 
		(1 + Adapter->TxRetryCount - try_count) : 0;
	os_upload_rx_packet(priv, Adapter->CurrentTxSkb);
	Adapter->CurrentTxSkb = NULL;
	priv->adapter->TxLockFlag = 0;
	os_start_queue(priv);
}
