/*****************************************************************************
 *
 * Microchip DeviceNet Stack (CAN Driver Source)
 *
 *****************************************************************************
 * FileName:        CAN.c
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.20.00 or higher
 * Linker:          MPLINK 03.40.00 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *
 *	This version of the CAN driver for DeviceNet is designed to be very
 *	simple. Filtering options are static and limited. Receive buffering is
 *	is restricted to only the two hardware receive buffers. And transmit 
 *	buffering is restricted to only one of three transmit buffers. However,
 *	the architecture of this driver lends itself to be upgradable to achieve 
 *	better performance without changes to the firmware using this driver.
 * 
 *
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			04/28/03	...	
 * 
 *****************************************************************************/



#include	"P18F458.H"			// Hardware specific header
#include	"dnet.def"			// Global definitions file
#include	"CAN.h"				 


/*********************************************************************
 * Private definitions derived from Maestro Generated Definitions
 ********************************************************************/
 
#define	_B125k_BRGCON1	(B125k_BRG1_SJW << 6) | B125k_BRG1_PRESCALE
#define	_B125k_BRGCON2	(B125k_BRG2_SEG2PHTS << 7) | (B125k_BRG2_SAM << 6) | (B125k_BRG2_SEG1PH << 3) | B125k_BRG2_PRSEG
#define	_B125k_BRGCON3	(B125k_BRG3_WAKFIL << 6) | B125k_BRG3_SEG2PH 

#define	_B250k_BRGCON1	(B250k_BRG1_SJW << 6) | B250k_BRG1_PRESCALE
#define	_B250k_BRGCON2	(B250k_BRG2_SEG2PHTS << 7) | (B250k_BRG2_SAM << 6) | (B250k_BRG2_SEG1PH << 3) | B250k_BRG2_PRSEG
#define	_B250k_BRGCON3	(B250k_BRG3_WAKFIL << 6) | B250k_BRG3_SEG2PH 

#define	_B500k_BRGCON1	(B500k_BRG1_SJW << 6) | B500k_BRG1_PRESCALE
#define	_B500k_BRGCON2	(B500k_BRG2_SEG2PHTS << 7) | (B500k_BRG2_SAM << 6) | (B500k_BRG2_SEG1PH << 3) | B500k_BRG2_PRSEG
#define	_B500k_BRGCON3	(B500k_BRG3_WAKFIL << 6) | B500k_BRG3_SEG2PH 




/*********************************************************************
 * Private types
 ********************************************************************/
typedef union _T_INT
{
	unsigned int word;
	struct _BYTES_
	{
		unsigned char LSB;
		unsigned char MSB;
	}bytes;

}_INT;

// This type is used to copy all first fragment messages
typedef struct T_CAN_DATA_TYP0		 
{
	unsigned char	D0;
	unsigned char	D1;
	unsigned char	D2;
	unsigned char	D3;
	unsigned char	D4;
	unsigned char	D5;
	unsigned char	D6;
	unsigned char	D7;
}_CAN_DATA_TYP0;

// This type is used for fragmented I/O messages
typedef struct T_CAN_DATA_TYP1
{
	unsigned char	D1;
	unsigned char	D2;
	unsigned char	D3;
	unsigned char	D4;
	unsigned char	D5;
	unsigned char	D6;
	unsigned char	D7;
}_CAN_DATA_TYP1;

// This type is specifically used for fragmented explicit messages
typedef struct T_CAN_DATA_TYP2	
{
	unsigned char	D2;
	unsigned char	D3;
	unsigned char	D4;
	unsigned char	D5;
	unsigned char	D6;
	unsigned char	D7;
}_CAN_DATA_TYP2;


/*********************************************************************
 * Registers for CAN buffer control
 ********************************************************************/	
#if USE_ACCESS == 1
#pragma	udata access	_A_CAN_FUNCTIONS
#define	NEAR near
#else
#define NEAR
#endif 

NEAR unsigned char _msgTxTag;		// Tag used to remember the last instance
NEAR unsigned char _msgRxBuf;		// Reference to the current active rx buffer
//NEAR unsigned char _mac_filter;



 

/*********************************************************************
 * Function:        NEAR unsigned char CANOpen(void)
 *
 * PreCondition:   	Bitrate and filters should be setup 
 *
 * Input:       	void
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	 
 *
 * Overview:        Open CAN communications 
 *
 * Note:            This function should be interprited as a request
 *					rather than a command. Use CANIsOpen() to 
 *					verify the bus has been opened.
 ********************************************************************/
NEAR unsigned char CANOpen(void)
{
	// Place the CAN peripherial into Normal mode
	CANCON = 0;
	return (1);
}

/*********************************************************************
 * Function:        NEAR unsigned char CANClose(void)
 *
 * PreCondition:   	Bus communications should be open. 
 *
 * Input:       	void
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	 
 *
 * Overview:        Close CAN communications 
 *
 * Note:            This function should be interprited as a request
 *					rather than a command. Use CANIsOpen() to 
 *					verify the bus is still open.
 ********************************************************************/
NEAR unsigned char CANClose(void)
{
	// Place the CAN peripherial into Config mode
	CANCON = 0x40;
	CANCON = 0x80;
	return (1);
}

/*********************************************************************
 * Function:        NEAR unsigned char CANIsOpen(void)
 *
 * PreCondition:   	 
 *
 * Input:       	void
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	 
 *
 * Overview:        Return the status of communication. 
 *					TRUE = Open, FALSE = Closed 
 *
 * Note:            
 ********************************************************************/
NEAR unsigned char CANIsOpen(void)
{
	return (!(CANSTAT & 0xE0));
}

/*********************************************************************
 * Function:        NEAR unsigned char CANSetFilter(NEAR unsigned int filterID)
 *
 * PreCondition:   	 
 *
 * Input:       	16-bit filterID
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	Communication may be temporarily interrupted
 *
 * Overview:        Adds a filter to the scan list.
 *
 * Note:            No scan list is actually created in this version;
 *					a hardware filter is to allow a CAN ID or group 
 *					of IDs to be received.
 ********************************************************************/
NEAR unsigned char CANSetFilter(NEAR unsigned int filterID)
{	
	NEAR _INT filters;
	NEAR unsigned char oldState;

	// Remember the previous state
	oldState = CANCON;
	
	// Place the CAN peripherial into Config mode
	//CANCON = 0x40;
	CANCON = 0x80;
	
	// Stall
	while ((CANSTAT & 0xE0) != 0x80);



	filters.word = filterID;
	
	// Setup masks and filters according to the message
	switch (filters.bytes.LSB & 0xE0)
	{
		case 0x00:		
			RXF0SIDL = (NEAR unsigned char)(filters.bytes.LSB);
			RXF0SIDH = (NEAR unsigned char)(filters.bytes.MSB);
			RXM0SIDL = 0xFF;
			RXM0SIDH = 0xFF;
			break;

		case 0x20:
			RXF1SIDL = (NEAR unsigned char)(filters.bytes.LSB);
			RXF1SIDH = (NEAR unsigned char)(filters.bytes.MSB);
			RXM0SIDL = 0xFF;
			RXM0SIDH = 0xFF;
			break;

		case 0x40:
		case 0x60:
		case 0x80:
		case 0xA0:
		case 0xC0:
		case 0xE0:
			RXF2SIDL = (NEAR unsigned char)(filters.bytes.LSB);
			RXF2SIDH = (NEAR unsigned char)(filters.bytes.MSB);
			RXM1SIDL = 0x00;
			RXM1SIDH = 0xFF;
			break;
	}
	 
	// Place the bus back into run mode
	CANCON = oldState;	 
	 
	return (1);
}

/*********************************************************************
 * Function:        NEAR unsigned char CANClrFilter(NEAR unsigned int filterID)
 *
 * PreCondition:   	 
 *
 * Input:       	16-bit filterID
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	Communication may be temporarily interrupted
 *
 * Overview:        Removes a filter from the scan list. 
 *
 * Note:           	No scan list is actually created in this version;
 *					a hardware filter is used to allow a CAN ID or 
 *					group of IDs to	be received.
 ********************************************************************/
NEAR unsigned char CANClrFilter(NEAR unsigned int filterID)
{
	NEAR unsigned char oldState;

	// Remember the previous state
	oldState = CANCON;
	
	// Place the CAN peripherial into Config mode
	//CANCON = 0x40;
	CANCON = 0x80;
	
	// Stall
	while ((CANSTAT & 0xE0) != 0x80);

	
	// Clear filters
	switch ((NEAR unsigned char)(filterID & 0xE0))
	{
		case 0x00:		
			RXF0SIDLbits.EXIDEN = 1;
			break;

		case 0x20:
			RXF1SIDLbits.EXIDEN = 1;
			break;

		case 0x40:
		case 0x60:
		case 0x80:
		case 0xA0:
		case 0xC0:
		case 0xE0:
			break;
	}

	// Place the bus back into run mode
	CANCON = oldState;

	return (1);
}

/*********************************************************************
 * Function:        NEAR unsigned char CANSetBitRate(NEAR unsigned char bitrate)
 *
 * PreCondition:   	 
 *
 * Input:       	bitrate
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	Communication may be temporarily interrupted
 *
 * Overview:        Changes the bitrate of the node. 
 *
 * Note:           	
 ********************************************************************/
NEAR unsigned char CANSetBitRate(NEAR unsigned char bitrate)
{
	switch (bitrate)
	{
		case 0:
			BRGCON1 = _B125k_BRGCON1;
			BRGCON2 = _B125k_BRGCON2;
			BRGCON3 = _B125k_BRGCON3;
			break; 
		case 1:
			BRGCON1 = _B250k_BRGCON1;
			BRGCON2 = _B250k_BRGCON2;
			BRGCON3 = _B250k_BRGCON3;
			break;
		case 2:
			BRGCON1 = _B500k_BRGCON1;
			BRGCON2 = _B500k_BRGCON2;
			BRGCON3 = _B500k_BRGCON3;
			break;
		default:
			return (0);			// This could be autobaud
	}
	return (1);
}


/*********************************************************************
 * Function:        NEAR unsigned char CANIsBusError(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	
 *
 * Overview:        Returns an error code. 
 *
 * Note:           	
 ********************************************************************/
NEAR unsigned char CANIsBusOffError(void)
{
	// If there is an error then return the Bus-off flag
	if (PIR3bits.ERRIF)
	{
		// Reset the interrupt flag
		PIR3bits.ERRIF = 0;
		
		// Return the bus off bit
		return(COMSTATbits.TXBO);
	}
	// Otherwise return no error
	else 
		return (0);
}



/*********************************************************************
 * Function:        NEAR unsigned char CANIsRxRdy(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	
 *
 * Overview:        Returns TRUE if there is data waiting. 
 *
 * Note:           	
 ********************************************************************/
NEAR unsigned char CANIsRxRdy(void)
{
	// Check hardware buffer 0
	if (RXB0CONbits.RXFUL)
	{
		_msgRxBuf = 0;
		return (RXB0CONbits.RXFUL);
	}
	else 
	
	// Check hardware buffer 1
	if (RXB1CONbits.RXFUL)
	{
		_msgRxBuf = 1;
		return (RXB1CONbits.RXFUL);
	}
	else
		return (0);
}

/*********************************************************************
 * Function:        void CANRead(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	none
 *
 * Side Effects:   	
 *
 * Overview:        Indicates to the module that the data has been read.
 *
 * Note:           	This effectively frees the active hardware buffer 
 *					to continue receiving.
 ********************************************************************/
void CANRead(void)
{
	if (_msgRxBuf == 0)
	{
		RXB0CONbits.RXFUL = 0;
	}
	else 
	if (_msgRxBuf == 1)
	{
		RXB1CONbits.RXFUL = 0;
	}	
}

/*********************************************************************
 * Function:        NEAR unsigned char CANIsTxRdy(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	
 *
 * Overview:        Returns TRUE if the transmit engine is ready to 
 *					accept new data to transmit.
 *
 * Note:           	This function is as simple as it gets, i.e. direct
 *					access to only one TX buffer. 
 ********************************************************************/
NEAR unsigned char CANIsTxRdy(void)
{
	return (!(TXB0CONbits.TXREQ));
}

/*********************************************************************
 * Function:        NEAR unsigned char CANIsMsgSent(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char status of the request
 *
 * Side Effects:   	
 *
 * Overview:        Returns the tag of the data that was last transmitted. 
 *					Any additional calls without additional transmissions
 *					will return NULL.
 *
 * Note:           	This is useful to determine when and who's data was 
 *					placed on the bus. This may seem redundent since
 *					only one hardware buffer is used; however, this
 *					architecture allows for expansion to deep software/
 *					hardware buffer designs without major changes to 
 *					the upper level firmware.
 ********************************************************************/
NEAR unsigned char CANIsMsgSent(void)
{
	NEAR unsigned char temp;
	
	if (!(TXB0CONbits.TXREQ))
	{
		temp = _msgTxTag;
		_msgTxTag = 0;
		return (temp);
	}
	else 
		return (0);
}

/*********************************************************************
 * Function:        void CANSend(NEAR unsigned char txTag)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	none
 *
 * Side Effects:   	
 *
 * Overview:        Tell the transmit engine that the loaded data is
 *					is ready to send.
 *
 * Note:          	Data may not be sent immediately. The tag is used
 *					to trace the transmission since other data may
 *					also be queued to transmit. 	
 ********************************************************************/
void CANSend(NEAR unsigned char txTag)
{
	// Remember the message that was sent
	_msgTxTag = txTag;

	TXB0CONbits.TXREQ = 1;
}

/*********************************************************************
 * Function:        NEAR unsigned int CANGetRxCID(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned int connection ID (11-bit CAN ID)
 *
 * Side Effects:   	
 *
 * Overview:        Get the connection ID from the current receive
 *					message. 
 *
 * Note:          	The CAN ID is assumed to be left justified within
 *					the 16-bit word. 	
 ********************************************************************/
NEAR unsigned int CANGetRxCID(void)
{
	NEAR _INT cid;
	
	// Only return the lower byte
	if (_msgRxBuf == 0)
	{
		cid.bytes.LSB = RXB0SIDL;
		cid.bytes.MSB = RXB0SIDH;
	}
	else //if (_msgRxBuf == 1)
	{
		cid.bytes.LSB = RXB1SIDL;
		cid.bytes.MSB = RXB1SIDH;	
	}
	
	return (cid.word);
}

/*********************************************************************
 * Function:        NEAR unsigned char CANGetRxCnt(void)
 *
 * PreCondition:   	Data should have been received prior to calling 
 *					this function; otherwise, it is likely the returned
 *					count is either not valid or associated to the 
 *					previous message. Use CANIsRxRdy() to determine if 
 *					data is available.
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char count of the data
 *
 * Side Effects:   	
 *
 * Overview:        This function returns the number of bytes waiting
 *					in the receive buffer.
 *
 * Note:          	
 ********************************************************************/
NEAR unsigned char CANGetRxCnt(void)
{
	NEAR unsigned char count;
	
	// Read out of the selected message buffer
	if (_msgRxBuf == 0)
	{
		count = RXB0DLC;
	}
	else //if (_msgRxBuf == 1)
	{
		count = RXB1DLC;	
	}
	
	return (count);
}



/*********************************************************************
 * Function:        unsigned char * NEAR CANGetRxDataPtr(void)
 *
 * PreCondition:   	Data should have been received prior to calling 
 *					this function; otherwise, it is likely the returned
 *					pointer is either not valid or associated to the 
 *					previous message. Use CANIsRxRdy() to determine if 
 *					data is available. 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:   	
 *
 * Overview:        This function returns the pointer to the RX buffer.
 *					
 * Note:          	Use the direct block copy methods for faster data
 *					movement. 	
 ********************************************************************/
unsigned char * NEAR CANGetRxDataPtr(void)
{	
	// Read out of the selected message buffer
	if (_msgRxBuf == 0)
	{
		return(&RXB0D0);
	}
	else //if (_msgRxBuf == 1)
	{
		return(&RXB1D0);	
	}
}



/*********************************************************************
 * Function:        void CANGetRxDataTyp0(unsigned char * NEAR usrBuf)
 *					void CANGetRxDataTyp1(unsigned char * NEAR usrBuf)
 *					void CANGetRxDataTyp2(unsigned char * NEAR usrBuf)
 *
 * PreCondition:   	Data should have been received prior to calling 
 *					this function; otherwise, it is likely the returned
 *					count is either not valid or associated to the 
 *					previous message. Use CANIsRxRdy() to determine if 
 *					data is available. 
 *
 * Input:       	pointer to user supplied buffer
 *                  
 * Output:      	void
 *
 * Side Effects:   	
 *
 * Overview:        Get block data from the buffer. 
 *					
 * Note:          	These functions copy blocks of data rather than 
 *					single bytes. This method is very fast; however,
 *					data limits are not checked. It is up to the 
 *					caller to verify the count to toss data that is  
 *					not valid.
 ********************************************************************/
void CANGetRxDataTyp0(unsigned char * NEAR usrBuf)
{	
	// Read out of the selected message buffer
	if (_msgRxBuf == 0)
	{
		*((_CAN_DATA_TYP0 *)usrBuf) = *((_CAN_DATA_TYP0 *)(&RXB0D0));
	}
	else //if (_msgRxBuf == 1)
	{
		*((_CAN_DATA_TYP0 *)usrBuf) = *((_CAN_DATA_TYP0 *)(&RXB1D0));	
	}
}

void CANGetRxDataTyp1(unsigned char * NEAR usrBuf)
{	
	// Read out of the selected message buffer
	if (_msgRxBuf == 0)
	{
		*((_CAN_DATA_TYP1 *)usrBuf) = *((_CAN_DATA_TYP1 *)((&RXB0D0)+1));
	}
	else //if (_msgRxBuf == 1)
	{
		*((_CAN_DATA_TYP1 *)usrBuf) = *((_CAN_DATA_TYP1 *)((&RXB1D0)+1));	
	}
}

void CANGetRxDataTyp2(unsigned char * NEAR usrBuf)
{	
	// Read out of the selected message buffer
	if (_msgRxBuf == 0)
	{
		*((_CAN_DATA_TYP2 *)usrBuf) = *((_CAN_DATA_TYP2 *)((&RXB0D0)+2));
	}
	else //if (_msgRxBuf == 1)
	{
		*((_CAN_DATA_TYP2 *)usrBuf) = *((_CAN_DATA_TYP2 *)((&RXB1D0)+2));	
	}
}



/*********************************************************************
 * Function:        void CANPutTxCID(NEAR unsigned int txCID)
 *
 * PreCondition:   	The function CANIsTxRdy() must be called prior
 *					to using this function; otherwise, the previous
 *					transmission request may be corrupted. 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:   	
 *
 * Overview:        Write the CID to the TX buffer.
 *					
 * Note:          	 	
 ********************************************************************/
void CANPutTxCID(NEAR unsigned int txCID)
{
	NEAR _INT temp;

	temp.word = txCID;

	// Write the CID to the buffer
	TXB0SIDL = temp.bytes.LSB;
	TXB0SIDH = temp.bytes.MSB;
}



/*********************************************************************
 * Function:        void CANPutTxCnt(NEAR unsigned char txCount)
 *
 * PreCondition:   	The function CANIsTxRdy() must be called prior
 *					to using this function; otherwise, the previous
 *					transmission request may be corrupted.  
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:   	
 *
 * Overview:        Write the count to the TX buffer.
 *					
 *
 * Note:          	 	
 ********************************************************************/
void CANPutTxCnt(NEAR unsigned char txCount)
{
	TXB0DLC = txCount & 0x0F;
}


/*********************************************************************
 * Function:        unsigned char * NEAR CANGetTxDataPtr(void)
 *
 * PreCondition:   	The function CANIsTxRdy() must be called prior
 *					to using this function; otherwise, the pointer
 *					may be invalid.
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:   	
 *
 * Overview:        Return a pointer to the buffer.
 *					
 *
 * Note:          	 	
 ********************************************************************/
unsigned char * NEAR CANGetTxDataPtr(void)
{
	return (&TXB0D0);
}


/*********************************************************************
 * Function:        void CANPutTxDataTyp0(unsigned char * NEAR usrBuf)
 *					void CANPutTxDataTyp1(unsigned char * NEAR usrBuf)
 *					void CANPutTxDataTyp2(unsigned char * NEAR usrBuf)
 *
 * PreCondition:   	The function CANIsTxRdy() must be called prior
 *					to using this function; otherwise, the previous
 *					transmission request may be corrupted.  
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:   	
 *
 * Overview:        Put a block of data in the TX buffer.
 *					
 * Note:          	Copying a block into the buffer does not imply
 *					that the entire block must be transmitted. This
 *					function mearly loads the data into the buffer
 *					following the specified format TYP0, TYP1, or TYP2. 	
 ********************************************************************/
void CANPutTxDataTyp0(unsigned char * NEAR usrBuf)
{
	*((_CAN_DATA_TYP0 *)((&TXB0D0))) = *((_CAN_DATA_TYP0 *)usrBuf);
}

void CANPutTxDataTyp1(unsigned char * NEAR usrBuf)
{
	*((_CAN_DATA_TYP1 *)((&TXB0D0)+1)) = *((_CAN_DATA_TYP1 *)usrBuf);
}

void CANPutTxDataTyp2(unsigned char * NEAR usrBuf)
{
	*((_CAN_DATA_TYP2 *)((&TXB0D0)+2)) = *((_CAN_DATA_TYP2 *)usrBuf);
}



/*********************************************************************
 * Function:        void CANInit(void)
 *
 * PreCondition:   	 
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char * 
 *
 * Side Effects:  	Filters are removed. 	
 *
 * Overview:        Initialize the module.
 *					
 * Note:          	
 ********************************************************************/
void CANInit(void)
{
	CANClose();
	
	// Init some internal control registers
	_msgTxTag = 0;
	_msgRxBuf = 0;

	// Init device specific hardware for CAN
	CIOCON = 0x20;		
	LATBbits.LATB2 = 1;
	TRISBbits.TRISB2 = 0;
	TRISBbits.TRISB3 = 1;
	
	RXM0SIDH = 0xFF;
	RXM0SIDL = 0xFF;
	RXM1SIDH = 0xFF;
	RXM1SIDL = 0xFF;
	RXF0SIDL = 0xFF;
	RXF1SIDL = 0xFF;
	RXF2SIDL = 0xFF;
	RXF3SIDL = 0xFF;
	RXF4SIDL = 0xFF;
	RXF5SIDL = 0xFF;
	
	RXB0CON = 0x20;
	RXB1CON = 0x20;
}




 
		 
