/*****************************************************************************
 *
 * Microchip DeviceNet Stack (Explicit Messaging Manager Source)
 *
 *****************************************************************************
 * FileName:        emm.c
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.20.00 or higher
 * Linker:          MPLINK 03.40.00 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *
 * This file is simply a managing routine that parses data from the 
 * Explicit Messaging Connection for the router.
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			05/03/03	...	
 * 
 *****************************************************************************/




#include	"dnet.def"				// Global definitions file
#include 	"typedefs.h"

#include	"route.h"				// Router prototypes and macros

#include	"services.h"			// Service codes
#include	"errors.h"				// Error codes
#include	"class.h"				// Class codes

#include 	"conn.h"				// Connection object
#include	"dnet.h"				// DeviceNet object


#define	_HDR_SIZE 		sizeof(route.header)  
#define _CID_SIZE		sizeof(route.classID)
#define	_IID_SIZE		sizeof(route.instanceID)
#define	_SVC_SIZE		sizeof(route.service)
#define	_HSCI_SIZE		_HDR_SIZE + _SVC_SIZE +	_CID_SIZE +	_IID_SIZE




/*********************************************************************
 * Function:        unsigned char ExplicitMsgManager(void)
 *
 * PreCondition:    
 *
 * Input:       	none		
 *                  
 * Output:      	status of the process	
 *
 * Side Effects:    none
 *
 * Overview:        This function manages any Explicit Message data
 *					and provides parsing for the Router.
 *
 * Note:           
 ********************************************************************/
unsigned char ExplicitMsgManager(void)
{
	USINT retStatus;

	retStatus = 0;

	// Do only if the connection object is prepared to receive and transmit
	if (_ConnReadRdy(1) && _ConnWriteRdy(1))
	{
		// Insure that sufficient data has been received to process
		if (uConn1.rx.len >= (_HSCI_SIZE))
		{	
			// Set the buffer pointers and other info based on the connection object settings
			route.pInBuf = uConn1RxBuffer;  
			route.pOutBuf = uConn1TxBuffer + (_HDR_SIZE + _SVC_SIZE);
			route.inBufLen = uConn1.rx.lenMax - (_HSCI_SIZE);
			route.inBufDataLen = uConn1.rx.len - (_HSCI_SIZE);
			route.outBufLen = uConn1.tx.lenMax - (_HDR_SIZE + _SVC_SIZE);
			route.outBufDataLen = 0;


			// Parse the buffer using pointers provided from the connection object
			route.header = *route.pInBuf; route.pInBuf++; 
			route.service = *route.pInBuf; route.pInBuf++;
			#if (CLASS_WIDTH_16BIT) 
			route.classID.bytes.LSB = *route.pInBuf; route.pInBuf++;
			route.classID.bytes.MSB = *route.pInBuf; route.pInBuf++; 
			#else
			route.classID = *route.pInBuf; route.pInBuf++;
			#endif
			#if (INSTANCE_WIDTH_16BIT)
			route.instanceID.bytes.LSB = *route.pInBuf; route.pInBuf++;
			route.instanceID.bytes.MSB = *route.pInBuf; route.pInBuf++; 
			#else
			route.instanceID = *route.pInBuf; route.pInBuf++;
			#endif
			route.attributeID = *route.pInBuf;
		   
			// Route the message to the object
			retStatus = RouteMessage();		
			
			// Write the transmit buffer header and service info
			// all other data should be provided by the appropriate class
			uConn1.tx.len = route.outBufDataLen + (_HDR_SIZE + _SVC_SIZE);
			if(uConn1.tx.len > 8) route.header |= 0x80;
			else route.header &= 0x7F;
			route.pOutBuf = uConn1TxBuffer;
			*route.pOutBuf = route.header; route.pOutBuf++;
			*route.pOutBuf = route.service | 0x80;		// Set the response bit
								
			_ConnRead(1);					// Unlock the read for explicit messaging
			
			if (retStatus) _ConnWrite(1);	// Enable write for explicit messaging
		}
		else
			return (retStatus);
	}
	else 
		return (0);
}
