/*****************************************************************************
 *
 * Microchip DeviceNet Stack (Unconnected Explicit Messaging Connection Object Source)
 *
 *****************************************************************************
 * FileName:        conn6.c
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.20.00 or higher
 * Linker:          MPLINK 03.40.00 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *
 * This file contains firmware to handle Unconnected Explicit Messaging within
 * the Connection Object. The code in this object is much simpler than I/O
 * and Explicit Objects because the requirements defined in the specification
 * are significantly less.
 * 
 *
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			04/28/03	...	
 * 
 *****************************************************************************/

#include	"dnet.def"			// Global definitions file
#include 	"typedefs.h"

#include	"conn.h"			// Connection prototypes and symbols

#include	"services.h"		// Service codes
#include	"errors.h"			// Error codes
#include	"class.h"			// Class codes

#include	"route.h"			// Global symbols defined by the router
#include	"dnet.h"			// DeviceNet prototypes and symbols
#include	"CAN.h"


#pragma	udata
/*********************************************************************
 * Connection related variables
 ********************************************************************/
CONN_PRIV	 	uConn6;

unsigned char	uConn6RxBuffer[8];
unsigned char	uConn6TxBuffer[8];



/*********************************************************************
 * Function:        unsigned char _Conn6Create(void)
 *
 * PreCondition:    The CAN (or other) driver must be prepared to 
 *					accept some filter settings.
 *
 * Input:       	none	
 *                  
 * Output:      	unsigned char handle to the connection	
 *
 * Side Effects:    none
 *
 * Overview:        This function creates a connection 
 *					in the predefined set and returns a 
 *					handle to the connection. 
 *
 * Note:            This function is not called directly by 
 *					application code.
 ********************************************************************/
unsigned char _Conn6Create(void)
{
	unsigned int test;
	
	// Setup the pointers to the buffers
	uConn6.rx.pMsg = uConn6RxBuffer;
	uConn6.tx.pMsg = uConn6TxBuffer;

	// Setup the maximum length of the buffers
   	uConn6.rx.lenMax = uConn6.tx.lenMax = 8;
	
	// Setup the produced and consummed CID	
	uConn6.attrib.consumed_cid.bytes.LSB = 0xC0;
	uConn6.attrib.produced_cid.bytes.LSB = 0x60;
	uConn6.attrib.consumed_cid.bytes.MSB = uConn6.attrib.produced_cid.bytes.MSB = uDNet.MACID | 0x80;
//	uConn6.attrib.consumed_cid.word = uConn6.attrib.produced_cid.word = 1024 | (uDNet.MACID << 3) | 6;
			
	//Issue a request to the driver to start receiving the CID	
	CANSetFilter(uConn6.attrib.consumed_cid.word);
	return (6);
}


/*********************************************************************
 * Function:        unsigned char _Conn6Close()
 *
 * PreCondition:    The connection should have already been open.
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char instance that closed
 *
 * Side Effects:    none
 *
 * Overview:        Closes the specified connection. 
 *
 * Note:            none
 ********************************************************************/
unsigned char _Conn6Close()
{
	CANClrFilter(uConn6.attrib.consumed_cid.word);

	return(1);
}



/*********************************************************************
 * Function:        void _Conn6RxEvent(void)
 *
 * PreCondition:    none
 *
 * Input:       	none
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process received data for this connection.
 *
 * Note:            This event occures when data has been received
 *					for this connection instance.
 ********************************************************************/
void _Conn6RxEvent(void)
{
	BYTE header, classId, service;
	unsigned char *pRxData;
	
	// Get the pointer to the received data
	pRxData = CANGetRxDataPtr();

	// Extract the header, service, and class information
	header.byte = *pRxData;
	service.byte = *(pRxData + 1);
	classId.byte = *(pRxData + 2);


	if (((header.byte ^ uDNet.AllocInfo.MasterMACID & 0x3F) == 0) ||
		(uDNet.AllocInfo.MasterMACID == 0xFF) ||
		((classId.byte == CLASS_DEVICENET) && (service.byte == SRVS_RELEASE_CONN)))
	{

		// Get the length of the message
		uConn6.rx.len = CANGetRxCnt();

		// Copy the message to the connection buffer
		CANGetRxDataTyp0(uConn6RxBuffer);
				
		// Indicate a complete section of data has been received
		_rxFlag.bits.uexpl = 1;
	}

	// Release the hardware
	CANRead();
}




/*********************************************************************
 * Function:        void _Conn6TxOpenEvent(void)
 *
 * PreCondition:    A transmit request must have been made.
 *
 * Input:       	none	
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process open transmit que event
 *
 * Note:            This event occurs when the buffer is available 
 *					for this connection instance to transmit. A 
 *					transmit request must have been made to enter 
 *					this function.
 ********************************************************************/
void _Conn6TxOpenEvent(void)
{
	// Copy the message to the hardware buffer
	CANPutTxDataTyp0(uConn6TxBuffer);
   	
   	// Set the produced CID
   	CANPutTxCID(uConn6.attrib.produced_cid.word);

  	// Set the length of the message
   	CANPutTxCnt(uConn6.tx.len);

   	// Request the hardware to queue the message to send
   	CANSend(6);

	// Clear the transmit flag to open access to the write buffer
	_txFlag.bits.uexpl = 0;
}


/*********************************************************************
 * Function:        void _Conn6TxEvent(void)
 *
 * PreCondition:    Data must have been queued to transmit.
 *
 * Input:       	none	
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process data for this connection.
 *
 * Note:            This event occurs when the buffer has successfully
 *					placed the requested data on the bus.
 ********************************************************************/
void _Conn6TxEvent(void)
{
	// Set flag indicating data has been placed on the bus
	_txFinFlags.bits.uexpl = 1;
	
	// This particular connection does nothing with this event
}









