/*****************************************************************************
 *
 * Microchip DeviceNet Stack (Duplicate ID Messaging Connection Object Source)
 *
 *****************************************************************************
 * FileName:        conn7.c
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.20.00 or higher
 * Linker:          MPLINK 03.40.00 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *
 * This file contains firmware to handle Duplicate ID Messaging within
 * the Connection Object. The code in this object is much simpler than I/O
 * and Explicit Objects because the requirements defined in the specification
 * are significantly less.
 * 
 *
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			04/28/03	...	
 * 
 *****************************************************************************/

#include	"dnet.def"			// Global definitions file
#include 	"typedefs.h"

#include	"conn.h"			// Connection prototypes and symbols

#include	"services.h"		// Service codes
#include	"errors.h"			// Error codes
#include	"class.h"			// Class codes

#include	"route.h"			// Global symbols defined by the router
#include	"dnet.h"			// DeviceNet prototypes and symbols
#include	"CAN.h"

#pragma	udata
/*********************************************************************
 * Connection related variables
 ********************************************************************/
CONN_PRIV	 	uConn7;

unsigned char	uConn7RxBuffer[8];
unsigned char	uConn7TxBuffer[8];

 
/*********************************************************************
 * Function:        unsigned char _Conn7Create(void)
 *
 * PreCondition:    The CAN (or other) driver must be prepared to 
 *					accept some filter settings.
 *
 * Input:       	none	
 *                  
 * Output:      	unsigned char handle to the connection	
 *
 * Side Effects:    none
 *
 * Overview:        This function creates a connection 
 *					in the predefined set and returns a 
 *					handle to the connection. 
 *
 * Note:            This function is not called directly by 
 *					application code.
 ********************************************************************/
unsigned char _Conn7Create()
{
	// Setup the pointers to the buffers
	uConn7.rx.pMsg = uConn7RxBuffer;
	uConn7.tx.pMsg = uConn7TxBuffer;

	// Setup the maximum length of the buffers
   	uConn7.rx.lenMax = uConn7.tx.lenMax = 8;
   	
	uConn7.tx.flags.byte = 0;
			
	// Setup the produced and consummed CID	
	uConn7.attrib.consumed_cid.bytes.LSB = uConn7.attrib.produced_cid.bytes.LSB = 0xE0;
	uConn7.attrib.consumed_cid.bytes.MSB = uConn7.attrib.produced_cid.bytes.MSB = uDNet.MACID | 0x80;
//	uConn7.attrib.consumed_cid.word = uConn7.attrib.produced_cid.word = 1024 | (uDNet.MACID << 3) | 7;
	
	//Issue a request to the driver to start receiving the CID
	CANSetFilter(uConn7.attrib.consumed_cid.word);
	return (7);
}


/*********************************************************************
 * Function:        unsigned char _Conn7Close()
 *
 * PreCondition:    The connection should have already been open.
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char instance that closed
 *
 * Side Effects:    none
 *
 * Overview:        Closes the specified connection. 
 *
 * Note:            none
 ********************************************************************/
unsigned char _Conn7Close()
{
	CANClrFilter(uConn7.attrib.consumed_cid.word);

	return(7);
}






/*********************************************************************
 * Function:        void _Conn7UpdateTimer(void)
 *
 * PreCondition:    none
 *
 * Input:       	none	
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Update timer and process any timer events.
 *
 * Note:            None
 ********************************************************************/
void _Conn7TimerEvent(void)
{
	// Not used at this time	   
		   
}




/*********************************************************************
 * Function:        void _Conn7RxEvent(void)
 *
 * PreCondition:    none
 *
 * Input:       	none
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process received data for this connection.
 *
 * Note:            This event occures when data has been received
 *					for this connection instance.
 ********************************************************************/
void _Conn7RxEvent(void)
{
	// The receive cannot continue if a message from this instance 
	// has just been written to the bus; the _Conn7TxEvent must be
	// handled first.
	if (!uConn7.tx.flags.bits.b0) 
	{
		// Get the length of the message
		uConn7.rx.len = CANGetRxCnt();

		// Copy the message to the connection buffer
		CANGetRxDataTyp0(uConn7RxBuffer);
				
		// Indicate a complete section of data has been received
		_rxFlag.bits.dupid = 1;

		// Release the hardware
		CANRead();
	}
}




/*********************************************************************
 * Function:        void _Conn7TxOpenEvent(void)
 *
 * PreCondition:    A transmit request must have been made.
 *
 * Input:       	none	
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process open transmit que event
 *
 * Note:            This event occurs when the buffer is available 
 *					for this connection instance to transmit. A 
 *					transmit request must have been made to enter 
 *					this function.
 ********************************************************************/
void _Conn7TxOpenEvent(void)
{
	// Copy the message to the hardware buffer
	CANPutTxDataTyp0(uConn7TxBuffer);
   	
   	// Set the produced CID
   	CANPutTxCID(uConn7.attrib.produced_cid.word);

  	// Set the length of the message
   	CANPutTxCnt(uConn7.tx.len);

   	// Request the hardware to queue the message to send
   	CANSend(7);

	// Clear the transmit flag to open access to the write buffer
	_txFlag.bits.dupid = 0;

	//Set a flag indicating message has been sent. This places a lock
	//on receive events until transmit is complete.
	uConn7.tx.flags.bits.b0 = 1;
}


/*********************************************************************
 * Function:        void _Conn2TxEvent(void)
 *
 * PreCondition:    Data must have been queued to transmit.
 *
 * Input:       	none	
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process data for this connection.
 *
 * Note:            This event occurs when the buffer has successfully
 *					placed the requested data on the bus.
 ********************************************************************/
void _Conn7TxEvent(void)
{
	// Set flag indicating data has been placed on the bus
	_txFinFlags.bits.dupid = 1;

	// Clear a flag indicating the message has been sent. This releases 
	// the lock on receive events.
	uConn7.tx.flags.bits.b0 = 0;
	
	// Ignore any data that was the result of an echo from transmit.
	CANRead();
}




