/*
 * Experimental driver for the CPLD in the HTC Magician
 *
 * Copyright 2006 Philipp Zabel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/kernel.h>	/* KERN_ERR */
#include <linux/module.h>
#include <linux/version.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>

#include <asm/io.h>		/* ioremap */
#include <asm/mach/irq.h>	/* IRQF.., IRQT.. */
#include <asm/arch/pxa-regs.h>	/* GPSR, GPCR macros */
#include <asm/arch/magician.h>	/* magician specific (E)GPIOs */

/* debug */
#define CPLD_DEBUG 1
#if CPLD_DEBUG
#define dbg(format, arg...) printk(format, ## arg)
#else
#define dbg(format, arg...)
#endif

struct magician_cpld_registers
{
	u32 egpio[3];
	u32 irq_state;
	u32 cable_state;
	u32 cpld14;
	u32 cpld18;
};

struct magician_cpld_data
{
	struct magician_cpld_registers *mapping;
	int irq_nr;
	u32 cached_egpio[3];	// cached egpio registers
};

static inline int u8pos(int bit)
{
	return bit/8;
}

static inline int u8bit(int bit)
{
	return 1<<(bit & (8-1));
}

void magician_egpio_enable (struct platform_device *dev, int bit)
{
	unsigned long flags;
	struct magician_cpld_data *cpld = platform_get_drvdata(dev);
	int pos = u8pos(bit);
	int mask = u8bit(bit);

	local_irq_save(flags);

	if (!(cpld->cached_egpio[pos] & mask)) {
		cpld->cached_egpio[pos] |= mask;
		dbg("egpio set: reg %d = 0x%02x\n", pos, cpld->cached_egpio[pos]);
		cpld->mapping->egpio[pos] = cpld->cached_egpio[pos];
	}

	local_irq_restore(flags);
}

void magician_egpio_disable (struct platform_device *dev, int bit)
{
	unsigned long flags;
	struct magician_cpld_data *cpld = platform_get_drvdata(dev);
	int pos = u8pos(bit);
	int mask = u8bit(bit);

	local_irq_save(flags);

	if (cpld->cached_egpio[pos] & mask) {
		cpld->cached_egpio[pos] &= ~mask;
		dbg("egpio set: reg %d = 0x%02x\n", pos, cpld->cached_egpio[pos]);
		cpld->mapping->egpio[pos] = cpld->cached_egpio[pos];
	}

	local_irq_restore(flags);
}

int magician_cable_state (struct platform_device *dev)
{
	struct magician_cpld_data *cpld = platform_get_drvdata(dev);

	return cpld->mapping->cable_state;
}

int magician_get_sd_ro(struct platform_device *dev)
{
	struct magician_cpld_data *cpld = platform_get_drvdata(dev);
	return (!(cpld->mapping->cpld14 & 0x10));
}

int magician_ep_state (struct platform_device *dev)
{
	struct magician_cpld_data *cpld = platform_get_drvdata(dev);

	return cpld->mapping->cpld18 & 0x2;
}

static void magician_cpld_ack_gpio(unsigned int irq)
{
	dbg ("magician_cpld_ack_gpio(%d)\n", irq);
}

static void magician_cpld_mask_irq(unsigned int irq)
{
}

static void magician_cpld_unmask_irq(unsigned int irq)
{
}

static struct irq_chip magician_cpld_chip = {
	.ack		= magician_cpld_ack_gpio,
	.mask		= magician_cpld_mask_irq,
	.unmask		= magician_cpld_unmask_irq,
};

/*
 * Demux handler for CPLD edge detect interrupts
 */
static void magician_cpld_irq_handler(unsigned int irq, struct irq_desc *desc)
{
	struct magician_cpld_data *cpld = desc->chip_data;
	int result, cable_state;

	/* Acknowledge the parent IRQ */
	desc->chip->ack(irq);

	result = cpld->mapping->irq_state;
	dbg ("cpld irq 0x%02x ", result);
	cpld->mapping->irq_state = result;			// ACK (clear)
	dbg ("(0x%02x)\n", cpld->mapping->irq_state);

	cable_state = cpld->mapping->cable_state;

	switch (result) {
		case 0x1:
			irq = IRQ_MAGICIAN_SD;
			desc = &irq_desc[irq];
			desc_handle_irq(irq, desc);
			break;
		case 0x2:
			irq = IRQ_MAGICIAN_EP;
			dbg ("ep_state = 0x%02x\n", cpld->mapping->cpld18);
			desc = &irq_desc[irq];
			desc_handle_irq(irq, desc);
			break;
		case 0x4:
			irq = IRQ_MAGICIAN_BT;
			desc = &irq_desc[irq];
			desc_handle_irq(irq, desc);
			break;
		case 0x8:
			irq = IRQ_MAGICIAN_AC;
			dbg ("cable_state = 0x%02x\n", cable_state);
			desc = &irq_desc[irq];
			desc_handle_irq(irq, desc);
			break;
		default:
			printk (KERN_ERR "magician_cpld: unknown irq 0x%x\n", result);
	}
}

static int magician_cpld_probe (struct platform_device *dev)
{
	struct magician_cpld_data *cpld;
	struct resource *res;

	cpld = kmalloc (sizeof (struct magician_cpld_data), GFP_KERNEL);
	if (!cpld)
		return -ENOMEM;
	memset (cpld, 0, sizeof (*cpld));

	platform_set_drvdata(dev, cpld);

	res = platform_get_resource(dev, IORESOURCE_MEM, 0);
	cpld->mapping = (struct magician_cpld_registers *)ioremap (res->start, res->end - res->start);
	if (!cpld->mapping) {
		printk (KERN_ERR "magician_cpld: couldn't ioremap\n");
		kfree (cpld);
		return -ENOMEM;
	}

	printk(KERN_INFO "MainBoardID = %x\n", cpld->mapping->cpld14 & 0x7);
	printk(KERN_INFO "magician_cpld: lcd_select = %d (%s)\n", cpld->mapping->cpld14 & 0x8,
			(cpld->mapping->cpld14 & 0x8) ? "samsung" : "toppoly");

	/*
	 * Initial values. FIXME: if we're not booting from wince, we should set them ourselves.
	 */
	cpld->cached_egpio[0] = cpld->mapping->egpio[0];
	cpld->cached_egpio[1] = 0x88; /* BL_POWER and IN_SEL = internal mic */
	cpld->cached_egpio[2] = cpld->mapping->egpio[2];
	dbg("initial egpios: 0x%02x 0x%02x 0x%02x\n", cpld->cached_egpio[0], cpld->cached_egpio[1], cpld->cached_egpio[2]);

	cpld->irq_nr = platform_get_irq(dev, 0);
	if (cpld->irq_nr >= 0) {
		unsigned int irq;

		/*
		 * Setup handlers for CPLD irqs, muxed into IRQ_GPIO(13)
		 */
		for (irq = IRQ_MAGICIAN_SD; irq <= IRQ_MAGICIAN_AC; irq++) {
			set_irq_chip(irq, &magician_cpld_chip);
			set_irq_handler(irq, handle_simple_irq);
			set_irq_flags(irq, IRQF_VALID | IRQF_PROBE); // | ACK
		}

		set_irq_chained_handler(cpld->irq_nr, magician_cpld_irq_handler);
		set_irq_type (cpld->irq_nr, IRQ_TYPE_EDGE_RISING);
		set_irq_data (cpld->irq_nr, cpld);
		set_irq_chip_data(cpld->irq_nr, cpld);
	}

	PWER |= (1<<13); /* wake up on CPLR interrupt */
	PRER |= (1<<13);
	PFER |= (1<<13);

	return 0;
}

static int magician_cpld_remove (struct platform_device *dev)
{
	struct magician_cpld_data *cpld = platform_get_drvdata(dev);

	if (cpld->irq_nr >= 0) {
		unsigned int irq;

		for (irq = IRQ_MAGICIAN_SD; irq <= IRQ_MAGICIAN_AC; irq++) {
			set_irq_chip(irq, NULL);
			set_irq_handler(irq, NULL);
			set_irq_flags(irq, 0);
		}

		set_irq_chained_handler(cpld->irq_nr, NULL);
	}

	iounmap((void *)cpld->mapping);
	kfree(cpld);
	platform_set_drvdata(dev, NULL);

	PWER &= ~(1<<13);
	PRER &= ~(1<<13);
	PFER &= ~(1<<13);

	return 0;
}

#ifdef CONFIG_PM
static int magician_cpld_resume(struct platform_device *dev)
{
	struct magician_cpld_data *cpld = platform_get_drvdata(dev);
	int result, cable_state;

	result = cpld->mapping->irq_state;
	dbg ("resume cpld irq 0x%02x ", result);
	cpld->mapping->irq_state = result;		// ACK (clear)
	dbg ("(0x%02x)\n", cpld->mapping->irq_state);

	cable_state = cpld->mapping->cable_state;

	switch (result) {
		case 0x1:
			dbg ("SD");
			break;
		case 0x2:
			dbg ("EP\nep_state = 0x%02x\n", cpld->mapping->cpld18);
			break;
		case 0x4:
			dbg ("BT\n");
			break;
		case 0x8:
			dbg ("AC\ncable_state = 0x%02x\n", cable_state);
			break;
		default:
			printk (KERN_ERR "magician_cpld: unknown irq 0x%x\n", result);
	}

	return 0;
}
#else
#define magician_cpld_resume NULL
#endif

int magician_cpld_get_wakeup_irq(struct platform_device *dev)
{
	struct magician_cpld_data *cpld = platform_get_drvdata(dev);
	int result;

	result = cpld->mapping->irq_state;
	dbg ("wakeup cpld irq 0x%02x ", result);
	cpld->mapping->irq_state = result;		// ACK (clear)
	dbg ("(0x%02x)\n", cpld->mapping->irq_state);

	switch (result) {
	case 0x1:
		return IRQ_MAGICIAN_SD;
	case 0x2:
		return IRQ_MAGICIAN_EP;
	case 0x4:
		return IRQ_MAGICIAN_BT;
	case 0x8:
		return IRQ_MAGICIAN_AC;
	default:
		return -1;
	}

	return 0;
}

struct platform_driver magician_cpld_driver = {
	.probe    = magician_cpld_probe,
	.remove   = magician_cpld_remove,
	.resume   = magician_cpld_resume,
	.driver   = {
		.name     = "magician-cpld",
	},
};

static int magician_cpld_init (void)
{
	return platform_driver_register (&magician_cpld_driver);
}

static void magician_cpld_exit (void)
{
	platform_driver_unregister (&magician_cpld_driver);
}

module_init (magician_cpld_init);
module_exit (magician_cpld_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Philipp Zabel <philipp.zabel@gmail.com>");
MODULE_DESCRIPTION("Experimental driver for the CPLD in the HTC Magician");
MODULE_SUPPORTED_DEVICE("magician-cpld");

EXPORT_SYMBOL(magician_egpio_enable);
EXPORT_SYMBOL(magician_egpio_disable);
EXPORT_SYMBOL(magician_cpld_get_wakeup_irq);
EXPORT_SYMBOL(magician_cable_state);
EXPORT_SYMBOL(magician_ep_state);
EXPORT_SYMBOL(magician_get_sd_ro);
